#pragma once

#include "common.h"

#include <maps/wikimap/mapspro/services/mrc/libs/traffic_signs/include/yandex/maps/mrc/traffic_signs/signs.h>

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/units.h>
#include <maps/libs/sql_chemistry/include/gateway_access.h>
#include <maps/libs/introspection/include/comparison.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/image_box.h>

#include <util/generic/array_ref.h>
#include <unordered_map>
#include <vector>

namespace maps::mrc::db {

class Sign {
public:
    using Type = traffic_signs::TrafficSign;

    TId id() const { return id_; }

    Type type() const { return type_; }

    geolib3::Point2 mercatorPos() const { return mercatorPosition_; }

    geolib3::Point2 geodeticPos() const {
        return geolib3::convertMercatorToGeodetic(mercatorPos());
    }

    double positionVarianceMeters() const { return positionVariance_; }

    /// Represents the normal to sign surface.
    geolib3::Heading heading() const { return geolib3::Heading{heading_}; }

    geolib3::Degrees headingVariance() const { return geolib3::Degrees{headingVariance_}; }

    Sign& setType(Type type) {
        type_ = type;
        return *this;
    }

    Sign& setGeodeticPos(geolib3::Point2 position, double variance) {
        return setMercatorPos(geolib3::convertGeodeticToMercator(position), variance);
    }

    Sign& setMercatorPos(geolib3::Point2 position, double variance) {
        mercatorPosition_ = position;
        positionVariance_ = variance;
        return *this;
    }

    Sign& setHeading(geolib3::Heading heading, geolib3::Degrees variance) {
        heading_ = heading.value();
        headingVariance_ = variance.value();
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<Sign>;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.type_, t.mercatorPosition_, t.positionVariance_, t.heading_,
            t.headingVariance_);
    }

    Sign() = default;

    TId id_{0};
    Type type_{Type::Unknown};
    geolib3::Point2 mercatorPosition_{};
    double positionVariance_{};
    //FIXME: use tagged type after https://st.yandex-team.ru/MAPSCORE-4496
    double heading_{};
    double headingVariance_{};

public:
    auto introspect() const { return introspect(*this); }
};

using Signs = std::vector<Sign>;

class SignFeature {
public:
    //WARN: constructing SignFeature which is not bound to a sign yet
    SignFeature(TId featureId, size_t minX, size_t minY, size_t maxX, size_t maxY)
        : signId_(0)
        , featureId_(featureId)
        , minX_(minX)
        , minY_(minY)
        , maxX_(maxX)
        , maxY_(maxY) {
        REQUIRE(minX_ <= maxX_ && minY_ <= maxY_,
            "SignFeature bounding box [" << minX_ << ", " << minY_ << ", " <<
                maxX_ << ", " << maxY << "] is not correct"
        );
    }

    TId signId() const { return signId_; }

    TId featureId() const { return featureId_; }

    /// Coordinates of top corner of sign image box.
    int32_t minX() const { return minX_; }

    int32_t minY() const { return minY_; }

    /// Coordinates of bottom corner of sign image box.
    int32_t maxX() const { return maxX_; }

    int32_t maxY() const { return maxY_; }

    SignFeature& setSignId(TId signId) {
        signId_ = signId;
        return *this;
    }

    common::ImageBox imageBox() const { return common::ImageBox(minX_, minY_, maxX_, maxY_); }

private:
    friend class sql_chemistry::GatewayAccess<SignFeature>;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.signId_, t.featureId_, t.minX_, t.minY_, t.maxX_, t.maxY_);
    }

    SignFeature() = default;

    TId signId_{};
    TId featureId_{};
    int32_t minX_{}, minY_{}, maxX_{}, maxY_{};

public:
    auto introspect() const { return introspect(*this); }
};

using SignFeatures = std::vector<SignFeature>;

class SignPanorama {
public:
    //WARN: constructing SignPanorama which is not bound to a sign yet
    SignPanorama(TId panoramaId, size_t minX, size_t minY, size_t maxX, size_t maxY)
        : signId_(0)
        , panoramaId_(panoramaId)
        , minX_(minX)
        , minY_(minY)
        , maxX_(maxX)
        , maxY_(maxY) {
        REQUIRE(minX_ <= maxX_ && minY_ <= maxY_,
            "SignPanorama bounding box [" << minX_ << ", " << minY_ << ", " <<
                maxX_ << ", " << maxY << "] is not correct"
        );
    }

    TId signId() const { return signId_; }

    TId panoramaId() const { return panoramaId_; }

    /// Coordinates of top corner of sign image box.
    int32_t minX() const { return minX_; }

    int32_t minY() const { return minY_; }

    /// Coordinates of bottom corner of sign image box.
    int32_t maxX() const { return maxX_; }

    int32_t maxY() const { return maxY_; }

    SignPanorama& setSignId(TId signId) {
        signId_ = signId;
        return *this;
    }

    common::ImageBox imageBox() const { return common::ImageBox(minX_, minY_, maxX_, maxY_); }

private:
    friend class sql_chemistry::GatewayAccess<SignPanorama>;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.signId_, t.panoramaId_, t.minX_, t.minY_, t.maxX_, t.maxY_);
    }

    SignPanorama() = default;

    TId signId_{};
    TId panoramaId_{};
    int32_t minX_{}, minY_{}, maxX_{}, maxY_{};

public:
    auto introspect() const { return introspect(*this); }
};

using SignPanoramas = std::vector<SignPanorama>;

// TODO: remove it from header
using introspection::operator==;

} // namespace maps::mrc::db
