#pragma once

#include "common.h"

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/heading.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/units.h>
#include <maps/libs/sql_chemistry/include/gateway_access.h>
#include <maps/libs/introspection/include/comparison.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/image_box.h>

#include <maps/wikimap/mapspro/services/mrc/libs/nirvana/include/nirvana_workflow.h>

#include <util/generic/array_ref.h>
#include <unordered_map>
#include <vector>

namespace maps::mrc::db {

class TrafficLight {
public:
    TId id() const { return id_; }

    geolib3::Point2 mercatorPos() const { return mercatorPosition_; }

    geolib3::Point2 geodeticPos() const {
        return geolib3::convertMercatorToGeodetic(mercatorPos());
    }

    double positionVarianceMeters() const { return positionVariance_; }

    /// Represents the normal to traffic light surface.
    geolib3::Heading heading() const { return geolib3::Heading{heading_}; }

    geolib3::Degrees headingVariance() const { return geolib3::Degrees{headingVariance_}; }

    TrafficLight& setGeodeticPos(geolib3::Point2 position, double variance) {
        return setMercatorPos(geolib3::convertGeodeticToMercator(position), variance);
    }

    TrafficLight& setMercatorPos(geolib3::Point2 position, double variance) {
        mercatorPosition_ = position;
        positionVariance_ = variance;
        return *this;
    }

    TrafficLight& setHeading(geolib3::Heading heading, geolib3::Degrees variance) {
        heading_ = heading.value();
        headingVariance_ = variance.value();
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<TrafficLight>;

    TrafficLight() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.mercatorPosition_, t.positionVariance_, t.heading_,
            t.headingVariance_);
    }

    TId id_{0};
    geolib3::Point2 mercatorPosition_{};
    double positionVariance_{};
    //FIXME: use tagged type after https://st.yandex-team.ru/MAPSCORE-4496
    double heading_{};
    double headingVariance_{};

public:
    auto introspect() const { return introspect(*this); }
};

using TrafficLights = std::vector<TrafficLight>;

class TrafficLightFeature {
public:
    //WARN: constructing TrafficLightFeature which is not bound to a sign yet
    TrafficLightFeature(TId featureId, size_t minX, size_t minY, size_t maxX, size_t maxY)
        : trafficLightId_(0)
        , featureId_(featureId)
        , minX_(minX)
        , minY_(minY)
        , maxX_(maxX)
        , maxY_(maxY) {
        REQUIRE(minX_ <= maxX_ && minY_ <= maxY_,
            "TrafficLightFeature bounding box [" << minX_ << ", " << minY_ << ", " <<
                maxX_ << ", " << maxY << "] is not correct"
        );
    }

    TId trafficLightId() const { return trafficLightId_; }

    TId featureId() const { return featureId_; }

    /// Coordinates of top corner of sign image box.
    int32_t minX() const { return minX_; }

    int32_t minY() const { return minY_; }

    /// Coordinates of bottom corner of sign image box.
    int32_t maxX() const { return maxX_; }

    int32_t maxY() const { return maxY_; }

    TrafficLightFeature& setTrafficLightId(TId trafficLightId) {
        trafficLightId_ = trafficLightId;
        return *this;
    }

    common::ImageBox imageBox() const { return common::ImageBox(minX_, minY_, maxX_, maxY_); }

private:
    friend class sql_chemistry::GatewayAccess<TrafficLightFeature>;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.trafficLightId_, t.featureId_, t.minX_, t.minY_, t.maxX_, t.maxY_);
    }

    TrafficLightFeature() = default;

    TId trafficLightId_{};
    TId featureId_{};
    int32_t minX_{}, minY_{}, maxX_{}, maxY_{};

public:
    auto introspect() const { return introspect(*this); }
};

using TrafficLightFeatures = std::vector<TrafficLightFeature>;

class TrafficLightPanorama {
public:
    //WARN: constructing TrafficLightPanorama which is not bound to a sign yet
    TrafficLightPanorama(TId panoramaId, size_t minX, size_t minY, size_t maxX, size_t maxY)
        : trafficLightId_(0)
        , panoramaId_(panoramaId)
        , minX_(minX)
        , minY_(minY)
        , maxX_(maxX)
        , maxY_(maxY) {
        REQUIRE(minX_ <= maxX_ && minY_ <= maxY_,
            "SignPanorama bounding box [" << minX_ << ", " << minY_ << ", " <<
                maxX_ << ", " << maxY << "] is not correct"
        );
    }

    TId trafficLightId() const { return trafficLightId_; }

    TId panoramaId() const { return panoramaId_; }

    /// Coordinates of top corner of sign image box.
    int32_t minX() const { return minX_; }

    int32_t minY() const { return minY_; }

    /// Coordinates of bottom corner of sign image box.
    int32_t maxX() const { return maxX_; }

    int32_t maxY() const { return maxY_; }

    TrafficLightPanorama& setTrafficLightId(TId trafficLightId) {
        trafficLightId_ = trafficLightId;
        return *this;
    }

    common::ImageBox imageBox() const { return common::ImageBox(minX_, minY_, maxX_, maxY_); }

private:
    friend class sql_chemistry::GatewayAccess<TrafficLightPanorama>;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.trafficLightId_, t.panoramaId_, t.minX_, t.minY_, t.maxX_, t.maxY_);
    }

    TrafficLightPanorama() = default;

    TId trafficLightId_{};
    TId panoramaId_{};
    int32_t minX_{}, minY_{}, maxX_{}, maxY_{};

public:
    auto introspect() const { return introspect(*this); }
};

using TrafficLightPanoramas = std::vector<TrafficLightPanorama>;

class TrafficLightToloka {
public:
    TrafficLightToloka(
        TId hypothesisId,
        const std::string& workflowId,
        const std::string& workflowInstanceId,
        const std::string& blockGUID)
    : hypothesisId_(hypothesisId)
    , workflowId_(workflowId)
    , workflowInstanceId_(workflowInstanceId)
    , blockGUID_(blockGUID)
    , state_(nirvana::nirvana_state::RUNNING)
    {}

    TId hypothesisId() const { return hypothesisId_; }

    const std::string& workflowId() const {
        return workflowId_;
    }

    const std::string& workflowInstanceId() const {
        return workflowInstanceId_;
    }

    const std::string& blockGUID() const {
        return blockGUID_;
    }

    const std::string& state() const {
        return state_;
    }

    TrafficLightToloka& restartNirvana(
        const std::string& workflowId,
        const std::string& workflowInstanceId,
        const std::string& blockGUID)
    {
        workflowId_ = workflowId;
        workflowInstanceId_ = workflowInstanceId;
        blockGUID_ = blockGUID;
        state_ = nirvana::nirvana_state::RUNNING;
        return *this;
    }

    TrafficLightToloka& setState(const std::string& state) {
        state_ = state;
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<TrafficLightToloka>;

    TrafficLightToloka() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.hypothesisId_, t.workflowId_,
            t.workflowInstanceId_, t.blockGUID_, t.state_);
    }

    TId hypothesisId_{};
    std::string workflowId_;
    std::string workflowInstanceId_;
    std::string blockGUID_;
    std::string state_;

public:
    auto introspect() const { return introspect(*this); }
};

using TrafficLightTolokas = std::vector<TrafficLightToloka>;

} // namespace maps::mrc::db
