#pragma once

#include "traffic_light.h"

#include <maps/libs/sql_chemistry/include/gateway.h>

namespace maps::mrc::db {
namespace table {
using namespace sql_chemistry;

struct TrafficLight : Table<db::TrafficLight> {
    static constexpr std::string_view name_{"traffic_light.traffic_light"sv};

    static constexpr BigSerialKey id{"traffic_light_id"sv, name_};
    static constexpr MercatorColumn<geolib3::Point2> position{"position"sv, name_};
    static constexpr DoubleColumn positionVariance{"position_variance"sv, name_};
    static constexpr DoubleColumn heading{"heading"sv, name_};
    static constexpr DoubleColumn headingVariance{"heading_variance"sv, name_};

    static constexpr auto columns_() {
        return std::tie(id, position, positionVariance, heading, headingVariance);
    }
};

struct TrafficLightFeature : Table<db::TrafficLightFeature> {
    static constexpr std::string_view name_{"traffic_light.traffic_light_feature"sv};

    static constexpr NumericColumn<TId> trafficLightId{"traffic_light_id"sv, name_};
    static constexpr NumericColumn<TId> featureId{"feature_id"sv, name_};
    static constexpr NumericColumn<size_t> minX{"min_x"sv, name_};
    static constexpr NumericColumn<size_t> minY{"min_y"sv, name_};
    static constexpr NumericColumn<size_t> maxX{"max_x"sv, name_};
    static constexpr NumericColumn<size_t> maxY{"max_y"sv, name_};

    static constexpr auto columns_() { return std::tie(trafficLightId, featureId, minX, minY, maxX, maxY); }
};

struct TrafficLightPanorama : Table<db::TrafficLightPanorama> {
    static constexpr std::string_view name_{"traffic_light.traffic_light_panorama"sv};

    static constexpr NumericColumn<TId> trafficLightId{"traffic_light_id"sv, name_};
    static constexpr NumericColumn<TId> panoramaId{"panorama_id"sv, name_};
    static constexpr NumericColumn<size_t> minX{"min_x"sv, name_};
    static constexpr NumericColumn<size_t> minY{"min_y"sv, name_};
    static constexpr NumericColumn<size_t> maxX{"max_x"sv, name_};
    static constexpr NumericColumn<size_t> maxY{"max_y"sv, name_};

    static constexpr auto columns_() { return std::tie(trafficLightId, panoramaId, minX, minY, maxX, maxY); }
};

struct TrafficLightToloka : Table<db::TrafficLightToloka> {
    static constexpr std::string_view name_{"traffic_light.traffic_light_toloka"};

    static constexpr Int64PrimaryKey hypothesisId{"hypothesis_id", name_};
    static constexpr StringColumn workflowId{"workflow_id", name_};
    static constexpr StringColumn workflowInstanceId{"workflow_instance_id", name_};
    static constexpr StringColumn blockGUID{"block_guid", name_};
    static constexpr StringColumn state{"state", name_};

    static constexpr auto columns_() {
        return std::tie(hypothesisId, workflowId, workflowInstanceId, blockGUID, state);
    }
};

} // namespace table

using TrafficLightGateway = sql_chemistry::Gateway<table::TrafficLight>;

class TrafficLightFeatureGateway : public sql_chemistry::Gateway<table::TrafficLightFeature> {
public:
    using Tb = table::TrafficLightFeature;
    using Base = Gateway<Tb>;
    using Base::Base;
    using Base::Entities;

    Entities loadByTrafficLightIds(TIds trafficLightIds) {
        return load(Tb::trafficLightId.in(std::move(trafficLightIds)));
    }

    Entities loadByTrafficLightId(TId trafficLightId) {
        return load(Tb::trafficLightId == trafficLightId);
    }
};

class TrafficLightPanoramaGateway : public sql_chemistry::Gateway<table::TrafficLightPanorama> {
public:
    using Tb = table::TrafficLightPanorama;
    using Base = Gateway<Tb>;
    using Base::Base;
    using Base::Entities;

    Entities loadByTrafficLightIds(TIds trafficLightIds) {
        return load(Tb::trafficLightId.in(std::move(trafficLightIds)));
    }

    Entities loadByTrafficLightId(TId trafficLightId) {
        return load(Tb::trafficLightId == trafficLightId);
    }
};

using TrafficLightTolokaGateway = sql_chemistry::Gateway<table::TrafficLightToloka>;

} // namespace maps::mrc::db
