#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>

#include <maps/libs/sql_chemistry/include/gateway_access.h>
#include <maps/libs/introspection/include/hashing.h>
#include <maps/libs/enum_io/include/enum_io_fwd.h>

namespace maps::mrc::db::ugc {

enum class AssignmentStatus {
    Active,
    Abandoned,
    Completed,
    Accepted,
    Revoked,
};

DECLARE_ENUM_IO(AssignmentStatus);

/// use assignTo() for creating assignments
class Assignment {
public:
    TId id() const { return id_; }

    TId taskId() const { return taskId_; }

    const UserId& assignedTo() const { return assignedTo_; }

    AssignmentStatus status() const { return status_; }

    chrono::TimePoint acquiredAt() const { return acquiredAt_; }

    const OptionalTimePoint& submittedAt() const { return submittedAt_; }

    const OptionalTimePoint& evaluatedAt() const { return evaluatedAt_; }

    Assignment& markAsAbandoned() {
        status_ = AssignmentStatus::Abandoned;
        return *this;
    }

    Assignment& markAsRevoked() {
        status_ = AssignmentStatus::Revoked;
        return *this;
    }

    Assignment& markAsCompleted() {
        status_ = AssignmentStatus::Completed;
        submittedAt_ = chrono::TimePoint::clock::now();
        return *this;
    }

    Assignment& markAsAccepted() {
        status_ = AssignmentStatus::Accepted;
        evaluatedAt_ = chrono::TimePoint::clock::now();
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<Assignment>;
    friend class Task;

    Assignment() = default;

    Assignment(TId taskId, UserId assignedTo)
        : taskId_(taskId)
        , status_{AssignmentStatus::Active}
        , assignedTo_(std::move(assignedTo))
        , acquiredAt_(chrono::TimePoint::clock::now())
    {}

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.taskId_, t.status_, t.assignedTo_, t.acquiredAt_,
            t.submittedAt_, t.evaluatedAt_, t.xMin_);
    }

    TId id_{0};
    TId taskId_{};
    AssignmentStatus status_{};
    UserId assignedTo_{};
    chrono::TimePoint acquiredAt_{};
    OptionalTimePoint submittedAt_{};
    OptionalTimePoint evaluatedAt_{};

    // Never sent to client.
    uint64_t xMin_{0};

public:
    auto introspect() const { return introspect(*this); }
};

inline size_t computeElementaryHash(const Assignment& assignment) {
    return introspection::computeHash(introspection::introspect(assignment));
};

using Assignments = std::vector<Assignment>;

} // namespace maps::mrc::db::ugc
