#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/assignment_review.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>

#include <maps/libs/sql_chemistry/include/gateway.h>

namespace maps::mrc::db::ugc {
namespace table {
using namespace sql_chemistry;
using geolib3::PolylinesVector;

struct AssignmentReview : Table<db::ugc::AssignmentReview> {
    static constexpr std::string_view name_{"ugc.assignment_review"sv};

    // FOREIGN KEY ugc.assignment
    static constexpr Int64PrimaryKey assignmentId{"assignment_id"sv, name_};

    static constexpr Int16PrimaryKey cameraDeviation{
        "camera_deviation"sv, name_};

    static constexpr EnumColumn<TolokaStatus> tolokaStatus{"toloka_status"sv, name_};
    static constexpr DoubleColumn coverageFraction{"coverage_fraction"sv, name_};
    static constexpr NullableDoubleColumn goodCoverageFraction{"good_coverage_fraction"sv, name_};
    static constexpr NullableDoubleColumn trackDistanceInMeters{"track_distance_in_meters"sv, name_};
    static constexpr NullableInt64Column trackDurationInSeconds{"track_duration_in_seconds"sv, name_};
    static constexpr NullableTimePointColumn actualizationDate{"actualization_date"sv, name_};
    static constexpr NullableNumericColumn<size_t> processedPhotos{"processed_photos"sv, name_};
    static constexpr NullableNumericColumn<size_t> processedPoints{"processed_points"sv, name_};
    static constexpr NullableNumericColumn<size_t> goodPhotos{"good_photos"sv, name_};
    static constexpr NullableTimePointColumn firstShotTime{"first_shot_time"sv, name_};
    static constexpr NullableTimePointColumn lastShotTime{"last_shot_time"sv, name_};
    static constexpr NullableGeodeticColumn<PolylinesVector> coveredGeom{"covered_geom"sv, name_};
    static constexpr NullableGeodeticColumn<PolylinesVector> uncoveredGeom{"uncovered_geom"sv, name_};
    static constexpr NullableGeodeticColumn<PolylinesVector> trackGeom{"track_geom"sv, name_};

    static constexpr auto columns_() {
        return std::tie(assignmentId, tolokaStatus, coverageFraction,
            goodCoverageFraction, trackDistanceInMeters, trackDurationInSeconds,
            actualizationDate, processedPhotos, processedPoints, goodPhotos,
            firstShotTime, lastShotTime, coveredGeom, uncoveredGeom, trackGeom,
            cameraDeviation);
    }
};

} // namespace table

class AssignmentReviewGateway : public sql_chemistry::Gateway<table::AssignmentReview> {
public:
    using sql_chemistry::Gateway<table::AssignmentReview>::Gateway;

    template<class... Clauses>
    AssignmentReviews loadWithoutGeometry(const Clauses&... clauses)
    {
        using Tuple = std::tuple<
            TId,
            TolokaStatus,
            double,
            std::optional<double>,
            std::optional<double>,
            std::optional<int64_t>,
            std::optional<chrono::TimePoint>,
            std::optional<size_t>,
            std::optional<size_t>,
            std::optional<size_t>,
            std::optional<chrono::TimePoint>,
            std::optional<chrono::TimePoint>,
            int16_t>;

        auto tuples = this->loadMultipleColumns<Tuple>(
            std::tie(
                Table::assignmentId,
                Table::tolokaStatus,
                Table::coverageFraction,
                Table::goodCoverageFraction,
                Table::trackDistanceInMeters,
                Table::trackDurationInSeconds,
                Table::actualizationDate,
                Table::processedPhotos,
                Table::processedPoints,
                Table::goodPhotos,
                Table::firstShotTime,
                Table::lastShotTime,
                Table::cameraDeviation),
            clauses...);

        AssignmentReviews result;
        for (const auto& tuple: tuples) {
            auto& review = result.emplace_back(std::get<0>(tuple))
                .setTolokaStatus(std::get<1>(tuple))
                .setCoverageFraction(std::get<2>(tuple))
                .setCameraDeviation(static_cast<CameraDeviation>(std::get<12>(tuple)));
            if (auto& opt = std::get<3>(tuple)) {
                review.setGoodCoverageFraction(*opt);
            }
            if (auto& opt = std::get<4>(tuple)) {
                review.setTrackDistanceInMeters(*opt);
            }
            if (auto& opt = std::get<5>(tuple)) {
                review.setTrackDuration(std::chrono::seconds{*opt});
            }
            if (auto& opt = std::get<6>(tuple)) {
                review.setActualizationDate(*opt);
            }
            if (auto& opt = std::get<7>(tuple)) {
                review.setProcessedPhotos(*opt);
            }
            if (auto& opt = std::get<8>(tuple)) {
                review.setProcessedPoints(*opt);
            }
            if (auto& opt = std::get<9>(tuple)) {
                review.setGoodPhotos(*opt);
            }
            if (auto& opt = std::get<10>(tuple)) {
                review.setFirstShotTime(*opt);
            }
            if (auto& opt = std::get<11>(tuple)) {
                review.setLastShotTime(*opt);
            }
        }
        return result;
    }
};

} // namespace maps::mrc::db::ugc
