#pragma once

#include "common.h"
#include "dataset.h"

#include <maps/libs/sql_chemistry/include/gateway_access.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/polygon.h>

#include <vector>

namespace maps::mrc::db {

class Video {
public:
    Video(Dataset dataset,
          std::string sourceId,
          chrono::TimePoint startedAt,
          float durationSeconds,
          mds::Key mdsKey)
        : dataset_(dataset)
        , sourceId_(std::move(sourceId))
        , startedAt_(startedAt)
        , durationSeconds_(durationSeconds)
        , mdsGroupId_(std::move(mdsKey.groupId))
        , mdsPath_(std::move(mdsKey.path))
    {}

    TId id() const { return id_; }

    Dataset dataset() const { return dataset_; }

    const std::string& sourceId() const { return sourceId_; }

    chrono::TimePoint startedAt() const { return startedAt_; }

    float durationSeconds() const { return durationSeconds_; }

    mds::Key mdsKey() const { return {mdsGroupId_, mdsPath_}; }
    const std::string& mdsGroupId() const { return mdsGroupId_; }
    const std::string& mdsPath() const { return mdsPath_; }

    const std::optional<TId>& eventId() const { return eventId_; }

    const std::optional<std::int64_t>& eventHash() const { return eventHash_; }

    Video& setSourceId(std::string sourceId) {
        sourceId_ = std::move(sourceId);
        return *this;
    }

    Video& setTimestamp(chrono::TimePoint timePoint) {
        startedAt_ = timePoint;
        return *this;
    }

    Video& setDuration(float durationSeconds) {
        durationSeconds_ = durationSeconds;
        return *this;
    }

    Video& setMdsKey(mds::Key key) {
        mdsGroupId_ = std::move(key.groupId);
        mdsPath_ = std::move(key.path);
        return *this;
    }

    Video& setEventId(TId eventId) {
        eventId_ = eventId;
        return *this;
    }

    Video& setEventHash(std::int64_t eventHash) {
        eventHash_ = eventHash;
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<Video>;

    Video() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.dataset_, t.sourceId_,
                        t.startedAt_, t.durationSeconds_,
                        t.mdsGroupId_, t.mdsPath_, t.eventId_, t.eventHash_);
    }

    TId id_{0};
    Dataset dataset_;
    std::string sourceId_;
    chrono::TimePoint startedAt_;
    float durationSeconds_;
    std::string mdsGroupId_;
    std::string mdsPath_;
    std::optional<TId> eventId_;
    std::optional<std::int64_t> eventHash_;

public:
    auto introspect() const { return introspect(*this); }
};

using Videos = std::vector<Video>;


class FrameToVideo {
public:
    FrameToVideo(TId frameId,
          TId videoId,
          float secondsFromStart)
        : frameId_(frameId)
        , videoId_(videoId)
        , secondsFromStart_(secondsFromStart)
    {}

    TId id() const { return id_; }

    TId frameId() const { return frameId_; }

    TId videoId() const { return videoId_; }

    float secondsFromStart() const { return secondsFromStart_; }

    FrameToVideo& setSecondsFromStart(float secondsFromStart) {
        secondsFromStart_ = secondsFromStart;
        return *this;
    }

private:
    friend class sql_chemistry::GatewayAccess<FrameToVideo>;

    FrameToVideo() = default;

    template <typename T>
    static auto introspect(T& t) {
        return std::tie(t.id_, t.frameId_, t.videoId_, t.secondsFromStart_);
    }

    TId id_{0};
    TId frameId_;
    TId videoId_;
    float secondsFromStart_;

public:
    auto introspect() const { return introspect(*this); }
};

using FrameToVideos = std::vector<FrameToVideo>;

} // namespace maps::mrc::db
