#include "fixture.h"

#include <assignment_object_feedback_task.h>
#include <assignment_object_feedback_task_gateway.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>
#include <maps/libs/geolib/include/polygon.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

namespace maps::mrc::db::tests {

TEST_F(Fixture, test_assignment_object_feedback_task) {
    const geolib3::MultiPolygon2 TEST_HULL{
        {geolib3::Polygon2{
            geolib3::PointsVector{{0, 0}, {0, 1}, {1, 1}, {0, 0}}},
            geolib3::Polygon2{
                geolib3::PointsVector{{0, 0}, {-1, 0}, {-1, -1}, {0, 0}}}}};
    const std::chrono::seconds TEST_DURATION = std::chrono::hours(24) * 80;
    const double TEST_DISTANCE_IN_METERS = 40000000;
    const std::string TEST_USER_ID = "0xDEADBEAF";
    const std::string TEST_FEEDBACK_TASK_ID = "100500";

    TId taskId = 0;
    TId assignmentId = 0;
    TId postedObjectId = 0;
    TId notPostedObjectId = 0;

    // Create a task
    {
        auto txn = txnHandle();

        ugc::Task task{};
        task.setStatus(ugc::TaskStatus::New)
            .setDuration(TEST_DURATION)
            .setDistanceInMeters(TEST_DISTANCE_IN_METERS)
            .setGeodeticHull(TEST_HULL)
            .addName(boost::lexical_cast<Locale>("ru"), "Таска");
        ugc::TaskGateway{*txn}.insert(task);
        taskId = task.id();
        txn->commit();
    }

    // Assign it and save some assignment objects
    {
        auto txn = txnHandle();
        auto task = ugc::TaskGateway{*txn}.loadById(taskId);
        auto assignment = task.assignTo(TEST_USER_ID);
        ugc::TaskGateway{*txn}.update(task);
        ugc::AssignmentGateway{*txn}.insert(assignment);
        assignmentId = assignment.id();

        ugc::AssignmentObjects objects;
        objects.emplace_back(assignmentId, assignment.acquiredAt(),
            geolib3::Point2(1, 2),
            ugc::AssignmentObjectType::Barrier);
        objects.emplace_back(assignmentId, assignment.acquiredAt(),
            geolib3::Point2(3, 4),
            ugc::AssignmentObjectType::Deadend);
        ugc::AssignmentObjectGateway{*txn}.insert(objects);

        AssignmentObjectFeedbackTasks feedbackTasks;
        for (const auto& object : objects) {
            feedbackTasks.emplace_back(object.objectId());
        }
        // "Post" one feedback task and leave another one unposted
        feedbackTasks.back().setFeedbackTaskId(TEST_FEEDBACK_TASK_ID);
        AssignmentObjectFeedbackTaskGateway{*txn}.upsert(feedbackTasks);

        txn->commit();

        notPostedObjectId = feedbackTasks.at(0).objectId();
        postedObjectId = feedbackTasks.back().objectId();
    }

    // Check there is one posted feedback task and one not posted yet
    {
        auto txn = txnHandle();
        AssignmentObjectFeedbackTaskGateway feedbackGtw(*txn);

        const auto postedFeedbackTasks = feedbackGtw.loadWithFeedback();
        EXPECT_EQ(postedFeedbackTasks.size(), 1u);
        EXPECT_EQ(postedFeedbackTasks.at(0).objectId(), postedObjectId);
        EXPECT_EQ(postedFeedbackTasks.at(0).feedbackTaskId(), TEST_FEEDBACK_TASK_ID);

        const auto notPostedFeedbackTasks = feedbackGtw.loadWithoutFeedback();
        EXPECT_EQ(notPostedFeedbackTasks.size(), 1u);
        EXPECT_EQ(notPostedFeedbackTasks.at(0).objectId(), notPostedObjectId);
        EXPECT_FALSE(notPostedFeedbackTasks.at(0).feedbackTaskId());

        const auto postedObjects = ugc::AssignmentObjectGateway{*txn}.loadWithFeedback();
        EXPECT_EQ(postedObjects.size(), 1u);
        EXPECT_EQ(postedObjects.at(0).objectId(), postedObjectId);

        const auto notPostedObjects = ugc::AssignmentObjectGateway{*txn}.loadWithoutFeedback();
        EXPECT_EQ(notPostedObjects.size(), 1u);
        EXPECT_EQ(notPostedObjects.at(0).objectId(), notPostedObjectId);
    }
}

} // namespace maps::mrc::db::tests
