#include <maps/wikimap/mapspro/services/mrc/libs/db/tests/eye/common.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_attrs.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/object_gateway.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

namespace maps::mrc::db::eye::tests {

TEST(eye_object, traffic_light_attrs)
{
    const json::Value json = json::Value::fromString(R"({})");
    TrafficLightAttrs attrs = TrafficLightAttrs::fromJson(json);

    EXPECT_EQ(toJson(attrs), json);
}

TEST(eye_object, sign_attrs)
{
    const json::Value json = json::Value::fromString(R"(
        {
          "type" : "prohibitory_no_parking",
          "temporary" : true
        }
    )");
    const SignAttrs attrs = SignAttrs::fromJson(json);

    EXPECT_EQ(toJson(attrs), json);

    EXPECT_EQ(attrs.type, traffic_signs::TrafficSign::ProhibitoryNoParking);
    EXPECT_EQ(attrs.temporary, true);
}

TEST(eye_object, road_marking_attrs)
{
    const json::Value json = json::Value::fromString(R"(
        {
          "type" : "road_marking_lane_direction_f"
        }
    )");
    const RoadMarkingAttrs attrs = RoadMarkingAttrs::fromJson(json);

    EXPECT_EQ(toJson(attrs), json);

    EXPECT_EQ(attrs.type, traffic_signs::TrafficSign::RoadMarkingLaneDirectionF);
}

TEST(eye_object, house_number_attrs)
{
    const json::Value json = json::Value::fromString(R"(
        {
          "number" : "11"
        }
    )");
    const HouseNumberAttrs attrs = HouseNumberAttrs::fromJson(json);

    EXPECT_EQ(toJson(attrs), json);

    EXPECT_EQ(attrs.number, "11");
}

TEST_F(DetectionFixture, insert_object)
{
    const traffic_signs::TrafficSign signType = traffic_signs::TrafficSign::ProhibitoryNoParking;
    const bool temporary = true;
    const traffic_signs::TrafficSign roadMarkingType = traffic_signs::TrafficSign::RoadMarkingLaneDirectionF;
    const std::string number = "11";

    const TrafficLightAttrs trafficLightAttrs;
    const SignAttrs signAttrs{signType, temporary};
    const RoadMarkingAttrs roadMarkingAttrs{roadMarkingType};
    const HouseNumberAttrs houseNumberAttrs{number};

    Objects objects {
        {
            detections[0].id(),
            trafficLightAttrs
        },
        {
            detections[1].id(),
            signAttrs
        },
        {
            detections[2].id(),
            roadMarkingAttrs
        },
        {
            detections[3].id(),
            houseNumberAttrs
        },
    };

    auto txn = txnHandle();
    const TId txnId = ObjectGateway(*txn).insertx(objects);
    txn->commit();

    {   // check first
        const auto object = ObjectGateway(*txnHandle()).loadById(objects[0].id());

        EXPECT_EQ(object.txnId(), txnId);
        EXPECT_EQ(object.primaryDetectionId(), detections[0].id());
        EXPECT_EQ(object.deleted(), false);
        EXPECT_EQ(object.type(), ObjectType::TrafficLight);
        EXPECT_EQ(toJson(object.attrs<TrafficLightAttrs>()), toJson(trafficLightAttrs));
    }

    {   // check second
        const auto object = ObjectGateway(*txnHandle()).loadById(objects[1].id());

        EXPECT_EQ(object.txnId(), txnId);
        EXPECT_EQ(object.primaryDetectionId(), detections[1].id());
        EXPECT_EQ(object.deleted(), false);
        EXPECT_EQ(object.type(), ObjectType::Sign);
        EXPECT_EQ(toJson(object.attrs<SignAttrs>()), toJson(signAttrs));
    }

    {   // check third
        const auto object = ObjectGateway(*txnHandle()).loadById(objects[2].id());

        EXPECT_EQ(object.txnId(), txnId);
        EXPECT_EQ(object.primaryDetectionId(), detections[2].id());
        EXPECT_EQ(object.deleted(), false);
        EXPECT_EQ(object.type(), ObjectType::RoadMarking);
        EXPECT_EQ(toJson(object.attrs<RoadMarkingAttrs>()), toJson(roadMarkingAttrs));
    }

    {   // check forth
        const auto object = ObjectGateway(*txnHandle()).loadById(objects[3].id());

        EXPECT_EQ(object.txnId(), txnId);
        EXPECT_EQ(object.primaryDetectionId(), detections[3].id());
        EXPECT_EQ(object.deleted(), false);
        EXPECT_EQ(object.type(), ObjectType::HouseNumber);
        EXPECT_EQ(toJson(object.attrs<HouseNumberAttrs>()), toJson(houseNumberAttrs));
    }
}

TEST_F(DetectionFixture, update_object)
{
    Object object(detections[0].id(), ObjectType::TrafficLight);
    const auto date = chrono::parseSqlDateTime("2016-04-02 05:57:11+03");
    TId txnId;
    {
        auto txn = txnHandle();
        txnId = ObjectGateway(*txn).insertx(object);
        txn->commit();
    }

    TId updatedTxnId;
    {
        auto txn = txnHandle();
        object.setDeleted(true);
        object.setDisappearedAt(date);
        updatedTxnId = ObjectGateway(*txn).updatex(object);
        txn->commit();
    }

    {    // Check value
        const auto updated = ObjectGateway(*txnHandle()).loadById(object.id());

        EXPECT_TRUE(txnId < updatedTxnId);
        EXPECT_EQ(updated.txnId(), updatedTxnId);
        EXPECT_EQ(updated.deleted(), true);
        EXPECT_EQ(updated.disappearedAt(), date);
    }
}

TEST_F(DetectionFixture, unique_object)
{
    const TrafficLightAttrs trafficLightAttrs;

    Objects objects {
        {
            detections[0].id(),
            trafficLightAttrs
        },
        {
            detections[0].id(),
            trafficLightAttrs
        },
    };

    EXPECT_THROW(
        ObjectGateway(*txnHandle()).insertx(objects),
        maps::sql_chemistry::UniqueViolationError
    );
}

TEST_F(ObjectFixture, insert_object_location)
{
    auto txn = txnHandle();

    ObjectLocations locations {
        {
            objects[0].id(),
            {5.0, 3.0},
            Eigen::Quaterniond::Identity()
        },
        {
            objects[1].id(),
            {100.0, 53.0},
            Eigen::Quaterniond(
                Eigen::AngleAxisd(M_PI, Eigen::Vector3d::UnitZ())
            )
        },
    };

    const TId txnId = ObjectLocationGateway(*txn).insertx(locations);

    txn->commit();

    {   // first
        auto txn = txnHandle();
        const auto location = ObjectLocationGateway(*txn).loadById(locations[0].objectId());
        EXPECT_EQ(location.txnId(), txnId);
        const geolib3::Point2 mercator{5.0, 3.0};
        const auto geodetic = geolib3::convertMercatorToGeodetic(mercator);
        EXPECT_TRUE(geolib3::test_tools::approximateEqual(location.mercatorPos(), mercator, geolib3::EPS));
        EXPECT_TRUE(geolib3::test_tools::approximateEqual(location.geodeticPos(), geodetic, geolib3::EPS));
        const auto rotation = Eigen::Quaterniond::Identity();
        EXPECT_TRUE(location.rotation().isApprox(rotation, geolib3::EPS));
    }

    {   // second
        auto txn = txnHandle();
        const auto location = ObjectLocationGateway(*txn).loadById(locations[1].objectId());
        EXPECT_EQ(location.txnId(), txnId);
        const geolib3::Point2 mercator{100.0, 53.0};
        const auto geodetic = geolib3::convertMercatorToGeodetic(mercator);
        EXPECT_TRUE(geolib3::test_tools::approximateEqual(location.mercatorPos(), mercator, geolib3::EPS));
        EXPECT_TRUE(geolib3::test_tools::approximateEqual(location.geodeticPos(), geodetic, geolib3::EPS));
        const Eigen::Quaterniond rotation(Eigen::AngleAxisd(M_PI, Eigen::Vector3d::UnitZ()));
        EXPECT_TRUE(location.rotation().isApprox(rotation, geolib3::EPS));
    }
}

TEST_F(ObjectFixture, unique_object_location)
{
    auto txn = txnHandle();
    ObjectLocations locations {
        {objects[0].id(), {1.0, 0.0}, Eigen::Quaterniond::Identity()},
        {objects[0].id(), {0.0, 1.0}, Eigen::Quaterniond::Identity()},
    };
    EXPECT_THROW(
        ObjectLocationGateway(*txn).insertx(locations),
        maps::sql_chemistry::UniqueViolationError
    );
}

TEST_F(ObjectFixture, update_object_location)
{
    ObjectLocations locations {
        {
            objects[0].id(),
            {5.0, 3.0},
            Eigen::Quaterniond::Identity()
        },
        {
            objects[2].id(),
            {2.0, 4.0},
            Eigen::Quaterniond(
                Eigen::AngleAxisd(M_PI, Eigen::Vector3d::UnitZ())
            )
        },
        {
            objects[3].id(),
            {-2.0, -4.0},
            Eigen::Quaterniond(
                Eigen::AngleAxisd(M_PI, Eigen::Vector3d::UnitZ())
            )
        },
    };

    auto txn = txnHandle();
    ObjectLocationGateway(*txn).insertx(locations);
    txn->commit();

    TId txnId;
    {   // update
        auto txn = txnHandle();
        locations[0].setMercatorPos(geolib3::Point2(50, 10));
        locations[1].setGeodeticPos(geolib3::Point2(30, 40));
        locations[2].setRotation(
            Eigen::Quaterniond(
                Eigen::AngleAxisd(-M_PI, Eigen::Vector3d::UnitY())
            )
        );

        txnId = ObjectLocationGateway(*txn).updatex(locations);
        txn->commit();
    }

    {   // first
        auto txn = txnHandle();
        const auto location = ObjectLocationGateway(*txn).loadById(locations[0].objectId());
        EXPECT_EQ(location.txnId(), txnId);
        const geolib3::Point2 position{50, 10};
        EXPECT_EQ(location.mercatorPos(), position);
    }
    {   // second
        auto txn = txnHandle();
        const auto location = ObjectLocationGateway(*txn).loadById(locations[1].objectId());
        EXPECT_EQ(location.txnId(), txnId);
        const geolib3::Point2 position{30, 40};
        EXPECT_EQ(location.geodeticPos(), position);
    }
    {   // third
        auto txn = txnHandle();
        const auto location = ObjectLocationGateway(*txn).loadById(locations[2].objectId());
        EXPECT_EQ(location.txnId(), txnId);
        const Eigen::Quaterniond rotation(Eigen::AngleAxisd(-M_PI, Eigen::Vector3d::UnitY()));
        EXPECT_TRUE(location.rotation().isApprox(rotation, geolib3::EPS));
    }
}


TEST_F(ObjectFixture, insert_primary_detection_relation)
{
    auto txn = txnHandle();

    PrimaryDetectionRelation detectionRelation {
        detections[0].id(),
        detections[4].id()
    };

    const TId txnId = PrimaryDetectionRelationGateway(*txn).insertx(detectionRelation);

    txn->commit();

    {
        auto txn = txnHandle();
        const auto loadedRelation = PrimaryDetectionRelationGateway(*txn).loadById(detectionRelation.id());

        EXPECT_EQ(loadedRelation.txnId(), txnId);
        EXPECT_EQ(loadedRelation.primaryDetectionId(), detections[0].id());
        EXPECT_EQ(loadedRelation.detectionId(), detections[4].id());
        EXPECT_EQ(loadedRelation.deleted(), false);
    }
}

TEST_F(ObjectFixture, unique_relation)
{
    auto txn = txnHandle();
    PrimaryDetectionRelations detectionRelations {
        {detections[0].id(), detections[4].id()},
        {detections[0].id(), detections[4].id()},
    };
    EXPECT_THROW(
        PrimaryDetectionRelationGateway(*txn).insertx(detectionRelations),
        maps::sql_chemistry::UniqueViolationError
    );
}

TEST_F(ObjectFixture, unique_object_detection)
{
    auto txn = txnHandle();
    PrimaryDetectionRelations detectionRelations {
        {detections[0].id(), detections[4].id()},
        {detections[1].id(), detections[4].id()},
    };
    EXPECT_THROW(
        PrimaryDetectionRelationGateway(*txn).insertx(detectionRelations),
        maps::sql_chemistry::UniqueViolationError
    );
}

TEST_F(ObjectFixture, create_relation_for_not_existing_object)
{
    PrimaryDetectionRelations relations {{detections[4].id(), detections[0].id()}};

    auto txn = txnHandle();
    PrimaryDetectionRelationGateway(*txn).insertx(relations);
    EXPECT_THROW(txn->commit(), pqxx::integrity_constraint_violation);
}

TEST_F(ObjectFixture, create_relation_for_deleted_object)
{
    {
        auto object = objects[0];
        object.setDeleted(true);

        auto txn = txnHandle();
        ObjectGateway(*txn).upsertx(object);
        txn->commit();
    }

    PrimaryDetectionRelations relations {{detections[0].id(), detections[4].id()}};

    auto txn = txnHandle();
    PrimaryDetectionRelationGateway(*txn).insertx(relations);
    EXPECT_THROW(txn->commit(), pqxx::integrity_constraint_violation);
}

TEST_F(ObjectFixture, delete_object_keeping_relation)
{
    {
        PrimaryDetectionRelations relations {{detections[0].id(), detections[4].id()}};

        auto txn = txnHandle();
        PrimaryDetectionRelationGateway(*txn).insertx(relations);
        txn->commit();
    }

    auto object = objects[0];
    object.setDeleted(true);

    auto txn = txnHandle();
    ObjectGateway(*txn).upsertx(object);
    EXPECT_THROW(txn->commit(), pqxx::integrity_constraint_violation);
}

TEST_F(ObjectFixture, update_primary_detection_relation)
{
    PrimaryDetectionRelation detectionRelation{
        detections[0].id(),
        detections[4].id()
    };

    auto txn = txnHandle();
    TId txnId = PrimaryDetectionRelationGateway(*txn).insertx(detectionRelation);
    txn->commit();

    TId updatedTxnId;
    {   // update
        auto txn = txnHandle();
        detectionRelation.setDeleted(true);
        updatedTxnId = PrimaryDetectionRelationGateway(*txn).updatex(detectionRelation);
        txn->commit();
    }

    {   // first
        auto txn = txnHandle();
        const auto loadedRelation = PrimaryDetectionRelationGateway(*txn).loadById(detectionRelation.id());

        EXPECT_TRUE(txnId < updatedTxnId);
        EXPECT_EQ(loadedRelation.txnId(), updatedTxnId);
        EXPECT_EQ(loadedRelation.primaryDetectionId(), detections[0].id());
        EXPECT_EQ(loadedRelation.detectionId(), detections[4].id());
        EXPECT_EQ(loadedRelation.deleted(), true);
    }
}

TEST_F(ObjectFixture, create_object_missing_on_frame)
{
    const auto objectId = objects.at(0).id();
    const auto frameId  = frames.at(0).id();
    ObjectMissingOnFrame missing(objectId, frameId);

    auto txn = txnHandle();
    TId txnId = ObjectMissingOnFrameGateway(*txn).insertx(missing);
    txn->commit();

    TId updatedTxnId;
    {   // update
        auto txn = txnHandle();
        missing.setDeleted(true);
        updatedTxnId = ObjectMissingOnFrameGateway(*txn).updatex(missing);
        txn->commit();
    }

    {   // load
        auto txn = txnHandle();
        const auto loadedMissing = ObjectMissingOnFrameGateway(*txn).loadById(missing.id());

        EXPECT_TRUE(txnId < updatedTxnId);
        EXPECT_EQ(loadedMissing.txnId(), updatedTxnId);
        EXPECT_EQ(loadedMissing.objectId(), objectId);
        EXPECT_EQ(loadedMissing.frameId(), frameId);
        EXPECT_EQ(loadedMissing.deleted(), true);
    }

    {
        // check unique
        ObjectMissingOnFrame otherMissing(objectId, frameId);
        auto txn = txnHandle();
        EXPECT_THROW(
            ObjectMissingOnFrameGateway(*txn).insertx(otherMissing),
            maps::sql_chemistry::UniqueViolationError);
    }
}


TEST_F(DetectionFixture, check_eye_relation_and_object_consistency)
{
    auto txn = txnHandle();

    db::eye::Object txnObject(detections[0].id(), db::eye::HouseNumberAttrs{"12"});
    db::eye::ObjectGateway(*txn).insertx(txnObject);

    db::eye::PrimaryDetectionRelation txnRelation(detections[0].id(), detections[1].id());
    db::eye::PrimaryDetectionRelationGateway(*txn).insertx(txnRelation);

    txnRelation.setDeleted(true);
    db::eye::PrimaryDetectionRelationGateway(*txn).updatex(txnRelation);

    txnObject.setDeleted(true);
    db::eye::ObjectGateway(*txn).updatex(txnObject);

    txn->commit();
}

TEST_F(DetectionFixture, check_eye_object_has_no_relation)
{
    auto txn = txnHandle();

    db::eye::Object txnObject(detections[0].id(), db::eye::HouseNumberAttrs{"12"});
    txnObject.setDeleted(true);
    db::eye::ObjectGateway(*txn).insertx(txnObject);

    txnObject.setDeleted(false);
    db::eye::ObjectGateway(*txn).updatex(txnObject);

    db::eye::PrimaryDetectionRelation txnRelation(detections[0].id(), detections[1].id());
    db::eye::PrimaryDetectionRelationGateway(*txn).insertx(txnRelation);

    txn->commit();
}

TEST_F(DetectionFixture, insert_object_relation)
{
    const traffic_signs::TrafficSign signType = traffic_signs::TrafficSign::ProhibitoryNoParking;
    const traffic_signs::TrafficSign tableType = traffic_signs::TrafficSign::InformationStartZone;

    const SignAttrs signAttrs{signType, false};
    const SignAttrs tableAttrs{tableType, false};

    Objects objects {
        {
            detections[0].id(),
            signAttrs
        },
        {
            detections[1].id(),
            tableAttrs
        }
    };

    auto txnObject = txnHandle();
    ObjectGateway(*txnObject).insertx(objects);
    txnObject->commit();

    ObjectRelations relations {
        {objects[0].id(), objects[1].id()}
    };
    auto txnRelation = txnHandle();
    const TId txnId = ObjectRelationGateway(*txnRelation).insertx(relations);
    txnRelation->commit();

    EXPECT_EQ(relations[0].txnId(), txnId);
    EXPECT_EQ(relations[0].deleted(), false);
    EXPECT_EQ(relations[0].masterObjectId(), objects[0].id());
    EXPECT_EQ(relations[0].slaveObjectId(), objects[1].id());
}

} // namespace maps::mrc::db::eye::tests
