#include <maps/wikimap/mapspro/services/mrc/libs/db/tests/eye/common.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/frame_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/recognition_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

namespace maps::mrc::db::eye::tests {

TEST(eye_recognition, make_detected_panel_value)
{
    const common::ImageBox box(0, 0, 200, 300);
    const DetectedPanel detectedPanel{box};

    const auto expected = json::Value::fromString(R"({"box":[[0, 0],[200, 300]]})");
    EXPECT_EQ(toJson(detectedPanel), expected);

    EXPECT_EQ(detectedPanel.restBox, box);
}

TEST(eye_recognition, make_detected_sign_value)
{
    const common::ImageBox box(0, 0, 200, 300);
    const auto type = traffic_signs::TrafficSign::ProhibitoryNoParking;
    const double typeConfidence = 0.98;
    const bool temporary = false;
    const double temporaryConfidence = 0.78;

    const DetectedSigns detectedSigns({
        DetectedSign{box, type, typeConfidence, temporary, temporaryConfidence}
    });

    const std::string expectedSign = R"({
        "box": [[0, 0], [200, 300]],
        "tp": "prohibitory_no_parking",
        "tp_cnf": 0.98,
        "tmp": false,
        "tmp_cnf": 0.78
    })";

    const json::Value expectedSignJson = json(expectedSign);
    const json::Value expectedSignsJson = json("[" + expectedSign + "]");


    EXPECT_EQ(toJson(detectedSigns), expectedSignsJson);

    const auto sign = detectedSigns[0];
    EXPECT_EQ(toJson(sign), expectedSignJson);

    EXPECT_EQ(sign.box, box);
    EXPECT_EQ(sign.type, type);
    EXPECT_DOUBLE_EQ(sign.typeConfidence, typeConfidence);
    EXPECT_EQ(sign.temporary, temporary);
    EXPECT_DOUBLE_EQ(sign.temporaryConfidence, temporaryConfidence);
}

TEST(eye_recognition, make_detected_traffic_light_value)
{
    const common::ImageBox box(0, 0, 200, 300);
    const double confidence = 0.98;

    const DetectedTrafficLights detectedTrafficLights({
        DetectedTrafficLight {box, confidence}
    });

    const std::string expectedTrafficLight = R"({
        "box": [[0, 0], [200, 300]],
        "cnf": 0.98
    })";

    const json::Value expectedTrafficLightJson = json(expectedTrafficLight);
    const json::Value expectedTrafficLightsJson = json("[" + expectedTrafficLight + "]");

    EXPECT_EQ(toJson(detectedTrafficLights), expectedTrafficLightsJson);

    const auto trafficLight = detectedTrafficLights[0];
    EXPECT_EQ(toJson(trafficLight), expectedTrafficLightJson);

    EXPECT_EQ(trafficLight.box, box);
    EXPECT_DOUBLE_EQ(trafficLight.confidence, confidence);
}

TEST(eye_recognition, make_detected_house_number_value)
{
    const common::ImageBox box(0, 0, 200, 300);
    const double confidence = 0.98;
    const std::string number = "22";

    const DetectedHouseNumbers detectedHouseNumbers({
        DetectedHouseNumber {box, confidence, number}
    });

    const std::string expectedHouseNumber = R"({
        "box": [[0, 0], [200, 300]],
        "cnf": 0.98,
        "number": "22"
    })";

    const json::Value expectedHouseNumberJson = json(expectedHouseNumber);
    const json::Value expectedHouseNumbersJson = json("[" + expectedHouseNumber + "]");

    EXPECT_EQ(toJson(detectedHouseNumbers), expectedHouseNumbersJson);

    const auto houseNumber = detectedHouseNumbers[0];
    EXPECT_EQ(toJson(houseNumber), expectedHouseNumberJson);

    EXPECT_EQ(houseNumber.box, box);
    EXPECT_DOUBLE_EQ(houseNumber.confidence, confidence);
    EXPECT_EQ(houseNumber.number, number);
}

TEST(eye_recognition, make_detected_road_marking_value)
{
    const common::ImageBox box(0, 0, 200, 300);
    const auto type = traffic_signs::TrafficSign::RoadMarkingLaneDirectionF;
    const double confidence = 0.98;

    const DetectedRoadMarkings detectedRoadMarkings({
        DetectedRoadMarking  {box, type, confidence}
    });

    const std::string expectedRoadMarking = R"({
        "box": [[0, 0], [200, 300]],
        "tp": "road_marking_lane_direction_f",
        "cnf": 0.98
    })";

    const json::Value expectedRoadMarkingJson = json(expectedRoadMarking);
    const json::Value expectedRoadMarkingsJson = json("[" + expectedRoadMarking + "]");

    EXPECT_EQ(toJson(detectedRoadMarkings), expectedRoadMarkingsJson);

    const auto roadMarking = detectedRoadMarkings[0];
    EXPECT_EQ(toJson(roadMarking), expectedRoadMarkingJson);

    EXPECT_EQ(roadMarking.box, box);
    EXPECT_EQ(roadMarking.type, type);
    EXPECT_EQ(roadMarking.confidence, confidence);
}

TEST_F(FrameFixture, insert_recognition)
{
    const DetectedSigns value {
        DetectedSign {
            common::ImageBox(100, 100, 200, 200),
            traffic_signs::TrafficSign::ProhibitoryNoParking,
            0.98,
            false,
            0.78
        }
    };

    Recognitions recognitions {
        {
            frames[0].id(),
            common::ImageOrientation(common::Rotation::CW_0),
            RecognitionType::DetectSign,
            RecognitionSource::Model,
            RECOGNITION_VERSION,
            value
        },
        {
            frames[0].id(),
            common::ImageOrientation(common::Rotation::CW_90),
            RecognitionType::DetectSign,
            RecognitionSource::Model,
            RECOGNITION_VERSION,
            value
        },
        {
            frames[0].id(),
            common::ImageOrientation(common::Rotation::CW_0),
            RecognitionType::DetectSign,
            RecognitionSource::Toloka,
            RECOGNITION_VERSION,
        },
        {
            frames[1].id(),
            common::ImageOrientation(common::Rotation::CW_90),
            RecognitionType::DetectSign,
            RecognitionSource::Toloka,
            RECOGNITION_VERSION,
        },
    };

    auto txn = txnHandle();
    const TId txnId = RecognitionGateway(*txn).insertx(recognitions);
    txn->commit();

    {   // check first
        const auto recognition = RecognitionGateway(*txnHandle()).loadById(recognitions[0].id());

        EXPECT_EQ(recognition.txnId(), txnId);
        EXPECT_EQ(recognition.frameId(), frames[0].id());

        const common::ImageOrientation orientation(common::Rotation::CW_0);
        EXPECT_EQ(recognition.orientation(), orientation);

        EXPECT_EQ(recognition.type(), RecognitionType::DetectSign);
        EXPECT_EQ(recognition.source(), RecognitionSource::Model);
        EXPECT_EQ(recognition.version(), RECOGNITION_VERSION);

        EXPECT_EQ(toJson(recognition.value<DetectedSigns>()), toJson(value));
    }

    {   // check second
        const auto recognition = RecognitionGateway(*txnHandle()).loadById(recognitions[1].id());

        EXPECT_EQ(recognition.txnId(), txnId);
        EXPECT_EQ(recognition.frameId(), frames[0].id());

        const common::ImageOrientation orientation(common::Rotation::CW_90);
        EXPECT_EQ(recognition.orientation(), orientation);

        EXPECT_EQ(recognition.type(), RecognitionType::DetectSign);
        EXPECT_EQ(recognition.source(), RecognitionSource::Model);
        EXPECT_EQ(recognition.version(), RECOGNITION_VERSION);

        EXPECT_EQ(toJson(recognition.value<DetectedSigns>()), toJson(value));
    }

    {   // check third
        const auto recognition = RecognitionGateway(*txnHandle()).loadById(recognitions[2].id());

        EXPECT_EQ(recognition.txnId(), txnId);
        EXPECT_EQ(recognition.frameId(), frames[0].id());

        const common::ImageOrientation orientation(common::Rotation::CW_0);
        EXPECT_EQ(recognition.orientation(), orientation);

        EXPECT_EQ(recognition.type(), RecognitionType::DetectSign);
        EXPECT_EQ(recognition.source(), RecognitionSource::Toloka);
        EXPECT_EQ(recognition.version(), RECOGNITION_VERSION);

        EXPECT_EQ(recognition.hasValue(), false);
    }

    {   // check forth
        const auto recognition = RecognitionGateway(*txnHandle()).loadById(recognitions[3].id());

        EXPECT_EQ(recognition.txnId(), txnId);
        EXPECT_EQ(recognition.frameId(), frames[1].id());

        const common::ImageOrientation orientation(common::Rotation::CW_90);
        EXPECT_EQ(recognition.orientation(), orientation);

        EXPECT_EQ(recognition.type(), RecognitionType::DetectSign);
        EXPECT_EQ(recognition.source(), RecognitionSource::Toloka);
        EXPECT_EQ(recognition.version(), RECOGNITION_VERSION);

        EXPECT_EQ(recognition.hasValue(), false);
    }
}

TEST_F(FrameFixture, update_recognition)
{
    Recognition recognition {
        frames[0].id(),
        common::ImageOrientation(common::Rotation::CCW_90),
        RecognitionType::DetectSign,
        RecognitionSource::Toloka,
        RECOGNITION_VERSION
    };

    {   // Insert
        auto txn = txnHandle();
        RecognitionGateway(*txn).insertx(recognition);
        txn->commit();
    }

    const DetectedSigns value {
        DetectedSign {
            common::ImageBox(100, 100, 200, 200),
            traffic_signs::TrafficSign::ProhibitoryNoParking,
            0.98,
            true,
            0.99
        }
    };

    auto txn = txnHandle();
    recognition.setValue(value);

    const TId txnId = RecognitionGateway(*txn).updatex(recognition);
    txn->commit();

    {    // Check value
        const auto updated = RecognitionGateway(*txnHandle()).loadById(recognition.id());

        EXPECT_EQ(updated.txnId(), txnId);
        EXPECT_EQ(toJson(updated.value<DetectedSigns>()), toJson(value));
    }
}

TEST_F(FrameFixture, unique_recognition)
{
    Recognitions recognitions {
        {
            frames[0].id(),
            common::ImageOrientation(common::Rotation::CW_0),
            RecognitionType::DetectSign,
            RecognitionSource::Toloka,
            RECOGNITION_VERSION
        },
        {
            frames[0].id(),
            common::ImageOrientation(common::Rotation::CW_0),
            RecognitionType::DetectSign,
            RecognitionSource::Toloka,
            RECOGNITION_VERSION
        }
    };

    EXPECT_THROW(
        RecognitionGateway(*txnHandle()).insertx(recognitions),
        maps::sql_chemistry::UniqueViolationError
    );
}

TEST_F(FrameFixture, empty_recognition)
{
    Recognition recognition {
        frames[0].id(),
        common::ImageOrientation(common::Rotation::CW_0),
        RecognitionType::DetectSign,
        RecognitionSource::Toloka,
        RECOGNITION_VERSION
    };
    EXPECT_EQ(recognition.empty(), false);

    const DetectedSigns value {
        DetectedSign {
            common::ImageBox(100, 100, 200, 200),
            traffic_signs::TrafficSign::ProhibitoryNoParking,
            0.98,
            true,
            0.99
        }
    };
    recognition.setValue(value);
    EXPECT_EQ(recognition.empty(), false);

    recognition.setValue(DetectedSigns{});
    EXPECT_EQ(recognition.empty(), true);
}

TEST_F(FrameFixture, insert_detection_group)
{
    DetectionGroups groups {
        {frames[0].id(), DetectionType::Sign},
        {frames[0].id(), DetectionType::TrafficLight},
        {frames[1].id(), DetectionType::HouseNumber},
    };

    auto txn = txnHandle();
    const TId txnId = DetectionGroupGateway(*txn).insertx(groups);
    txn->commit();

    {   // Check first
        const auto group = DetectionGroupGateway(*txnHandle()).loadById(groups[0].id());

        EXPECT_EQ(group.txnId(), txnId);
        EXPECT_EQ(group.frameId(), frames[0].id());
        EXPECT_EQ(group.type(), DetectionType::Sign);
        EXPECT_EQ(group.approved(), false);
    }

    {   // Check second
        const auto group = DetectionGroupGateway(*txnHandle()).loadById(groups[1].id());

        EXPECT_EQ(group.txnId(), txnId);
        EXPECT_EQ(group.frameId(), frames[0].id());
        EXPECT_EQ(group.type(), DetectionType::TrafficLight);
        EXPECT_EQ(group.approved(), false);

    }

    {   // Check third
        const auto group = DetectionGroupGateway(*txnHandle()).loadById(groups[2].id());

        EXPECT_EQ(group.txnId(), txnId);
        EXPECT_EQ(group.frameId(), frames[1].id());
        EXPECT_EQ(group.type(), DetectionType::HouseNumber);
        EXPECT_EQ(group.approved(), false);
    }
}

TEST_F(FrameFixture, update_detection_group)
{
    DetectionGroup group {
        frames[0].id(),
        DetectionType::Sign,
    };

    {   // Insert
        auto txn = txnHandle();
        DetectionGroupGateway(*txn).insertx(group);
        txn->commit();
    }

    auto txn = txnHandle();
    group.setApproved(true);

    const TId txnId = DetectionGroupGateway(*txn).updatex(group);
    txn->commit();

    {    // Check flag
        const auto updated = DetectionGroupGateway(*txnHandle()).loadById(group.id());

        EXPECT_EQ(updated.txnId(), txnId);
        EXPECT_EQ(updated.approved(), true);
    }
}

TEST_F(FrameFixture, unique_detection_group)
{
    DetectionGroups groups {
        {frames[0].id(), DetectionType::Sign},
        {frames[0].id(), DetectionType::Sign},
    };

    EXPECT_THROW(
        DetectionGroupGateway(*txnHandle()).insertx(groups),
        maps::sql_chemistry::UniqueViolationError
    );
}

struct DetectionGroupFixture: public FrameFixture{
    DetectionGroupFixture();
    DetectionGroups detectionGroups;
};

DetectionGroupFixture::DetectionGroupFixture()
{
    detectionGroups = {
        {frames[0].id(), DetectionType::Sign},
        {frames[0].id(), DetectionType::HouseNumber},
        {frames[1].id(), DetectionType::TrafficLight},
        {frames[1].id(), DetectionType::RoadMarking},
    };

    auto txn = txnHandle();
    DetectionGroupGateway(*txn).insertx(detectionGroups);
    txn->commit();
}

TEST_F(DetectionGroupFixture, insert_detection)
{
    const DetectedSign first {
        common::ImageBox(100, 100, 200, 200),
        traffic_signs::TrafficSign::ProhibitoryNoParking,
        0.98,
        false,
        0.87
    };

    const DetectedSign second {
        common::ImageBox(300, 100, 400, 200),
        traffic_signs::TrafficSign::ProhibitoryNoEntry,
        0.97,
        true,
        0.78
    };

    const DetectedHouseNumber third {
        common::ImageBox(100, 100, 200, 200),
        0.87,
        "21"
    };

    Detections detections {
        {detectionGroups[0].id(), first},
        {detectionGroups[0].id(), second},
        {detectionGroups[1].id(), third},
    };

    auto txn = txnHandle();
    const TId txnId = DetectionGateway(*txn).insertx(detections);
    txn->commit();

    {    // Check first
        const auto detection = DetectionGateway(*txnHandle()).loadById(detections[0].id());

        EXPECT_EQ(detection.txnId(), txnId);
        EXPECT_EQ(detection.groupId(), detectionGroups[0].id());
        EXPECT_EQ(detection.deleted(), false);

        EXPECT_EQ(toJson(detection.attrs<DetectedSign>()), toJson(first));
    }

    {    // Check second
        const auto detection = DetectionGateway(*txnHandle()).loadById(detections[1].id());

        EXPECT_EQ(detection.txnId(), txnId);
        EXPECT_EQ(detection.groupId(), detectionGroups[0].id());
        EXPECT_EQ(detection.deleted(), false);

        EXPECT_EQ(toJson(detection.attrs<DetectedSign>()), toJson(second));
    }

    {    // Check third
        const auto detection = DetectionGateway(*txnHandle()).loadById(detections[2].id());

        EXPECT_EQ(detection.txnId(), txnId);
        EXPECT_EQ(detection.groupId(), detectionGroups[1].id());
        EXPECT_EQ(detection.deleted(), false);

        EXPECT_EQ(toJson(detection.attrs<DetectedHouseNumber>()), toJson(third));
    }
}

TEST_F(DetectionGroupFixture, update_detection)
{
    Detection detection {
        detectionGroups[0].id(),
        DetectedSign {
            common::ImageBox(100, 100, 200, 200),
            traffic_signs::TrafficSign::ProhibitoryNoParking,
            0.98,
            false,
            0.87
        }
    };

    {   // Insert
        auto txn = txnHandle();
        DetectionGateway(*txn).insertx(detection);
        txn->commit();
    }

    auto txn = txnHandle();
    detection.setDeleted();

    const TId txnId = DetectionGateway(*txn).updatex(detection);
    txn->commit();

    {    // Check flag
        const auto updated = DetectionGateway(*txnHandle()).loadById(detection.id());

        EXPECT_EQ(updated.txnId(), txnId);
        EXPECT_EQ(updated.deleted(), true);
    }
}

TEST_F(RecognitionTolokaFixture, insert_recognition_tasks)
{
    RecognitionTasks recognitionTasks{
        {recognitions[0].id(), tasks[0].id()},
        {recognitions[1].id(), tasks[1].id()}
    };

    {
        auto txn = txnHandle();
        RecognitionTaskGateway(*txn).insertx(recognitionTasks);
        txn->commit();
    }

    {
        auto txn = txnHandle();
        auto result = RecognitionTaskGateway(*txn).load(
            table::RecognitionTask::recognitionId == recognitions[0].id()
        );
        EXPECT_EQ(result.size(), 1u);
    }
}

TEST_F(RecognitionTolokaFixture, unique_recognition_tasks)
{
    RecognitionTasks recognitionTasks{
        {recognitions[0].id(), tasks[0].id()},
        {recognitions[1].id(), tasks[0].id()}
    };

    {
        auto txn = txnHandle();
        EXPECT_THROW(RecognitionTaskGateway(*txn).insertx(recognitionTasks), maps::Exception);
    }
}

TEST_F(RecognitionTolokaFixture, set_parent_recognition_task_id)
{
    RecognitionTask parentTask(recognitions[0].id(), tasks[0].id());

    {
        auto txn = txnHandle();
        RecognitionTaskGateway(*txn).insertx(parentTask);
        txn->commit();
    }

    RecognitionTask task(recognitions[0].id(), tasks[1].id(), parentTask.id());

    {
        auto txn = txnHandle();
        RecognitionTaskGateway(*txn).insertx(task);
        txn->commit();
    }

    {
        auto txn = txnHandle();
        auto result = RecognitionTaskGateway(*txn).load(
            table::RecognitionTask::parentId == parentTask.id()
        );
        EXPECT_EQ(result.size(), 1u);
    }
}

TEST_F(RecognitionTolokaFixture, set_not_exists_parent_recognition_task_id)
{
    RecognitionTask task(recognitions[0].id(), tasks[0].id(), 42);

    {
        auto txn = txnHandle();
        EXPECT_THROW(RecognitionTaskGateway(*txn).insertx(task), pqxx::sql_error);
    }
}

TEST_F(RecognitionTolokaFixture, insert_recognition_task_with_wrong_parent)
{
    RecognitionTask parentTask(recognitions[0].id(), tasks[0].id());

    {
        auto txn = txnHandle();
        RecognitionTaskGateway(*txn).insertx(parentTask);
        txn->commit();
    }

    RecognitionTask task(recognitions[1].id(), tasks[1].id(), parentTask.id());

    {
        auto txn = txnHandle();
        EXPECT_THROW(RecognitionTaskGateway(*txn).insertx(task), pqxx::sql_error);
    }
}

TEST_F(DetectionGroupFixture, insert_detection_relation)
{
    const DetectedSign sign {
        common::ImageBox(100, 100, 200, 200),
        traffic_signs::TrafficSign::ProhibitoryNoParking,
        1.0,
        false,
        1.0
    };

    const DetectedSign table {
        common::ImageBox(100, 250, 200, 400),
        traffic_signs::TrafficSign::InformationStartZone,
        1.0,
        false,
        1.0
    };

    Detections detections {
        {detectionGroups[0].id(), sign},
        {detectionGroups[0].id(), table},
    };

    auto txnDetection = txnHandle();
    DetectionGateway(*txnDetection).insertx(detections);
    txnDetection->commit();

    DetectionRelations relations {
        {detections[0].id(), detections[1].id()}
    };
    auto txnRelation = txnHandle();
    const TId txnId = DetectionRelationGateway(*txnRelation).insertx(relations);
    txnRelation->commit();

    EXPECT_EQ(relations[0].txnId(), txnId);
    EXPECT_EQ(relations[0].deleted(), false);
    EXPECT_EQ(relations[0].masterDetectionId(), detections[0].id());
    EXPECT_EQ(relations[0].slaveDetectionId(), detections[1].id());
}

} // namespace maps::mrc::db::eye::tests
