#include "construct.h"
#include "fixture.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/local_postgres/include/instance.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/hypothesis.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/hypothesis_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/sign.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/sign_gateway.h>
#include <maps/libs/geolib/include/units_literals.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

namespace maps::mrc::db::tests {

using namespace ::testing;
using namespace geolib3::literals;

Hypothesis makeTestHypothesis(pqxx::transaction_base& txn) {
    auto testSign = newSign()
        .setType(Sign::Type::ProhibitoryMaxSpeed60)
        .setMercatorPos({37, 55}, 0.0)
        .setHeading(geolib3::Heading(120), 0_deg);

    SignGateway(txn).upsert(testSign);

    Hypothesis testHypo(
        HypothesisType::WrongSpeedLimit,
        {37.1, 55.1},
        100,
        500,
        testSign.id()
    );

    HypothesisGateway(txn).upsert(testHypo);
    return testHypo;
};

TEST_F(Fixture, test_creating_hypothesis) {
    auto txn = txnHandle();
    auto testHypo = makeTestHypothesis(*txn);
    EXPECT_EQ(testHypo.id(), 1u);

    Hypotheses testHypes{
        Hypothesis(
            testHypo.type(),
            testHypo.mercatorPos(),
            testHypo.objectId(),
            testHypo.commitId(),
            testHypo.signId()
        ),
        Hypothesis(
            testHypo.type(),
            testHypo.mercatorPos(),
            testHypo.objectId(),
            testHypo.commitId(),
            testHypo.signId()
        ),
    };
    HypothesisGateway(* txn).upsert(testHypes);
    EXPECT_EQ(testHypes.front().id(), 2u);
    EXPECT_EQ(testHypes.back().id(), 3u);
}

TEST_F(Fixture, test_updating_hypothesis) {
    {
        auto txn = txnHandle();
        makeTestHypothesis(*txn);
        txn->commit();
    }

    {
        auto txn = txnHandle();
        auto testHypes = HypothesisGateway(*txn).load(
            table::Hypothesis::type.equals(HypothesisType::WrongSpeedLimit)
        );
        json::Value context = json::Value::fromString(R"(
{
    "foo": "bar",
    "ololo": 100500
}
)");
        testHypes.front().setContext(context);
        HypothesisGateway(* txn).upsert(testHypes.front());
        txn->commit();
    }

    {
        auto txn = txnHandle();
        auto testHypes = HypothesisGateway(*txn).load(
            table::Hypothesis::type.equals(HypothesisType::WrongSpeedLimit)
        );
        EXPECT_EQ(testHypes.front().context()["foo"].as<std::string>(), "bar");
        EXPECT_EQ(testHypes.front().context()["ololo"].as<int>(), 100500);
    }
}

TEST_F(Fixture, test_querying_hypothesis) {
    auto txn = txnHandle();
    makeTestHypothesis(*txn);

    {
        auto hypotheses = HypothesisGateway(*txn).load(
            table::Hypothesis::type.equals(HypothesisType::WrongSpeedLimit) &&
                table::Hypothesis::objectId.equals(100) &&
                table::Hypothesis::commitId.equals(500)
        );
        ASSERT_EQ(hypotheses.size(), 1u);
        EXPECT_EQ(hypotheses.front().mercatorPos().x(), 37.1);
        EXPECT_EQ(hypotheses.front().mercatorPos().y(), 55.1);
    }
    {
        auto hypotheses = HypothesisGateway(*txn).load(
            table::Hypothesis::type.equals(HypothesisType::WrongSpeedLimit) &&
                table::Hypothesis::position.intersects({
                    {36, 54},
                    {38, 56}
                })
        );
        ASSERT_EQ(hypotheses.size(), 1u);
        EXPECT_EQ(hypotheses.front().objectId(), 100u);
        EXPECT_EQ(hypotheses.front().commitId(), 500u);
    }
}

} // namespace maps::mrc::db::tests
