#include "construct.h"
#include "fixture.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>
#include <maps/libs/introspection/include/comparison.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/hypothesis_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/queued_photo_id_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ride_recording_report_gateway.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

namespace maps::mrc::db::rides {
using introspection::operator==;
} // namespace maps::mrc::db::rides

namespace maps::mrc::db::tests {
using namespace ::testing;

namespace {

using RidePhotoTuple = std::tuple<std::string,
    geolib3::Point2,
    geolib3::Heading,
    std::string,
    std::string,
    std::string,
    Dataset,
    std::string>;

Feature toRidePhoto(const RidePhotoTuple& tuple) {
    return newFeature()
        .setSourceId(std::get<0>(tuple))
        .setGeodeticPos(std::get<1>(tuple))
        .setHeading(std::get<2>(tuple))
        .setTimestamp(std::get<3>(tuple))
        .setMdsKey({std::get<4>(tuple), std::get<5>(tuple)})
        .setDataset(std::get<6>(tuple))
        .setUserId(std::get<7>(tuple));
}

const std::string USER_ID = "Serlock Holmes";
const auto START_TIME = chrono::parseIsoDateTime("2020-10-20 15:14:00");
const auto END_TIME = chrono::parseIsoDateTime("2020-10-20 18:11:00");
const std::string SOURCE_ID = "iPhone";
const geolib3::Polyline2 TRACK{
    geolib3::PointsVector{{-0.158481, 51.523768}, {-0.075354, 51.505545}}};
const double DISTANCE_IN_METERS = 54321;
const size_t PHOTOS = 12345;
const auto UPLOAD_START_TIME = chrono::parseIsoDateTime("2020-10-21 16:00:00");
const auto UPLOAD_END_TIME = chrono::parseIsoDateTime("2020-10-21 16:10:00");
const bool IS_DELETED = false;
const bool SHOW_AUTHORSHIP = false;
const auto RIDE_STATUS = RideStatus::Processed;
const auto CLIENT_RIDE_ID = std::string("b0c2d8c8-6fc6-45d0-9e8e-45e37bd29636");
const size_t PUBLISHED_PHOTOS = PHOTOS / 2;

} // namespace

TEST_F(Fixture, test_rides_test_queued_photo_id) {
    const std::vector<RidePhotoTuple> DATA
        = {RidePhotoTuple{"src1", geolib3::Point2{44.01, 56.30}, geolib3::Heading(10),
            "2016-09-01 05:10:00+03", "4510", "photo1.jpg",
            Dataset::Agents, "Helg"},
            RidePhotoTuple{"src2", geolib3::Point2{44.01, 56.31}, geolib3::Heading(20),
                "2016-09-01 05:20:00+03", "4510", "photo2.jpg",
                Dataset::Agents, "Ingvar"}};
    Features ridePhotos;
    rides::QueuedPhotoIds queuedPhotoIds;

    {
        auto txn = txnHandle();
        for (const auto& tuple : DATA) {
            ridePhotos.push_back(toRidePhoto(tuple));
        }
        FeatureGateway{*txn}.insert(ridePhotos);
        auto receivedAt
            = chrono::parseSqlDateTime("2016-09-02 10:10:00+03");
        for (const auto& ridePhoto : ridePhotos) {
            queuedPhotoIds.emplace_back(ridePhoto.id(),
                receivedAt);
        }
        QueuedPhotoIdGateway{*txn}.insert(queuedPhotoIds);
        txn->commit();
    }

    {
        auto txn = txnHandle();
        EXPECT_EQ(QueuedPhotoIdGateway{*txn}.load(), queuedPhotoIds);
    }

    {
        auto txn = txnHandle();
        QueuedPhotoIdGateway{*txn}.remove(
            table::QueuedPhotoId::photoId.equals(queuedPhotoIds.back().photoId()));
        queuedPhotoIds.pop_back();
        txn->commit();
    }

    {
        auto txn = txnHandle();
        EXPECT_EQ(QueuedPhotoIdGateway{*txn}.load(), queuedPhotoIds);
    }
}

TEST_F(Fixture, test_rides_test_ride_recording_report) {
    const std::string SOURCE_ID_1{"SOURCE_ID_1"};
    const std::string SOURCE_ID_2{"SOURCE_ID_2"};
    const maps::mds::Key MDS_KEY_1{"GROUP_ID_1", "PATH_1"};
    const maps::mds::Key MDS_KEY_2{"GROUP_ID_2", "PATH_2"};
    const chrono::TimePoint STARTED_AT_1 = chrono::TimePoint::clock::now();
    const chrono::TimePoint FINISHED_AT_1 = STARTED_AT_1 + std::chrono::seconds(1);
    const chrono::TimePoint STARTED_AT_2 = FINISHED_AT_1;
    const chrono::TimePoint FINISHED_AT_2 = STARTED_AT_2 + std::chrono::seconds(1);

    rides::RideRecordingReport report1{SOURCE_ID_1, STARTED_AT_1, FINISHED_AT_1};
    report1.setMdsGroupId(MDS_KEY_1.groupId).setMdsPath(MDS_KEY_1.path);
    rides::RideRecordingReport report2{SOURCE_ID_2, STARTED_AT_2, FINISHED_AT_2};
    report2.setMdsGroupId(MDS_KEY_2.groupId).setMdsPath(MDS_KEY_2.path);

    // Save reports
    {
        auto txn = txnHandle();

        rides::RideRecordingReportGateway gtw{*txn};
        gtw.insert(report1);
        gtw.insert(report2);
        txn->commit();
    }

    // Check loading reports by filter
    {
        auto txn = txnHandle();
        rides::RideRecordingReportGateway gtw(*txn);

        auto ids = gtw.loadIds();
        std::sort(ids.begin(), ids.end());
        TIds expected = {report1.id(), report2.id()};
        std::sort(expected.begin(), expected.end());
        EXPECT_EQ(ids, expected);
        EXPECT_TRUE(report1 == gtw.loadById(report1.id()));
        EXPECT_TRUE(report2 == gtw.loadById(report2.id()));

        auto reports = gtw.load(
            rides::table::RideRecordingReport::sourceId.equals(SOURCE_ID_1));
        EXPECT_EQ(reports.size(), 1u);
        EXPECT_TRUE(report1 == reports.front());

        reports = gtw.load(
            rides::table::RideRecordingReport::startedAt.greater(STARTED_AT_1));
        EXPECT_EQ(reports.size(), 1u);
        EXPECT_TRUE(report2 == reports.front());
    }

    // Update MDS key
    {
        report1.setMdsGroupId(MDS_KEY_2.groupId).setMdsPath(MDS_KEY_2.path);
        auto txn = txnHandle();
        rides::RideRecordingReportGateway gtw(*txn);
        gtw.update(report1);
        txn->commit();
    }

    {
        auto txn = txnHandle();
        rides::RideRecordingReportGateway gtw(*txn);

        auto report = gtw.loadOne(
            rides::table::RideRecordingReport::id.equals(report1.id()));
        EXPECT_TRUE(report1 == report);
    }
}

TEST_F(Fixture, test_rides_test_ride)
{
    TId rideId{};
    {
        auto txn = txnHandle();
        auto ride = Ride(USER_ID,
                         START_TIME,
                         END_TIME,
                         SOURCE_ID,
                         TRACK,
                         DISTANCE_IN_METERS,
                         PHOTOS,
                         UPLOAD_START_TIME,
                         UPLOAD_END_TIME,
                         IS_DELETED);
        ride.setShowAuthorship(SHOW_AUTHORSHIP)
            .setStatus(RIDE_STATUS)
            .setClientId(CLIENT_RIDE_ID)
            .setPublishedPhotos(PUBLISHED_PHOTOS)
            ;
        RideGateway{*txn}.insertx(ride);
        txn->commit();
        rideId = ride.rideId();
    }

    {
        auto txn = txnHandle();
        auto ride = RideGateway{*txn}.tryLoadById(rideId);
        EXPECT_TRUE(ride);
        EXPECT_EQ(ride->userId(), USER_ID);
        EXPECT_TRUE(ride->startTime() == START_TIME);
        EXPECT_TRUE(ride->endTime() == END_TIME);
        EXPECT_EQ(ride->sourceId(), SOURCE_ID);
        EXPECT_TRUE(geolib3::test_tools::approximateEqual(
            ride->geodeticTrack(), TRACK, geolib3::EPS));
        EXPECT_EQ(ride->distanceInMeters(), DISTANCE_IN_METERS);
        EXPECT_EQ(ride->photos(), PHOTOS);
        EXPECT_TRUE(ride->uploadStartTime() == UPLOAD_START_TIME);
        EXPECT_TRUE(ride->uploadEndTime() == UPLOAD_END_TIME);
        EXPECT_TRUE(ride->isDeleted() == IS_DELETED);
        EXPECT_TRUE(ride->showAuthorship() == SHOW_AUTHORSHIP);
        EXPECT_TRUE(ride->status() == RIDE_STATUS);
        EXPECT_TRUE(ride->clientId() == CLIENT_RIDE_ID);
        EXPECT_TRUE(ride->publishedPhotos() == PUBLISHED_PHOTOS);

        auto rideLiteView = RideLiteViewGateway{*txn}.tryLoadById(rideId);
        EXPECT_TRUE(rideLiteView);
        EXPECT_EQ(rideLiteView->userId(), USER_ID);
        EXPECT_TRUE(rideLiteView->startTime() == START_TIME);
        EXPECT_TRUE(rideLiteView->endTime() == END_TIME);
        EXPECT_EQ(rideLiteView->distanceInMeters(), DISTANCE_IN_METERS);
        EXPECT_EQ(rideLiteView->photos(), PHOTOS);
        EXPECT_EQ(rideLiteView->publishedPhotos(), PUBLISHED_PHOTOS);
        EXPECT_EQ(rideLiteView->isDeleted(), IS_DELETED);
    }

    {
        auto txn = txnHandle();
        RideGateway{*txn}.remove(table::Ride::userId == USER_ID and
                                 table::Ride::startTime <= END_TIME and
                                 table::Ride::endTime >= START_TIME);
        txn->commit();
    }

    {
        auto txn = txnHandle();
        EXPECT_FALSE(RideGateway{*txn}.tryLoadById(rideId));
    }
}

TEST_F(Fixture, test_rides_test_overlapping_rides)
{
    using namespace std::literals::chrono_literals;

    auto rideTemplate = Ride(USER_ID,
                             START_TIME,
                             END_TIME,
                             SOURCE_ID,
                             TRACK,
                             DISTANCE_IN_METERS,
                             PHOTOS,
                             UPLOAD_START_TIME,
                             UPLOAD_END_TIME,
                             IS_DELETED);

    {
        auto txn = txnHandle();
        auto ride = rideTemplate;
        RideGateway{*txn}.insertx(ride);
        txn->commit();
    }
    {
        auto txn = txnHandle();
        auto ride = rideTemplate;
        EXPECT_THROW(RideGateway{*txn}.insertx(ride),
                     pqxx::integrity_constraint_violation);
    }
    {
        auto txn = txnHandle();
        auto ride = rideTemplate;
        ride.setClientId(CLIENT_RIDE_ID);
        RideGateway{*txn}.insertx(ride);
        txn->commit();
    }
    {
        auto txn = txnHandle();
        auto ride = rideTemplate;
        ride.setEndTime(rideTemplate.endTime() + 2h);
        ride.setStartTime(rideTemplate.endTime() + 1h);
        RideGateway{*txn}.insertx(ride);
        txn->commit();
    }
}

TEST_F(Fixture, test_rides_test_negative_interval_ride)
{
    EXPECT_THROW(
        [] {
            return Ride(USER_ID,
                        END_TIME,
                        START_TIME,
                        SOURCE_ID,
                        TRACK,
                        DISTANCE_IN_METERS,
                        PHOTOS,
                        UPLOAD_START_TIME,
                        UPLOAD_END_TIME,
                        IS_DELETED);
        }(),
        maps::Exception);
}

TEST_F(Fixture, test_rides_test_default_ride)
{
    auto txn = txnHandle();
    auto ride = sql_chemistry::GatewayAccess<Ride>::construct();
    RideGateway{*txn}.insertx(ride);
    txn->commit();
}

TEST_F(Fixture, test_rides_test_deleted_interval)
{
    TId id{};
    {
        auto txn = txnHandle();
        auto deletedInterval = DeletedInterval(
            USER_ID, SOURCE_ID, CLIENT_RIDE_ID, START_TIME, END_TIME);
        DeletedIntervalGateway{*txn}.insert(deletedInterval);
        txn->commit();
        id = deletedInterval.id();
    }

    {
        auto txn = txnHandle();
        auto deletedInterval =
            DeletedIntervalGateway{*txn}.tryLoadById(id);
        EXPECT_TRUE(deletedInterval);
        EXPECT_EQ(deletedInterval->userId(), USER_ID);
        EXPECT_EQ(deletedInterval->sourceId(), SOURCE_ID);
        EXPECT_EQ(deletedInterval->clientRideId(), CLIENT_RIDE_ID);
        EXPECT_TRUE(deletedInterval->startedAt() == START_TIME);
        EXPECT_TRUE(deletedInterval->endedAt() == END_TIME);
    }
}

TEST_F(Fixture, test_rides_test_negative_deleted_interval)
{
    EXPECT_THROW(
        [] {
            return DeletedInterval(
                USER_ID, SOURCE_ID, CLIENT_RIDE_ID, END_TIME, START_TIME);
        }(),
        maps::Exception);
}

TEST_F(Fixture, test_rides_test_default_deleted_interval)
{
    auto txn = txnHandle();
    auto deletedInterval =
        sql_chemistry::GatewayAccess<DeletedInterval>::construct();
    DeletedIntervalGateway{*txn}.insert(deletedInterval);
    txn->commit();
}

TEST_F(Fixture, test_rides_test_ride_hypothesis)
{
    Ride ride = [&] {
        auto ride = Ride(USER_ID,
                         START_TIME,
                         END_TIME,
                         SOURCE_ID,
                         TRACK,
                         DISTANCE_IN_METERS,
                         PHOTOS,
                         UPLOAD_START_TIME,
                         UPLOAD_END_TIME,
                         IS_DELETED);
        ride.setShowAuthorship(SHOW_AUTHORSHIP)
            .setStatus(RIDE_STATUS)
            .setClientId(CLIENT_RIDE_ID);
        auto txn = txnHandle();
        RideGateway{*txn}.insertx(ride);
        txn->commit();
        return ride;
    }();

    eye::Hypothesis hypothesis = [&] {
        auto absentHouseNumberAttrs = eye::AbsentHouseNumberAttrs{"11"};
        auto hypothesis =
            eye::Hypothesis{geolib3::Point2{100, 200}, absentHouseNumberAttrs};
        auto txn = txnHandle();
        eye::HypothesisGateway(*txn).insertx(hypothesis);
        txn->commit();
        return hypothesis;
    }();

    RideHypothesis rideHypothesis = [&] {
        auto rideHypothesis = RideHypothesis{ride.rideId(), hypothesis.id()};
        auto txn = txnHandle();
        RideHypothesisGateway{*txn}.insert(rideHypothesis);
        txn->commit();
        return rideHypothesis;
    }();

    EXPECT_EQ(rideHypothesis.rideId(), ride.rideId());
    EXPECT_EQ(rideHypothesis.hypothesisId(), hypothesis.id());
    EXPECT_EQ(RideHypothesisGateway{*txnHandle()}.load(),
              RideHypotheses{rideHypothesis});
}

} // namespace maps::mrc::db::tests
