#include "construct.h"
#include "fixture.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/panorama_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/sign_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/database_fixture.h>
#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/log_disabling_fixture.h>
#include <maps/libs/geolib/include/units_literals.h>

namespace maps::mrc::db::tests {
using namespace ::testing;
using namespace geolib3::literals;

TEST_F(Fixture, sign_tests_test_signs) {
    Signs speeds{
        newSign()
            .setType(Sign::Type::ProhibitoryMaxSpeed30)
            .setMercatorPos(geolib3::Point2(10, 10), 0.1)
            .setHeading(geolib3::Heading(0.0), 1.0_deg),
        newSign()
            .setType(Sign::Type::ProhibitoryMaxSpeed40)
            .setMercatorPos(geolib3::Point2(10, 10), 0.1)
            .setHeading(geolib3::Heading(0.0), 1.0_deg)
    };

    Sign& maxSpeed30 = speeds[0];
    Sign& maxSpeed40 = speeds[1];

    auto parking = newSign()
        .setType(Sign::Type::InformationParking)
        .setMercatorPos(geolib3::Point2(0.0, 0.0), 0.1)
        .setHeading(geolib3::Heading(0.0), 1.0_deg);

    { // create signs
        auto txn = txnHandle();

        SignGateway gateway(*txn);
        gateway.upsert(speeds);
        gateway.upsert(parking);

        EXPECT_TRUE(maxSpeed30.id());
        EXPECT_TRUE(maxSpeed40.id());
        EXPECT_TRUE(parking.id());

        txn->commit();
    }

    { // update max speed 30
        maxSpeed30.setMercatorPos(geolib3::Point2(0, 0), 0.1);
        maxSpeed30.setHeading(geolib3::Heading(30.3), 1.0_deg);

        auto txn = txnHandle();
        SignGateway(* txn).upsert(maxSpeed30);

        txn->commit();
    }

    { // load signs by bbox
        auto txn = txnHandle();
        SignGateway gateway(*txn);

        const auto result = gateway.loadByBbox(
            geolib3::BoundingBox{
                geolib3::Point2(-1.0, -1.0),
                geolib3::Point2(1.0, 1.0),
            },
            Sign::Type::ProhibitoryMaxSpeed30);

        EXPECT_EQ(result.size(), 1u);

        const Sign& sign = result.front();
        EXPECT_EQ(sign.heading(), maxSpeed30.heading());
        EXPECT_EQ(sign.headingVariance(), maxSpeed30.headingVariance());
        EXPECT_EQ(
            geolib3::compare(sign.mercatorPos(), maxSpeed30.mercatorPos()), 0);
    }
}

TEST_F(Fixture, sign_tests_test_sign_features) {

    auto maxSpeed30 = newSign()
        .setType(Sign::Type::ProhibitoryMaxSpeed30)
        .setMercatorPos(geolib3::Point2(1, 1), 0.1)
        .setHeading(geolib3::Heading(0.0), 1.0_deg);

    auto maxSpeed40 = newSign()
        .setType(Sign::Type::ProhibitoryMaxSpeed40)
        .setMercatorPos(geolib3::Point2(2, 0), 0.1)
        .setHeading(geolib3::Heading(300.0), 1.0_deg);

    auto maxSpeed50 = newSign()
        .setType(Sign::Type::ProhibitoryMaxSpeed50)
        .setMercatorPos(geolib3::Point2(-1, -1), 0.1)
        .setHeading(geolib3::Heading(50), 1.0_deg);

    auto parking = newSign()
        .setType(Sign::Type::InformationParking)
        .setMercatorPos(geolib3::Point2(-1, -1), 0.1)
        .setHeading(geolib3::Heading(50), 1.0_deg);

    { // create signs
        auto txn = txnHandle();

        SignGateway gateway(*txn);
        gateway.upsert(maxSpeed30);
        gateway.upsert(maxSpeed40);
        gateway.upsert(maxSpeed50);
        gateway.upsert(parking);

        txn->commit();
    }

    const TId nonexistentId = 100500;

    { // create sign features with nonexistent feature id
        auto txn = txnHandle();

        SignFeatures signFeatures{
            SignFeature(nonexistentId, 10, 10, 20, 20).setSignId(parking.id()),
        };

        EXPECT_THROW(
            SignFeatureGateway(* txn).insert(signFeatures),
            pqxx::sql_error
        );
    }

    auto featureOne = newFeature()
        .setSourceId("src")
        .setGeodeticPos({44.01, 56.30})
        .setHeading(geolib3::Heading(10.0))
        .setTimestamp("2016-09-01 05:10:00+03")
        .setMdsKey({"4510", "photo1.jpg"});

    auto featureTwo = newFeature()
        .setSourceId("src")
        .setGeodeticPos({44.02, 56.32})
        .setHeading(geolib3::Heading(30.0))
        .setTimestamp("2016-09-01 05:30:00+03")
        .setMdsKey({"4510", "photo2.jpg"});

    { // create features
        auto txn = txnHandle();

        FeatureGateway gateway(*txn);
        gateway.insert(featureOne);
        gateway.insert(featureTwo);

        txn->commit();
    }

    SignFeatures featuresOfParking{
        SignFeature(featureOne.id(), 10, 10, 20, 20).setSignId(parking.id()),
        SignFeature(featureTwo.id(), 20, 20, 40, 40).setSignId(parking.id()),
    };

    SignFeatures featuresOfMaxSpeed30{
        SignFeature(featureOne.id(), 10, 10, 20, 20).setSignId(maxSpeed30.id())
    };

    SignFeatures featuresOfMaxSpeed50{
        SignFeature(featureOne.id(), 40, 40, 60, 60).setSignId(maxSpeed50.id()),
    };

    { // create sign features
        auto txn = txnHandle();
        SignFeatureGateway(* txn).insert(featuresOfParking);
        SignFeatureGateway(* txn).insert(featuresOfMaxSpeed30);
        SignFeatureGateway(* txn).insert(featuresOfMaxSpeed50);
        txn->commit();
    }

    { // create sign features with nonexistent sign id
        auto txn = txnHandle();

        SignFeatures signFeatures{
            SignFeature(featureOne.id(), 10, 10, 20, 20).setSignId(nonexistentId),
        };

        EXPECT_THROW(
            SignFeatureGateway(* txn).insert(signFeatures),
            pqxx::sql_error
        );
    }

    { // load sign features for max speed 30
        auto txn = txnHandle();

        auto features = SignFeatureGateway(*txn).loadBySignId(maxSpeed30.id());
        EXPECT_EQ(features.size(), 1u);
        EXPECT_EQ(features.front().featureId(), featureOne.id());
        EXPECT_EQ(features.front().signId(), maxSpeed30.id());

        EXPECT_EQ(SignFeatureGateway(* txn).loadBySignId(maxSpeed40.id()).size(), 0u);
    }

    { // load sign feature ids by filter
        auto txn = txnHandle();

        const TIds featureIds = SignFeatureGateway(*txn).loadFeatureIds(
            table::SignFeature::signId.in({maxSpeed30.id(), maxSpeed50.id()})
        );

        EXPECT_EQ(featureIds.size(), 1u);
        EXPECT_EQ(featureIds.front(), featureOne.id());
    }
}

TEST_F(Fixture, sign_tests_test_sign_panoramas) {

    auto maxSpeed30 = newSign()
        .setType(Sign::Type::ProhibitoryMaxSpeed30)
        .setMercatorPos(geolib3::Point2(1, 1), 0.1)
        .setHeading(geolib3::Heading(0.0), 1.0_deg);

    auto maxSpeed40 = newSign()
        .setType(Sign::Type::ProhibitoryMaxSpeed40)
        .setMercatorPos(geolib3::Point2(2, 0), 0.1)
        .setHeading(geolib3::Heading(300.0), 1.0_deg);

    auto maxSpeed50 = newSign()
        .setType(Sign::Type::ProhibitoryMaxSpeed50)
        .setMercatorPos(geolib3::Point2(-1, -1), 0.1)
        .setHeading(geolib3::Heading(50), 1.0_deg);

    auto parking = newSign()
        .setType(Sign::Type::InformationParking)
        .setMercatorPos(geolib3::Point2(-1, -1), 0.1)
        .setHeading(geolib3::Heading(50), 1.0_deg);

    { // create signs
        auto txn = txnHandle();

        SignGateway gateway(*txn);
        gateway.upsert(maxSpeed30);
        gateway.upsert(maxSpeed40);
        gateway.upsert(maxSpeed50);
        gateway.upsert(parking);

        txn->commit();
    }

    const TId nonexistentId = 100500;

    { // create sign panoramas with nonexistent panorama id
        auto txn = txnHandle();

        SignPanoramas signPanoramas{
            SignPanorama(nonexistentId, 10, 10, 20, 20).setSignId(parking.id()),
        };

        EXPECT_THROW(
            SignPanoramaGateway(* txn).insert(signPanoramas),
            pqxx::sql_error
        );
    }

    using namespace std::literals;
    Panorama panoramaOne{"mds_key_1", "src_key_1", chrono::TimePoint{1s}, 1, 1,
                         geolib3::Point2{1.0, 1.0}, 1.0, 1.0, 1.0,
                         256, 256, 512, 512, 0};
    Panorama panoramaTwo{"mds_key_2", "src_key_2", chrono::TimePoint{2s}, 2, 2,
                         geolib3::Point2{2.0, 2.0}, 2.0, 2.0, 2.0,
                         256, 256, 512, 512, 0};

    { // create panoramas
        auto txn = txnHandle();

        PanoramaGateway gateway(*txn);
        gateway.insert(panoramaOne);
        gateway.insert(panoramaTwo);

        txn->commit();
    }

    SignPanoramas panoramasOfParking{
        SignPanorama(panoramaOne.panoramaId(), 10, 10, 20, 20).setSignId(parking.id()),
        SignPanorama(panoramaTwo.panoramaId(), 20, 20, 40, 40).setSignId(parking.id()),
    };

    SignPanoramas panoramasOfMaxSpeed30{
        SignPanorama(panoramaOne.panoramaId(), 10, 10, 20, 20).setSignId(maxSpeed30.id())
    };

    SignPanoramas panoramasOfMaxSpeed50{
        SignPanorama(panoramaOne.panoramaId(), 40, 40, 60, 60).setSignId(maxSpeed50.id()),
    };

    { // create sign panoramas
        auto txn = txnHandle();
        SignPanoramaGateway(* txn).insert(panoramasOfParking);
        SignPanoramaGateway(* txn).insert(panoramasOfMaxSpeed30);
        SignPanoramaGateway(* txn).insert(panoramasOfMaxSpeed50);
        txn->commit();
    }

    { // create sign panoramas with nonexistent sign id
        auto txn = txnHandle();

        SignPanoramas signPanoramas{
            SignPanorama(panoramaOne.panoramaId(), 10, 10, 20, 20)
                .setSignId(nonexistentId),
        };

        EXPECT_THROW(
            SignPanoramaGateway(* txn).insert(signPanoramas),
            pqxx::sql_error
        );
    }

    { // load sign panoramas for max speed 30
        auto txn = txnHandle();

        auto panoramas = SignPanoramaGateway(*txn).loadBySignId(maxSpeed30.id());
        EXPECT_EQ(panoramas.size(), 1u);
        EXPECT_EQ(panoramas.front().panoramaId(), panoramaOne.panoramaId());
        EXPECT_EQ(panoramas.front().signId(), maxSpeed30.id());

        EXPECT_EQ(SignPanoramaGateway(* txn).loadBySignId(maxSpeed40.id()).size(), 0u);
    }

    { // load sign panorama ids by filter
        auto txn = txnHandle();

        const TIds panoramaIds = SignPanoramaGateway(*txn).loadPanoramaIds(
            table::SignPanorama::signId.in({maxSpeed30.id(), maxSpeed50.id()})
        );

        EXPECT_EQ(panoramaIds.size(), 1u);
        EXPECT_EQ(panoramaIds.front(), panoramaOne.panoramaId());
    }

    db::TId removedPanoramaId = panoramasOfParking[0].panoramaId();
    {   // remove panorama
        auto txn = txnHandle();

        PanoramaGateway gateway(*txn);
        gateway.removeById(removedPanoramaId);
        panoramasOfParking.erase(panoramasOfParking.begin());

        txn->commit();
    }

    {   // check that panorama is deleted too
        auto txn = txnHandle();

        SignPanoramaGateway gateway(*txn);
        EXPECT_FALSE(gateway.exists(
            table::SignPanorama::panoramaId.equals(removedPanoramaId)));
    }
}

} // namespace maps::mrc::db::tests
