#include "construct.h"
#include "fixture.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/traffic_light_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/traffic_light_hypothesis_gateway.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/panorama_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/database_fixture.h>
#include <maps/wikimap/mapspro/services/mrc/libs/unittest/include/yandex/maps/mrc/unittest/log_disabling_fixture.h>
#include <maps/libs/geolib/include/units_literals.h>

namespace maps::mrc::db::tests {
using namespace ::testing;
using namespace geolib3::literals;

TEST_F(Fixture, test_traffic_light) {
    TrafficLights trafficLights{
        newTrafficLight()
            .setMercatorPos(geolib3::Point2(10, 10), 0.1)
            .setHeading(geolib3::Heading(0.0), 1.0_deg),
        newTrafficLight()
            .setMercatorPos(geolib3::Point2(10, 10), 0.1)
            .setHeading(geolib3::Heading(10.0), 2.0_deg)
    };

    TrafficLight& firstLight = trafficLights[0];
    TId firstLightId = 0;
    TrafficLight& secondLight = trafficLights[1];
    TId secondLightId = 0;

    { // create traffic lights
        auto txn = txnHandle();

        TrafficLightGateway gateway(*txn);
        gateway.upsert(trafficLights);

        EXPECT_TRUE(firstLight.id());
        firstLightId = firstLight.id();

        EXPECT_TRUE(secondLight.id());
        secondLightId = secondLight.id();

        txn->commit();
    }

    { // update first traffic light
        firstLight.setMercatorPos(geolib3::Point2(0, 0), 0.1);
        firstLight.setHeading(geolib3::Heading(30.3), 1.0_deg);

        auto txn = txnHandle();
        TrafficLightGateway(*txn).upsert(firstLight);

        EXPECT_EQ(firstLight.id(), firstLightId);

        txn->commit();
    }

    { // load traffic light by id
        auto txn = txnHandle();
        TrafficLightGateway gateway(*txn);

        const TrafficLights result = gateway.loadByIds({secondLightId});

        EXPECT_EQ(result.size(), 1u);

        const TrafficLight& light = result.front();
        EXPECT_EQ(light.heading(), secondLight.heading());
        EXPECT_EQ(light.headingVariance(), secondLight.headingVariance());
        EXPECT_EQ(
            geolib3::compare(light.mercatorPos(), secondLight.mercatorPos()), 0);

    }

    { // load traffic light by bbox
        auto txn = txnHandle();
        TrafficLightGateway gateway(*txn);

        const TrafficLights result = gateway.load(table::TrafficLight::position.intersects(
            geolib3::BoundingBox{
                geolib3::Point2(-1.0, -1.0),
                geolib3::Point2(1.0, 1.0),
            }
        ));

        EXPECT_EQ(result.size(), 1u);

        const TrafficLight& light = result.front();
        EXPECT_EQ(light.heading(), firstLight.heading());
        EXPECT_EQ(light.headingVariance(), firstLight.headingVariance());
        EXPECT_EQ(
            geolib3::compare(light.mercatorPos(), firstLight.mercatorPos()), 0);
    }
}

TEST_F(Fixture, test_traffic_light_features) {
    TrafficLight trafficLight = newTrafficLight()
        .setMercatorPos(geolib3::Point2(10, 10), 0.1)
        .setHeading(geolib3::Heading(0.0), 1.0_deg);

    { // create traffic lights
        auto txn = txnHandle();

        TrafficLightGateway gateway(*txn);
        gateway.upsert(trafficLight);

        txn->commit();
    }

    const TId nonexistentId = 100500;

    { // create sign features with nonexistent feature id
        auto txn = txnHandle();

        TrafficLightFeatures trafficLightFeatures{
            TrafficLightFeature(nonexistentId, 10, 10, 20, 20)
                .setTrafficLightId(trafficLight.id()),
        };

        EXPECT_THROW(
            TrafficLightFeatureGateway(*txn).insert(trafficLightFeatures),
            pqxx::sql_error
        );
    }

    Feature featureOne = newFeature()
        .setSourceId("src")
        .setGeodeticPos({44.02, 56.33})
        .setHeading(geolib3::Heading(10.0))
        .setTimestamp("2017-09-01 05:10:00+03")
        .setMdsKey({"4512", "photo1.jpg"})
        .setDataset(Dataset::Agents);

    Feature featureTwo = newFeature()
        .setSourceId("src")
        .setGeodeticPos({44.03, 56.34})
        .setHeading(geolib3::Heading(30.0))
        .setTimestamp("2017-09-01 05:30:00+03")
        .setMdsKey({"4512", "photo2.jpg"})
        .setDataset(Dataset::Agents);

    { // create features
        auto txn = txnHandle();

        FeatureGateway gateway(*txn);
        gateway.insert(featureOne);
        gateway.insert(featureTwo);

        txn->commit();
    }

    TrafficLightFeatures featuresOfTrafficLight{
        TrafficLightFeature(featureOne.id(), 10, 10, 20, 20)
            .setTrafficLightId(trafficLight.id()),
        TrafficLightFeature(featureTwo.id(), 20, 20, 40, 40)
            .setTrafficLightId(trafficLight.id()),
    };

    { // create traffic light features
        auto txn = txnHandle();
        TrafficLightFeatureGateway(*txn).insert(featuresOfTrafficLight);
        txn->commit();
    }

    { // create traffic light features with nonexistent traffic light id
        auto txn = txnHandle();

        TrafficLightFeatures trafficLightFeatures{
            TrafficLightFeature(featureOne.id(), 10, 10, 20, 20)
                .setTrafficLightId(nonexistentId),
        };

        EXPECT_THROW(
            TrafficLightFeatureGateway(* txn).insert(trafficLightFeatures),
            pqxx::sql_error
        );
    }

    { // load traffic light feature ids by filter
        auto txn = txnHandle();

        const TrafficLightFeatures features
            = TrafficLightFeatureGateway(*txn).loadByTrafficLightId(trafficLight.id());

        EXPECT_EQ(features.size(), 2u);
        EXPECT_EQ(
            std::set<TId>({features[0].featureId(), features[1].featureId()}),
            std::set<TId>({featureOne.id(), featureTwo.id()})
        );
    }
}

TEST_F(Fixture, test_traffic_light_panoramas) {
    TrafficLight trafficLight = newTrafficLight()
        .setMercatorPos(geolib3::Point2(10, 10), 0.1)
        .setHeading(geolib3::Heading(0.0), 1.0_deg);

    { // create traffic lights
        auto txn = txnHandle();

        TrafficLightGateway gateway(*txn);
        gateway.upsert(trafficLight);

        txn->commit();
    }

    const TId nonexistentId = 100500;

    { // create traffic light panoramas with nonexistent panorama id
        auto txn = txnHandle();

        TrafficLightPanoramas trafficLightPanoramas{
            TrafficLightPanorama(nonexistentId, 10, 10, 20, 20)
                .setTrafficLightId(trafficLight.id()),
        };

        EXPECT_THROW(
            TrafficLightPanoramaGateway(*txn).insert(trafficLightPanoramas),
            pqxx::sql_error
        );
    }

    using namespace std::literals;
    Panorama panoramaOne{"mds_key_3", "src_key_3", chrono::TimePoint{1s}, 3, 3,
                         geolib3::Point2{3.0, 3.0}, 3.0, 3.0, 3.0,
                         256, 256, 512, 512, 0};
    Panorama panoramaTwo{"mds_key_4", "src_key_4", chrono::TimePoint{2s}, 4, 4,
                         geolib3::Point2{4.0, 4.0}, 4.0, 4.0, 4.0,
                         256, 256, 512, 512, 0};

    { // create panoramas
        auto txn = txnHandle();

        PanoramaGateway gateway(*txn);
        gateway.insert(panoramaOne);
        gateway.insert(panoramaTwo);

        txn->commit();
    }

    TrafficLightPanoramas panoramasOfTrafficLight{
        TrafficLightPanorama(panoramaOne.panoramaId(), 10, 10, 20, 20)
            .setTrafficLightId(trafficLight.id()),
        TrafficLightPanorama(panoramaTwo.panoramaId(), 20, 20, 40, 40)
            .setTrafficLightId(trafficLight.id()),
    };

    { // create traffic light panoramas
        auto txn = txnHandle();
        TrafficLightPanoramaGateway(*txn).insert(panoramasOfTrafficLight);
        txn->commit();
    }

    { // create traffic light panorama with nonexistent traffic light id
        auto txn = txnHandle();

        TrafficLightPanoramas trafficLightPanoramas{
            TrafficLightPanorama(panoramaOne.panoramaId(), 10, 10, 20, 20)
                .setTrafficLightId(nonexistentId),
        };

        EXPECT_THROW(
            TrafficLightPanoramaGateway(* txn).insert(trafficLightPanoramas),
            pqxx::sql_error
        );
    }

    { // load traffic light panoramas
        auto txn = txnHandle();

        const TrafficLightPanoramas panoramas
            = TrafficLightPanoramaGateway(*txn).loadByTrafficLightId(trafficLight.id());

        EXPECT_EQ(panoramas.size(), 2u);
        EXPECT_EQ(
            std::set<TId>({panoramas[0].panoramaId(), panoramas[1].panoramaId()}),
            std::set<TId>({panoramaOne.panoramaId(), panoramaTwo.panoramaId()})
        );
    }

    {   // remove panorama one
        auto txn = txnHandle();

        PanoramaGateway gateway(*txn);
        gateway.removeById(panoramaOne.panoramaId());

        txn->commit();
    }

    {   // check that traffic light panorama is not deleted yet
        auto txn = txnHandle();

        TrafficLightPanoramaGateway gateway(*txn);
        EXPECT_FALSE(
            gateway.exists(table::TrafficLightPanorama::panoramaId.equals(panoramaOne.panoramaId()))
        );
        EXPECT_TRUE(
            gateway.exists(table::TrafficLightPanorama::trafficLightId.equals(trafficLight.id()))
        );
    }

    {   // remove feature two
        auto txn = txnHandle();

        PanoramaGateway gateway(*txn);
        gateway.removeById(panoramaTwo.panoramaId());

        txn->commit();
    }

    {   // check that traffic light panorama is deleted too
        auto txn = txnHandle();

        TrafficLightPanoramaGateway gateway(*txn);
        EXPECT_FALSE(
            gateway.exists(table::TrafficLightPanorama::panoramaId.equals(panoramaTwo.panoramaId()))
        );
        EXPECT_FALSE(
            gateway.exists(table::TrafficLightPanorama::trafficLightId.equals(trafficLight.id()))
        );
    }
}

TEST_F(Fixture, test_traffic_light_hypothesis) {
    const TId objectId = 10;
    const TId commitId = 12;

    TrafficLights trafficLights{
        newTrafficLight()
            .setMercatorPos(geolib3::Point2(0, 0), 0.1)
            .setHeading(geolib3::Heading(0.0), 1.0_deg),
        newTrafficLight()
            .setMercatorPos(geolib3::Point2(10, 10), 0.1)
            .setHeading(geolib3::Heading(10.0), 2.0_deg)
    };

    { // create traffic lights
        auto txn = txnHandle();

        TrafficLightGateway(*txn).upsert(trafficLights);

        txn->commit();
    }

    TrafficLightHypotheses hypotheses({
        {
            geolib3::Point2(0, 0),
            objectId,
            commitId,
            trafficLights.front().id()
        },
        {
            geolib3::Point2(10, 10),
            objectId,
            commitId,
            trafficLights.back().id()
        }
    });

    { // create traffic light hypotheses
        auto txn = txnHandle();

        TrafficLightHypothesisGateway gateway(*txn);
        gateway.upsert(hypotheses);

        EXPECT_EQ(hypotheses.size(), 2u);
        EXPECT_TRUE(hypotheses.front().id());
        EXPECT_TRUE(hypotheses.back().id());

        txn->commit();
    }

    { // load created traffic light hypothesis by id
        auto txn = txnHandle();

        const TrafficLightHypotheses loadedHypotheses = TrafficLightHypothesisGateway(*txn).load(
            table::TrafficLightHypothesis::id.in({hypotheses.front().id()})
        );

        EXPECT_EQ(loadedHypotheses.size(), 1u);
        EXPECT_EQ(loadedHypotheses.front(), hypotheses.front());
    }

    { // load created traffic light hypothesis by bbox
        auto txn = txnHandle();
        TrafficLightGateway gateway(*txn);

        const TrafficLightHypotheses loadedHypotheses = TrafficLightHypothesisGateway(*txn).load(
            table::TrafficLightHypothesis::position.intersects(
                geolib3::BoundingBox{hypotheses.back().mercatorPos(), 5., 5.}
            )
        );

        EXPECT_EQ(loadedHypotheses.size(), 1u);
        EXPECT_EQ(loadedHypotheses.front(), hypotheses.back());
    }

    { // update traffic light hypothesis
        hypotheses.front().setContext(json::Value::fromString(R"({"context": 10})"));

        auto txn = txnHandle();

        TrafficLightHypothesisGateway gateway(*txn);
        gateway.upsert(hypotheses);

        txn->commit();
    }

    { // load updated traffic light hypothesis by id
        auto txn = txnHandle();

        const TrafficLightHypotheses loadedHypotheses = TrafficLightHypothesisGateway(*txn).load(
            table::TrafficLightHypothesis::id.in({hypotheses.front().id()})
        );

        EXPECT_EQ(loadedHypotheses.size(), 1u);
        EXPECT_EQ(loadedHypotheses.front(), hypotheses.front());
    }

    { // remove first traffic light
        auto txn = txnHandle();

        TrafficLightGateway gateway(*txn);
        gateway.removeById(trafficLights.front().id());

        txn->commit();
    }

    { // load traffic light hypothesis by id with removed traffic light
        auto txn = txnHandle();

        const TrafficLightHypotheses loadedHypotheses = TrafficLightHypothesisGateway(*txn).load(
            table::TrafficLightHypothesis::id.in({hypotheses.front().id()})
        );

        EXPECT_TRUE(loadedHypotheses.empty());
    }

    { // load left traffic light hypothesis by id
        auto txn = txnHandle();

        const TrafficLightHypotheses loadedHypotheses = TrafficLightHypothesisGateway(*txn).load(
            table::TrafficLightHypothesis::id.in({hypotheses.back().id()})
        );

        EXPECT_EQ(loadedHypotheses.size(), 1u);
        EXPECT_EQ(loadedHypotheses.front(), hypotheses.back());
    }

    { // remove first traffic light hypothesis
        auto txn = txnHandle();

        TrafficLightHypothesisGateway gateway(*txn);
        gateway.removeById(hypotheses.back().id());

        txn->commit();
    }

    { // check removed traffic light hypothesis by id
        auto txn = txnHandle();

        const TrafficLightHypotheses loadedHypotheses = TrafficLightHypothesisGateway(*txn).load(
            table::TrafficLightHypothesis::id.in({hypotheses.back().id()})
        );

        EXPECT_TRUE(loadedHypotheses.empty());
    }
}

TEST_F(Fixture, test_traffic_light_toloka) {
    const TId objectId = 10;
    const TId commitId = 12;

    TrafficLights trafficLights{
        newTrafficLight()
            .setMercatorPos(geolib3::Point2(0, 0), 0.1)
            .setHeading(geolib3::Heading(0.0), 1.0_deg),
        newTrafficLight()
            .setMercatorPos(geolib3::Point2(10, 10), 0.1)
            .setHeading(geolib3::Heading(10.0), 2.0_deg)
    };

    { // create traffic lights
        auto txn = txnHandle();

        TrafficLightGateway(*txn).upsert(trafficLights);

        txn->commit();
    }

    TrafficLightHypotheses hypotheses({
        {
            geolib3::Point2(0, 0),
            objectId,
            commitId,
            trafficLights.front().id()
        },
        {
            geolib3::Point2(10, 10),
            objectId,
            commitId,
            trafficLights.back().id()
        }
    });

    { // create traffic light hypotheses
        auto txn = txnHandle();

        TrafficLightHypothesisGateway gateway(*txn);
        gateway.upsert(hypotheses);

        txn->commit();
    }

    const std::string workflowId         = "NIRVANA-WORKFLOW-ID";
    const std::string workflowInstanceId = "NIRVANA-WORKFLOW-INSTANCE-ID";
    const std::string blockGUID          = "NIRVANA-BLOCK-GUID";

    const std::string workflowId2         = "NIRVANA-WORKFLOW-ID-2";
    const std::string workflowInstanceId2 = "NIRVANA-WORKFLOW-INSTANCE-ID-2";
    const std::string blockGUID2          = "NIRVANA-BLOCK-GUID-2";

    {
        auto txn = txnHandle();
        TrafficLightTolokaGateway gtw(*txn);

        TrafficLightTolokas trafficLightTolokas;
        for (const auto& hypothesis : hypotheses) {
            trafficLightTolokas.emplace_back(
                hypothesis.id(),
                workflowId,
                workflowInstanceId,
                blockGUID
            );
        }
        gtw.insert(trafficLightTolokas);
        txn->commit();
    }

    {
        auto txn = txnHandle();
        TrafficLightTolokaGateway gtw(*txn);
        TrafficLightTolokas trafficLightTolokas = gtw.load(
            table::TrafficLightToloka::state == nirvana::nirvana_state::RUNNING
        );

        EXPECT_EQ(hypotheses.size(), trafficLightTolokas.size());

        for (auto& trafficLightToloka : trafficLightTolokas) {
            trafficLightToloka.setState(nirvana::nirvana_state::FAILED);
        }

        gtw.update(trafficLightTolokas);
        txn->commit();
    }

    {
        auto txn = txnHandle();

        TrafficLightTolokaGateway gtw(*txn);

        TrafficLightTolokas trafficLightTolokas = gtw.load(
            (table::TrafficLightToloka::workflowId == workflowId) &&
            (table::TrafficLightToloka::workflowInstanceId == workflowInstanceId) &&
            (table::TrafficLightToloka::blockGUID == blockGUID)
        );

        EXPECT_EQ(hypotheses.size(), trafficLightTolokas.size());

        for (auto& trafficLightToloka : trafficLightTolokas) {
            trafficLightToloka.restartNirvana(
                workflowId2,
                workflowInstanceId2,
                blockGUID2
            );
        }

        gtw.upsert(trafficLightTolokas);
        txn->commit();
    }

    {
        auto txn = txnHandle();

        TrafficLightTolokaGateway gtw(*txn);

        TrafficLightTolokas failedTrafficLightTolokas = gtw.load(
            (table::TrafficLightToloka::workflowId == workflowId) &&
            (table::TrafficLightToloka::workflowInstanceId == workflowInstanceId) &&
            (table::TrafficLightToloka::blockGUID == blockGUID)
        );
        EXPECT_TRUE(failedTrafficLightTolokas.empty());

        TrafficLightTolokas trafficLightTolokas = gtw.load(
            (table::TrafficLightToloka::workflowId == workflowId2) &&
            (table::TrafficLightToloka::workflowInstanceId == workflowInstanceId2) &&
            (table::TrafficLightToloka::blockGUID == blockGUID2)
        );
        EXPECT_EQ(hypotheses.size(), trafficLightTolokas.size());
    }
}

} // namespace maps::mrc::db::tests
