#include <maps/wikimap/mapspro/services/mrc/libs/db/include/txn_id.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/tests/dummy_object.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/sql_chemistry/include/system_information.h>

namespace maps::mrc::db::tests {

struct TxnIdFixture: public ObjectFixtureBase {
    TxnIdFixture() {
        {   // first commit
            auto txn = txnHandle();

            Objects objects{Object("1"), Object("2")};
            setTxnId(*txn, objects);
            Gateway(*txn).insertx(objects);

            txn->commit();
        }

        {   // second commit
            auto txn = txnHandle();

            Objects objects{Object("3")};

            setTxnId(*txn, objects);
            Gateway(*txn).insertx(objects);

            txn->commit();
        }
    }
};


TEST_F(TxnIdFixture, txn_id_set)
{
    auto txn = txnHandle();

    Objects objects{{"4"}, {"5"}, {"6"}};
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    EXPECT_EQ(setTxnId(*txn, objects), txnId);

    EXPECT_EQ(objects[0].txnId(), txnId);
    EXPECT_EQ(objects[1].txnId(), txnId);
    EXPECT_EQ(objects[2].txnId(), txnId);

    Gateway(*txn).insertx(objects);
    txn->commit();

    EXPECT_EQ(Gateway(*txnHandle()).count(table::Object::txnId == txnId), 3u);
}

TEST_F(TxnIdFixture, txn_id_update)
{
    auto txn = txnHandle();

    Objects objects = Gateway(*txn).load(sql_chemistry::limit(2));

    objects[0].setValue("4");
    objects[1].setValue("5");

    const TId txnId = setTxnId(*txn, objects);
    Gateway(*txn).updatex(objects);

    txn->commit();

    EXPECT_EQ(Gateway(*txnHandle()).count(table::Object::txnId == txnId), 2u);
}

TEST_F(TxnIdFixture, txn_id_gateway_insertx_one)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Object object{"4"};
    EXPECT_EQ(Gateway(*txn).insertx(object), txnId);
    txn->commit();

    // Update object
    EXPECT_TRUE(object.id());
    EXPECT_EQ(object.txnId(), txnId);

    const auto filter = (table::Object::value == "4") and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 1u);
}

TEST_F(TxnIdFixture, txn_id_insertx_one_with_move)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Object object{"4"};
    EXPECT_EQ(Gateway(*txn).insertx(std::move(object)), txnId);
    txn->commit();

    const auto filter = (table::Object::value == "4") and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 1u);
}

TEST_F(TxnIdFixture, txn_id_insertx_array_ref)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Objects objects{{"4"}, {"5"}};
    EXPECT_EQ(Gateway(*txn).insertx(TArrayRef<Object>(objects)) ,txnId);
    txn->commit();

    // Update objects
    EXPECT_TRUE(objects[0].id());
    EXPECT_TRUE(objects[1].id());

    EXPECT_EQ(objects[0].txnId(), txnId);
    EXPECT_EQ(objects[1].txnId(), txnId);

    auto filter = table::Object::value.in({"4", "5"});
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 2u);
}

TEST_F(TxnIdFixture, txn_id_insertx_vector_with_move)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Objects objects{{"4"}, {"5"}};
    EXPECT_EQ(Gateway(*txn).insertx(std::move(objects)), txnId);
    txn->commit();

    const auto filter = table::Object::value.in({"4", "5"}) and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 2u);
}

TEST_F(TxnIdFixture, txn_id_updatex_one)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    auto object = Gateway(*txn).loadOne(table::Object::value == "3");
    EXPECT_EQ(Gateway(*txn).updatex(object.setValue("4")), txnId);

    txn->commit();

    const auto filter = (table::Object::value == "4") and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 1u);
}

TEST_F(TxnIdFixture, txn_id_updatex_one_with_move)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    auto object = Gateway(*txn).loadOne(table::Object::value == "3");
    EXPECT_EQ(Gateway(*txn).updatex(std::move(object.setValue("4"))), txnId);

    txn->commit();

    const auto filter = (table::Object::value == "4") and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 1u);
}

TEST_F(TxnIdFixture, txn_id_updatex_array_ref)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Objects objects = Gateway(*txn).load(table::Object::value.in({"2", "3"}));
    EXPECT_EQ(objects.size(), 2u);

    objects[0].setValue("4");
    objects[1].setValue("5");

    EXPECT_EQ(Gateway(*txn).updatex(TArrayRef<Object>(objects)), txnId);

    txn->commit();

    const auto filter = (table::Object::value.in({"4", "5"})) and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 2u);
}

TEST_F(TxnIdFixture, txn_id_updatex_vector_with_move)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Objects objects = Gateway(*txn).load(table::Object::value.in({"2", "3"}));
    EXPECT_EQ(objects.size(), 2u);

    objects[0].setValue("4");
    objects[1].setValue("5");

    EXPECT_EQ(Gateway(*txn).updatex(std::move(objects)), txnId);

    txn->commit();

    const auto filter = (table::Object::value.in({"4", "5"})) and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 2u);
}

TEST_F(TxnIdFixture, txn_id_upsert_one)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    auto object = Gateway(*txn).loadOne(table::Object::value == "3");
    EXPECT_EQ(Gateway(*txn).upsertx(object.setValue("4")), txnId);

    txn->commit();

    const auto filter = (table::Object::value == "4") and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 1u);
}

TEST_F(TxnIdFixture, txn_id_upsert_one_with_move)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    auto object = Gateway(*txn).loadOne(table::Object::value == "3");
    EXPECT_EQ(Gateway(*txn).upsertx(std::move(object.setValue("4"))), txnId);

    txn->commit();

    const auto filter = (table::Object::value == "4") and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 1u);
}

TEST_F(TxnIdFixture, txn_id_upsertx_array_ref)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Objects objects = Gateway(*txn).load(table::Object::value.in({"2", "3"}));
    EXPECT_EQ(objects.size(), 2u);

    objects[0].setValue("4");
    objects[1].setValue("5");

    EXPECT_EQ(Gateway(*txn).upsertx(TArrayRef<Object>(objects)), txnId);

    txn->commit();

    const auto filter = (table::Object::value.in({"4", "5"})) and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 2u);
}

TEST_F(TxnIdFixture, txn_id_upsertx_vector_with_move)
{
    auto txn = txnHandle();
    const TId txnId = sql_chemistry::SystemInformation(*txn).getCurrentTransactionId();

    Objects objects = Gateway(*txn).load(table::Object::value.in({"2", "3"}));
    EXPECT_EQ(objects.size(), 2u);

    objects[0].setValue("4");
    objects[1].setValue("5");

    EXPECT_EQ(Gateway(*txn).upsertx(std::move(objects)), txnId);

    txn->commit();

    const auto filter = (table::Object::value.in({"4", "5"})) and (table::Object::txnId == txnId);
    EXPECT_EQ(Gateway(*txnHandle()).count(filter), 2u);
}

TEST_F(TxnIdFixture, txn_id_load_txn_ids)
{
    auto txn = txnHandle();

    const db::TIds txnIds = Gateway(*txn).loadTxnIds(
        table::Object::value.in({"2", "3"}),
        sql_chemistry::orderBy(table::Object::value)
    );

    EXPECT_EQ(txnIds.size(), 2u);

    const auto objects = Gateway(*txn).load(
        table::Object::value.in({"2", "3"}),
        sql_chemistry::orderBy(table::Object::value)
    );

    EXPECT_EQ(objects.size(), 2u);

    EXPECT_EQ(txnIds[0], objects[0].txnId());
    EXPECT_EQ(txnIds[1], objects[1].txnId());
}

template<typename Collection>
Collection ordered(Collection collection)
{
    std::sort(collection.begin(), collection.end());
    return collection;
}

TEST_F(TxnIdFixture, txn_id_load_batch)
{
    auto txn = txnHandle();

    const auto objects = Gateway(*txn).load(sql_chemistry::orderBy(table::Object::value));

    const db::TId txnId = objects[0].txnId();
    const db::TIds expected {objects[0].id(), objects[1].id()};

    {
        const auto [frameIds, _, __] = Gateway(*txn).loadBatch(txnId, 1);
        EXPECT_EQ(ordered(frameIds), ordered(expected));
    }

    {
        const auto [frameIds, _, __] = Gateway(*txn).loadBatch(txnId, 2);
        EXPECT_EQ(ordered(frameIds), ordered(expected));
    }
}

TEST_F(TxnIdFixture, txn_id_load_batch_with_filter)
{
    auto txn = txnHandle();

    const auto objects = Gateway(*txn).load(sql_chemistry::orderBy(table::Object::value));

    const db::TId txnId = objects[0].txnId();

    const auto [frameIds, _, __] = Gateway(*txn).loadBatch(txnId, 2, table::Object::value != "2");
    const db::TIds expected {objects[0].id(), objects[2].id()};

    EXPECT_EQ(ordered(frameIds), ordered(expected));
}

TEST_F(TxnIdFixture, txn_id_queue_functions)
{
    auto txn = txnHandle();
    Gateway gtw(*txn);
    EXPECT_EQ(gtw.queueSize(0), 3u);
    EXPECT_TRUE(gtw.queueSizeIsGreaterThan(0, 2));
    EXPECT_FALSE(gtw.queueSizeIsGreaterThan(0, 3));
}


} // namespace maps::mrc::db::tests
