#include "construct.h"
#include "fixture.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/toloka/task_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/track_point_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/assignment_object_gateway.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/ugc/gateway.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/introspection/include/hashing.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/test_tools/comparison.h>

#include <boost/lexical_cast.hpp>

#include <algorithm>
#include <chrono>
#include <cstdint>
#include <string>
#include <vector>

namespace maps::mrc::db::ugc::tests {
using namespace ::testing;

namespace {

struct TargetData {
    geolib3::Polyline2 polyline;
    std::optional<std::uint32_t> forwardPos;
    std::optional<std::uint32_t> backwardPos;
};

const std::vector<TargetData> TEST_TARGETS{
    {
        geolib3::Polyline2{geolib3::PointsVector{
            {0, 0},
            {0, 1},
            {1, 1}
        }},
        1,
        4
    },
    {
        geolib3::Polyline2{geolib3::PointsVector{
            {0, 0},
            {-1, 0},
            {-1, -1}
        }},
        2,
        3
    }

};

const geolib3::MultiPolygon2 TEST_HULL{{
    geolib3::Polygon2{geolib3::PointsVector{
        {0, 0},
        {0, 1},
        {1, 1},
        {0, 0}
    }},
    geolib3::Polygon2{geolib3::PointsVector{
        {0, 0},
        {-1, 0},
        {-1, -1},
        {0, 0}
    }}
}};

const std::chrono::seconds TEST_DURATION = std::chrono::hours(24) * 80;
const double TEST_DISTANCE_IN_METERS = 40000000;

const Locale RUSSIAN = boost::lexical_cast<Locale>("ru");
const Locale ENGLISH = boost::lexical_cast<Locale>("en");
const Locale FRENCH = boost::lexical_cast<Locale>("fr");
const TaskNamesMap TEST_TASK_NAMES{
    {RUSSIAN, "Вокруг света за 80 дней"},
    {FRENCH, "Le tour du monde en quatre-vingts jours"},
    {ENGLISH, "Eighty days round the world"}
};

const std::string TEST_USER_ID = "0xDEADBEAF";
const std::string TEST_CONCURENT_USER_ID = "0xBADCODE";

const std::vector<geolib3::Point2> TEST_TRACK_POINTS{
    {0, 0.5},
    {0.5, 1}
};

const std::vector<mds::Key> TEST_PHOTO_MDS_KEYS{
    {"31337", "aaa"},
    {"73313", "bbb"}
};

using AssignmentObjectComment = std::optional<std::string>;
using AssignmentObjectData = std::tuple<geolib3::Point2,
    AssignmentObjectType,
    AssignmentObjectComment>;
const std::vector<AssignmentObjectData> TEST_ASSIGNMENT_OBJECTS{
    {{0, 1}, AssignmentObjectType::Barrier, AssignmentObjectComment{"I see a barrier"}},
    {{3, 4}, AssignmentObjectType::Deadend, std::nullopt},
    {{5, 6}, AssignmentObjectType::BadConditions, std::nullopt},
    {{7, 8}, AssignmentObjectType::NoEntry, std::nullopt}
};

Task createTestTask() {
    Task task{};
    task
        .setStatus(TaskStatus::New)
        .setDuration(TEST_DURATION)
        .setDistanceInMeters(TEST_DISTANCE_IN_METERS)
        .setGeodeticHull(TEST_HULL);

    for (const auto& taskName: TEST_TASK_NAMES) {
        task.addName(taskName.first, taskName.second);
    }

    for (const auto& targetData: TEST_TARGETS) {
        task.addTarget(
            targetData.polyline,
            Oneway::No,
            Traffic::RightHand,
            Direction::Bidirectional,
            targetData.forwardPos,
            targetData.backwardPos
        );
    }

    return task;
}

template <typename T>
bool isEqual(const T& lhs, const T& rhs);

template <>
bool isEqual(const geolib3::Polyline2& lhs, const geolib3::Polyline2& rhs) {
    return lhs.points() == rhs.points();
}

template <>
bool isEqual(const geolib3::PolylinesVector& lhs,
    const geolib3::PolylinesVector& rhs) {
    return lhs.size() == rhs.size()
        && std::equal(lhs.begin(), lhs.end(), rhs.begin(),
            isEqual<geolib3::Polyline2>);
}

} //anonymous namespace

using Fixture = maps::mrc::db::tests::Fixture;

TEST_F(Fixture, ugc_tests_creating_new_task) {
    TId taskId = 0;

    {
        //creating new task, pushing it into database
        auto txn = txnHandle();

        auto task = createTestTask();

        EXPECT_NO_THROW(task.names());
        EXPECT_EQ(task.names().size(), 3u);
        EXPECT_NO_THROW(task.targets());
        EXPECT_EQ(task.targets().size(), TEST_TARGETS.size());

        EXPECT_THROW(
            task.addName(FRENCH, "Vingt mille lieues sous les mers"),
            Exception
        );

        TaskGateway{*txn}.insert(task);

        EXPECT_NE(task.id(), 0u);
        EXPECT_NO_THROW(task.targets());
        EXPECT_NO_THROW(task.names());
        taskId = task.id();

        txn->commit();
    }

    {
        //loading previously created task, checking its state
        auto txn = txnHandle();

        Task task = TaskGateway{*txn}.loadById(taskId);

        EXPECT_THROW(task.names(), RuntimeError);
        EXPECT_THROW(task.targets(), RuntimeError);

        EXPECT_EQ(task.id(), taskId);

        EXPECT_EQ(
            task.durationInSeconds().count(),
            TEST_DURATION.count()
        );
        EXPECT_EQ(task.distanceInMeters(), TEST_DISTANCE_IN_METERS);
        EXPECT_TRUE(geolib3::test_tools::approximateEqual(
            task.geodeticHull(),
            TEST_HULL,
            0.0
        ));

        /*
         * When names are not loaded, an exception will be throw upon task save,
         * not upon adding the name to in-memory object
         */
        EXPECT_NO_THROW(
            task.addName(FRENCH, "Vingt mille lieues sous les mers")
        );
        EXPECT_THROW(TaskGateway{*txn}.update(task), RuntimeError);
    }
    {
        //loading previously created task along with its sibling objects
        auto txn = txnHandle();
        TaskGateway gtw{*txn};
        Task task = gtw.loadById(taskId);
        gtw.loadNames(task);
        gtw.loadTargets(task);

        EXPECT_NO_THROW(task.names());
        for (const auto& testName: TEST_TASK_NAMES) {
            EXPECT_EQ(task.names().at(testName.first), testName.second);
        }

        EXPECT_NO_THROW(task.targets());
        EXPECT_EQ(task.targets().size(), TEST_TARGETS.size());

        for (const auto& targetData : TEST_TARGETS) {
            bool found = false;
            for (const auto& target : task.targets()) {
                if (target.forwardPos() == targetData.forwardPos &&
                    target.backwardPos() == targetData.backwardPos &&
                    geolib3::test_tools::approximateEqual(
                        target.geodeticGeom(),
                        targetData.polyline,
                        0.
                    )
                    ) {
                    found = true;
                    break;
                }
            }
            EXPECT_TRUE(found);
        }
    }
}

TEST_F(Fixture, ugc_tests_modify_task) {
    TId taskId = 0;
    std::vector<CameraDeviation> CAMERA_DEVIATIONS
        = {CameraDeviation::Front, CameraDeviation::Right};

    {
        //creating new task, pushing it into database
        auto txn = txnHandle();
        auto task = createTestTask();
        task.setStatus(TaskStatus::Acquired);
        task.setCameraDeviations(CAMERA_DEVIATIONS);
        TaskGateway{*txn}.insert(task);
        txn->commit();

        taskId = task.id();
    }

    {
        auto txn = txnHandle();
        auto task = TaskGateway{*txn}.loadById(taskId);
        EXPECT_EQ(task.status(), TaskStatus::Acquired);
        EXPECT_EQ(task.cameraDeviations(), CAMERA_DEVIATIONS);
    }
}

TEST_F(Fixture, ugc_tests_task_concurrency_guard) {
    TId taskId = 0;
    {
        auto txn = txnHandle();
        auto task = createTestTask();
        TaskGateway{*txn}.insert(task);
        taskId = task.id();
        txn->commit();
    }
    {
        auto txn1 = txnHandle();
        auto txn2 = txnHandle();

        /*
         * It does not matter which gateway/ transaction
         * will be used to load the task.
         * Both objects are stateless and know nothing about the task
         */
        auto task = TaskGateway{*txn1}.loadById(taskId);

        EXPECT_NO_THROW(TaskGateway{*txn1}.update(task));
        EXPECT_THROW(TaskGateway{*txn2}.update(task), sql_chemistry::EditConflict);
    }
    {
        auto txn1 = txnHandle();
        auto txn2 = txnHandle();

        auto task1 = TaskGateway{*txn1}.loadById(taskId);
        auto task2 = TaskGateway{*txn2}.loadById(taskId);

        /*
         * It does not matter if first transaction will be committed
         * before applying save operation from the second.
         * sql_chemistry::EditConflict will be throw in any case
         */
        EXPECT_NO_THROW(TaskGateway{*txn1}.update(task1));
        txn1->commit();
        EXPECT_THROW(TaskGateway{*txn2}.update(task2), sql_chemistry::EditConflict);

        /*
         * task1 now has it's (internally stored) xMin value updated,
         * and can be subsequently saved via txn2
         */
        EXPECT_NO_THROW(TaskGateway{*txn2}.update(task1));
    }
    {
        auto txn1 = txnHandle();
        auto txn2 = txnHandle();
        auto txn3 = txnHandle();

        auto task1 = TaskGateway{*txn1}.loadById(taskId);
        EXPECT_NO_THROW(TaskGateway{*txn1}.update(task1));

        /*
         * txn1 now owns exclusive lock on row containing task being updated
         * concurrent updates issued before committing txn1 will hang waiting for this lock
         *
         * Due to statement_timeout set up above, we will get
         * failure bearing a message "canceling statement due to statement timeout"
         *
         * txn2 will be invalidated due to an exception
         */
        auto task2 = TaskGateway{*txn2}.loadById(taskId);
        txn2->exec("SET statement_timeout TO 500;");
        EXPECT_THROW(TaskGateway{*txn2}.update(task2), pqxx::sql_error);

        /*
         * Commiting txn1 releases the lock and makes further updates available.
         * Yet, we will get sql_chemistry::EditConflict since task2 xMin value was not updated
         */
        txn1->commit();
        EXPECT_THROW(TaskGateway{*txn3}.update(task2), sql_chemistry::EditConflict);
    }
}

TEST_F(Fixture, ugc_tests_assigning_a_task) {
    TId taskId = 0;
    TId assignmentId = 0;
    {
        auto txn = txnHandle();
        auto task = createTestTask();

        EXPECT_THROW(task.assignTo(TEST_USER_ID), RuntimeError);

        TaskGateway{*txn}.insert(task);
        taskId = task.id();
        txn->commit();
    }

    {
        auto txn = txnHandle();
        auto task = TaskGateway{*txn}.loadById(taskId);

        auto assignment = task.assignTo(TEST_USER_ID);
        EXPECT_EQ(task.status(), TaskStatus::Acquired);
        EXPECT_EQ(assignment.status(), AssignmentStatus::Active);

        EXPECT_THROW(task.assignTo(TEST_CONCURENT_USER_ID), RuntimeError);

        TaskGateway{*txn}.update(task);

        size_t oldHash = introspection::computeHash(assignment);
        AssignmentGateway{*txn}.insert(assignment);

        EXPECT_NE(assignment.id(), 0u);
        assignmentId = assignment.id();

        EXPECT_NE(
            oldHash,
            introspection::computeHash(assignment)
        );
        txn->commit();
    }

    {
        auto txn = txnHandle();

        Assignment assignment = AssignmentGateway{*txn}.loadById(assignmentId);

        EXPECT_EQ(assignment.status(), AssignmentStatus::Active);
        EXPECT_EQ(assignment.assignedTo(), TEST_USER_ID);

        assignment.markAsCompleted();
        EXPECT_EQ(assignment.status(), AssignmentStatus::Completed);
        assignment.markAsAccepted();
        EXPECT_EQ(assignment.status(), AssignmentStatus::Accepted);
    }

    {
        auto txn = txnHandle();

        auto assignments = AssignmentGateway{*txn}.load(
            ugc::table::Assignment::status == AssignmentStatus::Active);
        EXPECT_EQ(assignments.size(), 1u);
        EXPECT_EQ(assignments.front().status(), AssignmentStatus::Active);

        auto ids = AssignmentGateway{*txn}.loadIds(
            ugc::table::Assignment::status == AssignmentStatus::Active);
        EXPECT_EQ(ids.size(), 1u);
        EXPECT_EQ(ids.front(), assignments.front().id());

        auto completedIds = AssignmentGateway{*txn}.loadIds(
            ugc::table::Assignment::status == AssignmentStatus::Completed);
        EXPECT_TRUE(completedIds.empty());
    }

    {
        auto txn = txnHandle();
        auto task = createTestTask();
        TaskGateway{*txn}.insert(task);

        auto assignment = task.assignTo(TEST_USER_ID);
        TaskGateway{*txn}.update(task);
        AssignmentGateway{*txn}.insert(assignment);

        AssignmentReview review(assignmentId);
        review.setTolokaStatus(TolokaStatus::InProgress)
            .setCoverageFraction(0.)
            .setActualizationDate(chrono::TimePoint::clock::now());
        AssignmentReviewGateway{*txn}.insert(review);

        const TIds expectedIds = {review.assignmentId(), assignment.id()};
        const TIds ids = AssignmentGateway{*txn}.loadIdsOrderByUpdateTimeDesc(0, 100500);
        EXPECT_EQ(ids, expectedIds);
        EXPECT_EQ(AssignmentGateway{*txn}.count(), 2u);
        txn->commit();
    }
}

/*
 * NB: it makes no sense to check concurrency operations on Assignments
 * since concurrency there is controlled via the same xMin mechanism
 */

TEST_F(Fixture, ugc_tests_assignment_result_operations) {
    TId assignmentId = 0;
    TId tolokaId = 0;
    const double QUALITY = .7;
    const geolib3::Point2 POSITION = {43.9805, 56.2148};
    const geolib3::Heading HEADING(210);
    const double COVERAGE_FRACTION = .97;
    const double GOOD_COVERAGE_FRACTION = .77;
    const double TRACK_DISTANCE_IN_METERS = 101.;
    const std::chrono::seconds TRACK_DURATION{6};
    const chrono::TimePoint ACTUALIZATION_DATE
        = chrono::parseSqlDateTime("2017-10-04 13:15:59+00:00");
    const size_t PROCESSED_PHOTOS = 3;
    const size_t PROCESSED_POINTS = 4;
    const size_t GOOD_PHOTOS = 2;
    const chrono::TimePoint FIRST_SHOT_TIME
        = chrono::parseSqlDateTime("2017-10-02 11:15:59+00:00");
    const chrono::TimePoint LAST_SHOT_TIME
        = chrono::parseSqlDateTime("2017-10-03 12:15:59+00:00");
    const geolib3::PolylinesVector COVERED_GEOM
        = {geolib3::Polyline2(geolib3::PointsVector{{1, 1}, {2, 2}})};
    const geolib3::PolylinesVector UNCOVERED_GEOM
        = {geolib3::Polyline2(geolib3::PointsVector{{2, 2}, {3, 3}})};
    const geolib3::PolylinesVector TRACK_GEOM
        = {geolib3::Polyline2(geolib3::PointsVector{{3, 3}, {4, 4}})};
    const auto CAMERA_DEVIATION = CameraDeviation::Right;

    {
        auto txn = txnHandle();
        toloka::TaskGateway gtw{*txn};

        toloka::Task task(toloka::Platform::Toloka);
        task.setType(toloka::TaskType::ImageQualityClassification)
            .setStatus(toloka::TaskStatus::InProgress)
            .setInputValues("input-values")
            .setOverlap(3)
            .setCreatedAt(chrono::TimePoint::clock::now())
            .setCreatedAt(chrono::TimePoint::clock::now());
        gtw.insertx(task);
        txn->commit();

        tolokaId = task.id();
    }

    {
        auto txn = txnHandle();

        auto task = createTestTask();
        TaskGateway{*txn}.insert(task);

        auto assignment = task.assignTo(TEST_USER_ID);
        AssignmentGateway{*txn}.insert(assignment);

        assignmentId = assignment.id();

        TrackPoints trackPoints;
        trackPoints.reserve(TEST_TRACK_POINTS.size());
        for (const auto& pt : TEST_TRACK_POINTS) {
            trackPoints.emplace_back(
                TrackPoint{}
                    .setTimestamp(chrono::TimePoint::clock::now())
                    .setGeodeticPos(pt)
                    .setAssignmentId(assignmentId));
        }
        TrackPointGateway{*txn}.insert(trackPoints);
        for (const auto& trackPoint : trackPoints) {
            EXPECT_GT(trackPoint.id(), 0u);
        }

        Features photos;
        photos.reserve(TEST_PHOTO_MDS_KEYS.size());
        for (const auto& key : TEST_PHOTO_MDS_KEYS) {
            photos.push_back(
                db::tests::newFeature()
                    .setTimestamp(chrono::TimePoint::clock::now())
                    .setMdsKey(key)
                    .setAssignmentId(assignmentId)
                    .setSize({6, 9}));
        }
        FeatureGateway{*txn}.insert(photos);
        for (const auto& photo: photos) {
            EXPECT_GT(photo.id(), 0u);
        }

        AssignmentObjects objects;
        for (const auto& object : TEST_ASSIGNMENT_OBJECTS) {
            objects.emplace_back(assignmentId, chrono::TimePoint::clock::now(),
                std::get<geolib3::Point2>(object),
                std::get<AssignmentObjectType>(object));

            const auto comment = std::get<AssignmentObjectComment>(object);
            if (comment) {
                objects.back().setComment(std::string(*comment));
            }
        }
        AssignmentObjectGateway{*txn}.insert(objects);
        for (const auto& object : objects) {
            EXPECT_GT(object.objectId(), 0u);
        }

        const auto
            objectIds = AssignmentObjectGateway{*txn}.loadIdsByAssignmentId(assignmentId);
        for (const auto& object : objects) {
            EXPECT_TRUE(std::find(objectIds.begin(), objectIds.end(),
                object.objectId()) != objectIds.end());
        }

        photos.front()
            .setQuality(QUALITY)
            .setGeodeticPos(POSITION)
            .setHeading(HEADING);
        FeatureGateway{*txn}.update(photos, UpdateFeatureTxn::Yes);

        FeatureQaTasks tasks;
        tasks.emplace_back(photos.front().id(), tolokaId);
        FeatureQaTaskGateway{*txn}.insert(tasks);

        AssignmentReview review(assignmentId);
        review.setTolokaStatus(TolokaStatus::InProgress)
            .setCoverageFraction(COVERAGE_FRACTION)
            .setCameraDeviation(CAMERA_DEVIATION);
        AssignmentReviewGateway{*txn}.insert(review);

        txn->commit();
        txn = txnHandle();

        review.setTolokaStatus(TolokaStatus::Accepted);
        review.setGoodCoverageFraction(GOOD_COVERAGE_FRACTION);
        review.setTrackDistanceInMeters(TRACK_DISTANCE_IN_METERS);
        review.setTrackDuration(TRACK_DURATION);
        review.setActualizationDate(ACTUALIZATION_DATE);
        review.setProcessedPhotos(PROCESSED_PHOTOS);
        review.setProcessedPoints(PROCESSED_POINTS);
        review.setGoodPhotos(GOOD_PHOTOS);
        review.setFirstShotTime(FIRST_SHOT_TIME);
        review.setLastShotTime(LAST_SHOT_TIME);
        review.setCoveredGeodeticGeom(COVERED_GEOM);
        review.setUncoveredGeodeticGeom(UNCOVERED_GEOM);
        review.setTrackGeodeticGeom(TRACK_GEOM);
        AssignmentReviewGateway{*txn}.update(review);

        txn->commit();
    }

    {
        auto txn = txnHandle();

        auto trackPoints = TrackPointGateway{*txn}.load(
            db::table::TrackPoint::assignmentId.equals(assignmentId));
        std::sort(trackPoints.begin(), trackPoints.end(),
            [](const auto& lhs, const auto& rhs) {
                return lhs.id() < rhs.id();
            });
        EXPECT_EQ(trackPoints.size(), TEST_TRACK_POINTS.size());
        for (size_t i = 0; i < trackPoints.size(); ++i) {
            EXPECT_EQ(trackPoints[i].geodeticPos(), TEST_TRACK_POINTS[i]);
        }

        auto photos = FeatureGateway{*txn}.load(
            db::table::Feature::assignmentId.equals(assignmentId));
        std::sort(photos.begin(), photos.end(),
            [](const auto& lhs, const auto& rhs) {
                return lhs.id() < rhs.id();
            });
        EXPECT_EQ(photos.size(), TEST_PHOTO_MDS_KEYS.size());
        for (size_t i = 0; i < photos.size(); ++i) {
            auto& feature = photos[i];
            EXPECT_EQ(feature.mdsKey(), TEST_PHOTO_MDS_KEYS[i]);
            if (i == 0) {
                EXPECT_EQ(feature.quality(), QUALITY);
                EXPECT_EQ(feature.geodeticPos(), POSITION);
                EXPECT_EQ(feature.heading(), HEADING);
            } else {
                EXPECT_FALSE(feature.hasQuality());
                EXPECT_FALSE(feature.hasPos());
                EXPECT_FALSE(feature.hasHeading());
            }
        }

        auto objects = AssignmentObjectGateway{*txn}.loadByAssignmentId(assignmentId);
        EXPECT_EQ(objects.size(), TEST_ASSIGNMENT_OBJECTS.size());
        for (size_t i = 0; i < objects.size(); ++i) {
            EXPECT_TRUE(
                objects[i].geodeticPos()
                    == std::get<geolib3::Point2>(TEST_ASSIGNMENT_OBJECTS[i]));
            EXPECT_TRUE(objects[i].objectType()
                == std::get<AssignmentObjectType>(
                    TEST_ASSIGNMENT_OBJECTS[i]));
            EXPECT_TRUE(objects[i].comment()
                == std::get<AssignmentObjectComment>(
                    TEST_ASSIGNMENT_OBJECTS[i]));
        }

        TIds photoIds;
        for (const auto& photo : photos) {
            photoIds.push_back(photo.id());
        }
        auto tasks = FeatureQaTaskGateway{*txn}.load(
            db::table::FeatureQaTask::featureId.in(photoIds));
        EXPECT_EQ(tasks.size(), 1u);
        EXPECT_EQ(tasks.front().featureId(), photos.front().id());
        EXPECT_EQ(tasks.front().tolokaId(), tolokaId);

        auto review = AssignmentReviewGateway{*txn}.tryLoadById(assignmentId);
        EXPECT_TRUE(review);
        EXPECT_EQ(review->assignmentId(), assignmentId);
        EXPECT_EQ(review->tolokaStatus(),
            TolokaStatus::Accepted);
        EXPECT_EQ(review->coverageFraction(), COVERAGE_FRACTION);
        EXPECT_EQ(review->goodCoverageFraction(),
            GOOD_COVERAGE_FRACTION);
        EXPECT_EQ(review->trackDistanceInMeters(),
            TRACK_DISTANCE_IN_METERS);
        EXPECT_TRUE(review->trackDuration() == TRACK_DURATION);
        EXPECT_TRUE(review->actualizationDate() == ACTUALIZATION_DATE);
        EXPECT_EQ(review->processedPhotos(), PROCESSED_PHOTOS);
        EXPECT_EQ(review->processedPoints(), PROCESSED_POINTS);
        EXPECT_EQ(review->goodPhotos(), GOOD_PHOTOS);
        EXPECT_TRUE(review->firstShotTime() == FIRST_SHOT_TIME);
        EXPECT_TRUE(review->lastShotTime() == LAST_SHOT_TIME);
        EXPECT_TRUE(isEqual(review->coveredGeodeticGeom().value(), COVERED_GEOM));
        EXPECT_TRUE(isEqual(review->uncoveredGeodeticGeom().value(), UNCOVERED_GEOM));
        EXPECT_TRUE(isEqual(review->trackGeodeticGeom().value(), TRACK_GEOM));
        EXPECT_EQ(review->cameraDeviation(), CAMERA_DEVIATION);
    }

    TId assignmentPhotoId = 0;
    {
        auto txn = txnHandle();

        Feature assignmentPhoto =
            db::tests::newFeature()
                .setSourceId("src")
                .setGeodeticPos({43.9992, 56.3218})
                .setHeading(geolib3::Heading(111.71))
                .setTimestamp("2016-04-02 05:57:11+03")
                .setMdsKey({"4510", "1460732825/MRC_20160401_085711_1545540516.jpg"})
                .setAssignmentId(assignmentId)
                .setSize({6, 9});
        FeatureGateway{*txn}.insert(assignmentPhoto);
        assignmentPhotoId = assignmentPhoto.id();
        EXPECT_FALSE(assignmentPhoto.isPublished());
        assignmentPhoto.setAutomaticShouldBePublished(true).setIsPublished(true);
        FeatureGateway{*txn}.update(assignmentPhoto, UpdateFeatureTxn::Yes);

        txn->commit();
    }

    {
        auto txn = txnHandle();
        auto assignmentPhotos = FeatureGateway{*txn}.load(
            db::table::Feature::isPublished &&
                db::table::Feature::assignmentId.equals(assignmentId));
        auto it = std::find_if(
            assignmentPhotos.begin(), assignmentPhotos.end(),
            [assignmentPhotoId](const Feature& assignmentPhoto) {
                return assignmentPhoto.id() == assignmentPhotoId;
            });
        EXPECT_NE(it, assignmentPhotos.end());
        EXPECT_EQ(it->id(), assignmentPhotoId);
    }

    {
        auto txn = txnHandle();
        auto reviews = AssignmentReviewGateway{*txn}.loadWithoutGeometry();
        EXPECT_EQ(reviews.size(), 1u);
        auto& review = reviews.front();
        EXPECT_EQ(review.assignmentId(), assignmentId);
        EXPECT_EQ(review.tolokaStatus(),
            TolokaStatus::Accepted);
        EXPECT_EQ(review.coverageFraction(), COVERAGE_FRACTION);
        EXPECT_EQ(review.goodCoverageFraction(),
            GOOD_COVERAGE_FRACTION);
        EXPECT_EQ(review.trackDistanceInMeters(),
            TRACK_DISTANCE_IN_METERS);
        EXPECT_TRUE(review.trackDuration() == TRACK_DURATION);
        EXPECT_TRUE(review.actualizationDate() == ACTUALIZATION_DATE);
        EXPECT_EQ(review.processedPhotos(), PROCESSED_PHOTOS);
        EXPECT_EQ(review.processedPoints(), PROCESSED_POINTS);
        EXPECT_EQ(review.goodPhotos(), GOOD_PHOTOS);
        EXPECT_TRUE(review.firstShotTime() == FIRST_SHOT_TIME);
        EXPECT_TRUE(review.lastShotTime() == LAST_SHOT_TIME);
        EXPECT_EQ(review.cameraDeviation(), CAMERA_DEVIATION);
    }
}

TEST_F(Fixture, ugc_tests_assignment_recording_report_operations) {
    TId assignmentId = 0;
    const std::string SOURCE_ID = "source";
    const maps::mds::Key MDS_KEY1{"MDS_GROUP", "MDS_PATH1"};
    const maps::mds::Key MDS_KEY2{"MDS_GROUP", "MDS_PATH2"};

    EXPECT_THROW(
        {
            auto txn = txnHandle();
            AssignmentRecordingReport report(
                1, // missing assignment
                SOURCE_ID,
                MDS_KEY1
            );
            AssignmentRecordingReportGateway{*txn}.insert(report);
        }, std::exception
    );

    {
        auto txn = txnHandle();
        AssignmentReviewGateway gtw{*txn};
        EXPECT_FALSE(gtw.tryLoadById(1));
    }

    {
        auto txn = txnHandle();

        auto task = createTestTask();
        TaskGateway{*txn}.insert(task);

        auto assignment = task.assignTo(TEST_USER_ID);
        AssignmentGateway{*txn}.insert(assignment);

        assignmentId = assignment.id();
        txn->commit();
    }

    auto txn = txnHandle();
    AssignmentRecordingReportGateway gtw{*txn};
    AssignmentRecordingReport report(assignmentId, SOURCE_ID, MDS_KEY1);
    gtw.insert(report);

    auto loadedReports = gtw.loadByAssignmentId(assignmentId);
    ASSERT_EQ(loadedReports.size(), 1u);
    EXPECT_EQ(loadedReports[0].id(), report.id());
    EXPECT_EQ(loadedReports[0].assignmentId(), report.assignmentId());
    EXPECT_TRUE(loadedReports[0].mdsKey() == report.mdsKey());

    AssignmentRecordingReport nextReport(assignmentId, SOURCE_ID, MDS_KEY2);
    gtw.insert(nextReport);

    EXPECT_EQ(gtw.loadByAssignmentId(assignmentId).size(), 2u);
    txn->commit();
}

TEST_F(Fixture, ugc_tests_track_point_assignment_id)
{
    auto assignmentId = TId{};

    {
        auto txn = txnHandle();
        auto task = createTestTask();
        TaskGateway{*txn}.insert(task);
        auto assignment = task.assignTo(TEST_USER_ID);
        AssignmentGateway{*txn}.insert(assignment);
        assignmentId = assignment.id();
        auto trackPoints = TrackPoints{};

        // with assignment_id
        for (const auto& pos : TEST_TRACK_POINTS) {
            trackPoints.emplace_back(
                TrackPoint{}
                    .setSourceId("source_1")
                    .setTimestamp(chrono::TimePoint::clock::now())
                    .setGeodeticPos(pos)
                    .setAssignmentId(assignmentId));
        }

        // without assignment_id
        trackPoints.emplace_back(
            TrackPoint{}
                .setSourceId("source_2")
                .setTimestamp(chrono::TimePoint::clock::now())
                .setGeodeticPos({0, 0}));

        TrackPointGateway{*txn}.insert(trackPoints);

        txn->commit();
    }

    {
        auto txn = txnHandle();
        auto trackPoints = TrackPointGateway{*txn}.load(
            db::table::TrackPoint::assignmentId.equals(assignmentId));
        EXPECT_EQ(trackPoints.size(), TEST_TRACK_POINTS.size());
        std::sort(trackPoints.begin(),
                  trackPoints.end(),
                  [](auto& lhs, auto& rhs) { return lhs.id() < rhs.id(); });
        for (size_t i = 0; i < trackPoints.size(); ++i) {
            EXPECT_EQ(trackPoints[i].geodeticPos(), TEST_TRACK_POINTS[i]);
        }
    }
}

} // namespace maps::mrc::db::ugc::tests
