#include "construct.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/visibility.h>
#include <yandex/maps/geolib3/sproto.h>
#include <maps/libs/geolib/include/vector.h>

namespace maps::mrc::db::tests {
using namespace maps::geolib3;
using namespace ::testing;

TEST(visibility_tests, test_is_visible) {
    auto track = Polyline2{PointsVector{{37.2545, 55.1030},
        {37.2521, 55.1023},
        {37.2508, 55.1024},
        {37.2501, 55.1020}}};
    auto reversedTrack = track;
    reversedTrack.reverse();

    auto feature = newFeature()
        .setGeodeticPos(Point2{37.2544, 55.1031})
        .setHeading(geolib3::Heading(0)); // north

    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_FALSE(isVisible(feature, reversedTrack));

    feature.setGeodeticPos(Point2{37.2522, 55.1023});
    feature.setHeading(geolib3::Heading(90)); // east
    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_TRUE(isVisible(feature, reversedTrack));

    feature.setGeodeticPos(Point2{37.2510, 55.1025});
    feature.setHeading(geolib3::Heading(180)); // south
    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_FALSE(isVisible(feature, reversedTrack));

    feature.setHeading(geolib3::Heading(160)); // south-south-west
    EXPECT_TRUE(isVisible(feature, reversedTrack));

    feature.setGeodeticPos(Point2{37.2500, 55.1021});
    feature.setHeading(geolib3::Heading(270)); // west
    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_FALSE(isVisible(feature, reversedTrack));
}

TEST(visibility_tests, test_turn_heading) {
    auto track = Polyline2{
        PointsVector{{37.2551, 55.105}, {37.255, 55.110}, {37.250, 55.1101}}};
    auto reversedTrack = track;
    reversedTrack.reverse();
    auto corner = Point2{37.255, 55.110};

    auto feature = newFeature()
        .setGeodeticPos(corner + Vector2(+EPS, +EPS))
        .setHeading(geolib3::Heading(45));

    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_FALSE(isVisible(feature, reversedTrack));

    feature.setGeodeticPos(corner + Vector2(+EPS, -EPS));
    feature.setHeading(geolib3::Heading(135));
    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_TRUE(isVisible(feature, reversedTrack));

    feature.setGeodeticPos(corner + Vector2(-EPS, -EPS));
    feature.setHeading(geolib3::Heading(225));
    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_FALSE(isVisible(feature, reversedTrack));

    feature.setGeodeticPos(corner + Vector2(-EPS, +EPS));
    feature.setHeading(geolib3::Heading(315));
    EXPECT_TRUE(isVisible(feature, track));
    EXPECT_FALSE(isVisible(feature, reversedTrack));
}

TEST(visibility_tests, test_camera_deviation) {
    auto track = Polyline2{PointsVector{
        {44.007890, 56.328282},
        {44.008231, 56.328610},
        {44.007957, 56.328701},
        {44.007641, 56.328387}}};
    auto reversedTrack = track;
    reversedTrack.reverse();
    auto p1 = Point2{44.008252, 56.328637};

    auto feature = newFeature()
        .setGeodeticPos(p1)
        .setHeading(geolib3::Heading(45));

    // <--------
    //         |
    //         |
    // --------- o--->
    //  ^track   ^feature

    // feature's FOV does not intersect with any segment of the track
    EXPECT_FALSE(isVisible(feature, track));
    EXPECT_FALSE(isVisible(feature, reversedTrack));

    feature.setCameraDeviation(CameraDeviation::Right);

    // segment 1 is visible
    EXPECT_TRUE(isVisible(feature, track));
    // reversed segment 2 is visible
    EXPECT_TRUE(isVisible(feature, reversedTrack));

    feature.setCameraDeviation(CameraDeviation::Back);

    // segment 1 is visible
    EXPECT_TRUE(isVisible(feature, track));
    // reversed segment 2 is visible
    EXPECT_TRUE(isVisible(feature, reversedTrack));
}

} // namespace maps::mrc::db::tests
