#include <maps/libs/succinct_buffers/include/writers.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/impl/utility.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/features_writer.h>

#include <boost/algorithm/string/replace.hpp>
#include <boost/lexical_cast.hpp>

using namespace flatbuffers64;

namespace maps::mrc::fb {

namespace {

auto write(const db::ObjectsInPhoto& from, FlatBufferBuilder& to)
{
    auto objects = std::vector<ObjectInPhoto>{};
    for (const auto& objectInPhoto : from) {
        objects.emplace_back(encode(objectInPhoto.type()),
                             objectInPhoto.minX(),
                             objectInPhoto.minY(),
                             objectInPhoto.maxX(),
                             objectInPhoto.maxY());
    }
    return to.CreateVectorOfStructs(objects);
}

auto write(const TPhotoTimes& from, FlatBufferBuilder& to)
{
    auto photoTimes = std::vector<PhotoTime>{};
    for (const auto& [time, featureId] : from) {
        photoTimes.push_back(
            {toMilliseconds(time), static_cast<uint64_t>(featureId)});
    }
    return to.CreateVectorOfSortedStructs(&photoTimes);
}

auto write(const TPhotoTimelines::value_type& from, FlatBufferBuilder& to)
{
    auto sourceId = to.CreateString(from.first);
    auto photoTimes = write(from.second, to);
    auto photoTimelineBuilder = PhotoTimelineBuilder{to};
    photoTimelineBuilder.add_sourceId(sourceId);
    photoTimelineBuilder.add_photoTimes(photoTimes);
    return photoTimelineBuilder.Finish();
}

auto write(const TPhotoTimelines& from, FlatBufferBuilder& to)
{
    auto offsets = std::vector<Offset<PhotoTimeline>>{};
    for (const auto& photoTimeline : from) {
        offsets.emplace_back(write(photoTimeline, to));
    }
    return to.CreateVectorOfSortedTables(&offsets);
}

void dumpPhotoTimelines(const std::string& version,
                        const TPhotoTimelines& from,
                        const std::string& toFile)
{
    auto builder = FlatBufferBuilder{};
    auto ver = builder.CreateString(version);
    auto photoTimelines = write(from, builder);
    auto photoTimelinesBuilder = PhotoTimelinesBuilder{builder};
    photoTimelinesBuilder.add_version(ver);
    photoTimelinesBuilder.add_photoTimelines(photoTimelines);
    builder.Finish(photoTimelinesBuilder.Finish());
    maps::writeFlatBuffersToFile(builder, toFile);
}

void dumpWalkObjectPhotos(const std::string& version,
                          std::vector<WalkObjectPhoto>& from,
                          const std::string& toFile)
{
    auto builder = FlatBufferBuilder{};
    auto ver = builder.CreateString(version);
    auto walkObjectPhotos = builder.CreateVectorOfSortedStructs(&from);
    auto walkObjectPhotosBuilder = WalkObjectPhotosBuilder{builder};
    walkObjectPhotosBuilder.add_version(ver);
    walkObjectPhotosBuilder.add_walkObjectPhotos(walkObjectPhotos);
    builder.Finish(walkObjectPhotosBuilder.Finish());
    maps::writeFlatBuffersToFile(builder, toFile);
}

std::string makeMdsPathPattern(const db::Feature& feature)
{
    auto result = feature.mdsPath();
    boost::replace_all(result, std::to_string(feature.id()), PARAM_MARKER);
    return result;
}

}  // namespace

void FeaturesWriter::add(const db::Feature& feature,
                         const db::ObjectsInPhoto& objectsInPhoto)
{
    REQUIRE(!isDumped, "already dumped");

    auto sourceId = std::optional<Offset<String>>{};
    if (feature.sourceId() != db::feature::NO_SOURCE_ID) {
        sourceId = builder_.CreateSharedString(feature.sourceId());
    }

    auto mdsPathPattern =
        builder_.CreateSharedString(makeMdsPathPattern(feature));

    auto objects = std::optional<Offset<Vector<const ObjectInPhoto*>>>{};
    if (!objectsInPhoto.empty()) {
        objects = write(objectsInPhoto, builder_);
    }

    addFeature(feature, sourceId, mdsPathPattern, objects);
}

void FeaturesWriter::addFeature(
    const db::Feature& feature,
    std::optional<Offset<String>> sourceId,
    Offset<String> mdsPathPattern,
    std::optional<Offset<Vector<const ObjectInPhoto*>>> objects)
{
    /// If the default value is not specified in the schema, it will be 0 for
    /// scalar types, or null for other types. This makes the scalar optional.
    auto featureBuilder = FeatureBuilder(builder_);
    featureBuilder.add_id(feature.id());
    featureBuilder.add_time(toMilliseconds(feature.timestamp()));
    if (sourceId.has_value()) {
        featureBuilder.add_sourceId(sourceId.value());
        photoTimelines_[feature.sourceId()].push_back(
            {feature.timestamp(), feature.id()});
    }
    if (feature.walkObjectId().has_value()) {
        walkObjectPhotos_.push_back(
            {static_cast<uint64_t>(feature.walkObjectId().value()),
             static_cast<uint64_t>(feature.id())});
    }
    if (feature.userId().has_value() &&
        !feature.gdprDeleted().value_or(false) &&
        feature.showAuthorship().value_or(false)) {
        featureBuilder.add_userId(
            boost::lexical_cast<uint64_t>(feature.userId().value()));
    }
    featureBuilder.add_dataset(encode(feature.dataset()));
    featureBuilder.add_mdsGroupId(
        boost::lexical_cast<uint32_t>(feature.mdsGroupId()));
    featureBuilder.add_mdsPathPattern(mdsPathPattern);
    auto pos = feature.geodeticPos();
    featureBuilder.add_posX(pos.x());
    featureBuilder.add_posY(pos.y());
    leafNodes_.push_back(
        {.id = fb_rtree::Id(feature.id()), .bbox = pos.boundingBox()});
    featureBuilder.add_heading(feature.heading().value());
    auto size = feature.size();
    featureBuilder.add_width(size.width);
    featureBuilder.add_height(size.height);
    featureBuilder.add_orientation(static_cast<int>(feature.orientation()));
    featureBuilder.add_cameraDeviation(
        static_cast<int>(feature.cameraDeviation()));
    if (objects.has_value()) {
        featureBuilder.add_objectsInPhoto(objects.value());
    }
    featureBuilder.add_privacy(encode(feature.privacy()));
    featureBuilder.add_graph(encode(feature.graph()));
    if (feature.walkObjectId().has_value()) {
        featureBuilder.add_walkObjectId(feature.walkObjectId().value());
    }
    offsets_.push_back(featureBuilder.Finish());
}

void FeaturesWriter::dumpFeatures(const std::string& version,
                                  db::TId lastTxnId,
                                  TSchemaVersion schemaVersion,
                                  const std::string& toFile)
{
    auto ver = builder_.CreateString(version);
    auto features = builder_.CreateVectorOfSortedTables(&offsets_);
    auto featuresBuilder = FeaturesBuilder(builder_);
    featuresBuilder.add_version(ver);
    featuresBuilder.add_features(features);
    featuresBuilder.add_lastTxnId(lastTxnId);
    featuresBuilder.add_schemaVersion(schemaVersion);
    builder_.Finish(featuresBuilder.Finish());
    maps::writeFlatBuffersToFile(builder_, toFile);
}

void FeaturesWriter::dump(const std::string& version,
                          db::TId lastTxnId,
                          TSchemaVersion schemaVersion,
                          const std::string& toDirectory)
{
    REQUIRE(!isDumped, "already dumped");
    isDumped = true;

    dumpFeatures(
        version, lastTxnId, schemaVersion, toDirectory + "/" + FEATURES_FILE);
    builder_.Reset();
    offsets_.clear();

    dumpPhotoTimelines(
        version, photoTimelines_, toDirectory + "/" + PHOTO_TIMELINES_FILE);
    photoTimelines_.clear();

    dumpWalkObjectPhotos(version,
                         walkObjectPhotos_,
                         toDirectory + "/" + WALK_OBJECT_PHOTOS_FILE);
    walkObjectPhotos_.clear();

    fb_rtree::packRtree(
        version, std::move(leafNodes_), toDirectory + "/" + RTREE_FILE);
    leafNodes_.clear();
}

}  // namespace maps::mrc::fb
