#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/objects_reader.h>

#include <maps/libs/geolib/include/spatial_relation.h>

using namespace flatbuffers64;

namespace maps::mrc::fb {

ObjectsReader::ObjectsReader(const std::string& fromDirectory,
                             EMappingMode mappingMode)
    : objectsStorage_{common::blobFromFile(fromDirectory + "/" + OBJECTS_FILE,
                                           mappingMode)}
    , objects_{GetRoot<Objects>(objectsStorage_.Data())}
    , rtree_{fromDirectory + "/" + RTREE_FILE, mappingMode}
{
    REQUIRE(decode(*objects_->version()) == rtree_.version(),
            "inconsistent versions of "
                << OBJECTS_FILE << "(" << decode(*objects_->version())
                << "), and " << RTREE_FILE << "(" << rtree_.version() << ")");
}

std::string_view ObjectsReader::version() const
{
    return decode(*objects_->version());
}

size_t ObjectsReader::objectsNumber() const
{
    return objects_->objects()->size();
}

const Object* ObjectsReader::object(size_t offset) const
{
    REQUIRE(offset < objectsNumber(), "object offset is out of bounds");
    return objects_->objects()->Get(offset);
}

const Object* ObjectsReader::objectById(db::TId objectId) const
{
    return objects_->objects()->LookupByKey(objectId);
}

std::vector<const Object*> ObjectsReader::objectsInBbox(
    const geolib3::BoundingBox& geoBbox) const
{
    std::vector<const Object*> result;
    auto range = rtree_.allIdsInWindow(
        geoBbox,
        [this](const geolib3::BoundingBox& box, fb_rtree::Id id) {
        auto object = objectById(id);
        return object && geolib3::spatialRelation(
            box, getPos(*object), geolib3::Intersects);
    });
        
    for (db::TId objectId : range) {
        if (auto object = objectById(objectId)) {
            result.push_back(object);
        }
    }
    return result;
}

}  // namespace maps::mrc::fb
