#include <maps/libs/succinct_buffers/include/writers.h>
#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/impl/utility.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/objects_writer.h>

using namespace flatbuffers64;

namespace maps::mrc::fb {

namespace {

auto withDetection = [](auto& item) { return item.detection.has_value(); };
auto withoutDetection = [](auto& item) { return !item.detection.has_value(); };

auto writeTrafficSignAttrs(
    const db::eye::Object& object,
    FlatBufferBuilder& dest)
{
    auto attrs = object.attrs<db::eye::SignAttrs>();

    std::optional<Offset<String>> signTextOffset;
    auto signText = getSignText(attrs.type);
    if (signText) {
        signTextOffset = dest.CreateSharedString(signText.value());
    }

    auto attrsBuilder = TrafficSignAttrsBuilder(dest);
    attrsBuilder.add_trafficSignType(encode(attrs.type));
    if (signTextOffset) {
        attrsBuilder.add_trafficSignText(signTextOffset.value());
    }
    attrsBuilder.add_isTemporary(attrs.temporary);
    return attrsBuilder.Finish();
}

auto writeFeature(
    const db::eye::MrcUrlContext& mrcCtx,
    common::ImageOrientation orientation,
    FlatBufferBuilder& dest)
{
    auto featureBuilder = SourceFeatureBuilder(dest);
    featureBuilder.add_featureId(mrcCtx.featureId);
    featureBuilder.add_orientation(static_cast<int>(orientation));
    return featureBuilder.Finish();
}

auto writePanorama(
    const db::eye::PanoramaUrlContext& panoCtx,
    const db::eye::FrameLocation& frameLocation,
    FlatBufferBuilder& dest)
{
    auto oid = dest.CreateString(panoCtx.oid);

    auto panoramaBuilder = SourcePanoramaBuilder(dest);
    panoramaBuilder.add_oid(oid);
    panoramaBuilder.add_width(panoCtx.size.width);
    panoramaBuilder.add_height(panoCtx.size.height);
    auto pos = frameLocation.geodeticPos();
    panoramaBuilder.add_posX(pos.x());
    panoramaBuilder.add_posY(pos.y());
    panoramaBuilder.add_heading(panoCtx.heading.value());
    panoramaBuilder.add_tilt(panoCtx.tilt.value());
    panoramaBuilder.add_horizontalFov(panoCtx.horizontalFOV.value());
    return panoramaBuilder.Finish();
}

auto writeObjectFrame(
    const FrameWithOptDetection& frameWithOptDetection,
    FlatBufferBuilder& dest)
{
    const auto& frame = frameWithOptDetection.frame;
    const auto& location = frameWithOptDetection.frameLocation;
    const auto& detection = frameWithOptDetection.detection;

    std::optional<flatbuffers64::Offset<SourceFeature>> featureOffset;
    std::optional<flatbuffers64::Offset<SourcePanorama>> panoramaOffset;

    auto urlContext = frame.urlContext();
    if (urlContext.source() == db::eye::FrameSource::Mrc) {
        const auto& mrcCtx = urlContext.mrc();
        featureOffset = writeFeature(mrcCtx, frame.orientation(), dest);
    } else if (urlContext.source() == db::eye::FrameSource::Panorama) {
        const auto& panoCtx = urlContext.panorama();
        panoramaOffset = writePanorama(panoCtx, location, dest);
    }

    ObjectFrameBuilder frameBuilder(dest);
    frameBuilder.add_time(toMilliseconds(frame.time()));
    if (featureOffset) {
        frameBuilder.add_source_type(FrameSource_SourceFeature);
        frameBuilder.add_source(featureOffset.value().Union());
    } else if (panoramaOffset) {
        frameBuilder.add_source_type(FrameSource_SourcePanorama);
        frameBuilder.add_source(panoramaOffset.value().Union());
    }

    if (detection) {
        const auto& b = detection->box();
        auto bbox = ObjectBox(b.minX(), b.minY(), b.maxX(), b.maxY());
        frameBuilder.add_bbox(&bbox);
    }

    return frameBuilder.Finish();
}

template <typename Filter>
std::optional<Offset<Vector<Offset<ObjectFrame>>>>
 writeObjectFrames(
    const std::vector<FrameWithOptDetection>& detections,
    const Filter& filter,
    FlatBufferBuilder& dest
)
{
    auto offsets = std::vector<Offset<ObjectFrame>>{};
    for (const auto& item : detections) {
        if (filter(item)) {
            offsets.push_back(writeObjectFrame(item, dest));
        }
    }
    if (offsets.empty()) {
        return {};
    }

    return dest.CreateVectorOfSortedTables(&offsets);
}

auto writeInformationTables(
    const std::vector<traffic_signs::TrafficSign>& informationTables,
    FlatBufferBuilder& dest)
{
    std::optional<Offset<Vector<uint16_t>>> result;

    if (!informationTables.empty()) {
        std::vector<uint16_t> trafficSignTypes;
        trafficSignTypes.reserve(informationTables.size());
        for (auto item : informationTables) {
            trafficSignTypes.push_back(encode(item));
        }
        result = dest.CreateVector(trafficSignTypes);
    }
    return result;
}

} // namespace

void ObjectsWriter::add(
    const db::eye::Object& object,
    const db::eye::ObjectLocation& objectLocation,
    const std::vector<FrameWithOptDetection>& detections,
    const std::vector<traffic_signs::TrafficSign>& informationTables,
    const std::optional<db::TId>& signsGroupId)
{
    ASSERT(!detections.empty());

    // Inner objects must be written before ObjectBuilder is created
    auto attrs = writeTrafficSignAttrs(object, builder_);
    auto visibleOnFrames = writeObjectFrames(detections, withDetection, builder_);
    auto missingOnFrames = writeObjectFrames(detections, withoutDetection, builder_);
    auto infoTables = writeInformationTables(informationTables, builder_);

    auto objectBuilder = ObjectBuilder(builder_);
  
    objectBuilder.add_id(object.id());
    objectBuilder.add_type(encode(object.type()));
    auto pos = objectLocation.geodeticPos();
    objectBuilder.add_posX(pos.x());
    objectBuilder.add_posY(pos.y());

    leafNodes_.push_back(
        {.id = fb_rtree::Id(object.id()), .bbox = pos.boundingBox()});
    auto heading = eye::decomposeRotation(objectLocation.rotation()).heading;
    objectBuilder.add_heading(heading.value());

    auto detectedItr = std::find_if(detections.begin(), detections.end(),
        [](const auto& item) { return item.detection.has_value(); });
    if (detectedItr != detections.end()) {
        objectBuilder.add_detectedAt(
            toMilliseconds(detectedItr->frame.time()));
    }

    if (object.disappearedAt().has_value()) {
        objectBuilder.add_disappearedAt(
            toMilliseconds(object.disappearedAt().value()));
    }

    objectBuilder.add_attrs_type(ObjectAttrs_TrafficSignAttrs);
    objectBuilder.add_attrs(attrs.Union());
    if (visibleOnFrames) {
        objectBuilder.add_visibleOnFrames(visibleOnFrames.value());
    }
    if (missingOnFrames) {
        objectBuilder.add_missingOnFrames(missingOnFrames.value());
    }
    if (infoTables) {
        objectBuilder.add_informationTables(infoTables.value());
    }
    if (signsGroupId) {
        objectBuilder.add_signsGroupId(signsGroupId.value());
    }

    offsets_.push_back(objectBuilder.Finish());
}

void ObjectsWriter::dump(
    const std::string& version,
    const std::string& toDirectory)
{
    auto ver = builder_.CreateString(version);
    auto objects = builder_.CreateVectorOfSortedTables(&offsets_);
    auto objectsBuilder = ObjectsBuilder(builder_);
    objectsBuilder.add_version(ver);
    objectsBuilder.add_objects(objects);
    builder_.Finish(objectsBuilder.Finish());
    maps::writeFlatBuffersToFile(builder_, toDirectory + "/" + OBJECTS_FILE);
    offsets_.clear();

    fb_rtree::packRtree(
        version, std::move(leafNodes_), toDirectory + "/" + RTREE_FILE);
    leafNodes_.clear();
}

}  // namespace maps::mrc::fb
