#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/impl/utility.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/version.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb_rtree/include/rtree.h>

#include <boost/iterator/iterator_facade.hpp>
#include <boost/range/iterator_range.hpp>

#include <optional>
#include <string>

namespace maps::mrc::fb {

/**
 * Provides access to @ref db::Features by:
 * - featureId;
 * - sourceId and time;
 * - spatial index;
 * - walkObjectId;
 */
class FeaturesReader {
public:
    explicit FeaturesReader(const std::string& fromDirectory,
                            EMappingMode = EMappingMode::Standard);
    std::string_view version() const;
    size_t featuresNumber() const;
    db::TId lastTxnId() const;
    TSchemaVersion schemaVersion() const;

    /// @param offset in range [0, featuresNumber).
    db::Feature feature(size_t offset) const;
    db::ObjectsInPhoto objectsInPhoto(size_t offset) const;

    std::optional<db::Feature> featureById(db::TId) const;
    db::ObjectsInPhoto objectsInPhotoByFeatureId(db::TId) const;
    const fb_rtree::Rtree& rtree() const { return rtree_; };

    /**
     * @param sourceId sets the camera.
     * @return range (pair of iterators) to view all the photos of the camera.
     * The iterator is random access and can be used to access @ref TPhotoTime.
     * Photos are ordered by time.
     */
    auto photoTimes(const std::string& sourceId) const
    {
        auto ptr =
            photoTimelines_->photoTimelines()->LookupByKey(sourceId.c_str());
        if (ptr) {
            return boost::iterator_range<PhotoTimeIterator>(
                (const PhotoTime*)ptr->photoTimes()->data(),
                (const PhotoTime*)ptr->photoTimes()->data() +
                    ptr->photoTimes()->size());
        }
        else {
            return boost::iterator_range<PhotoTimeIterator>();
        }
    }

    db::TIds lookupFeatureIdsByWalkObjectId(db::TId) const;

private:
    const TBlob featuresStorage_;
    const Features* features_;
    const TBlob photoTimelinesStorage_;
    const PhotoTimelines* photoTimelines_;
    const fb_rtree::Rtree rtree_;
    const TBlob walkObjectPhotosStorage_;
    const WalkObjectPhotos* walkObjectPhotos_;

    struct PhotoTimeIterator
        : boost::iterator_adaptor<PhotoTimeIterator,
                                  const PhotoTime*,
                                  TPhotoTime,
                                  std::random_access_iterator_tag,
                                  TPhotoTime> {
        PhotoTimeIterator() = default;
        PhotoTimeIterator(base_type base) : iterator_adaptor_(base) {}

        TPhotoTime dereference() const
        {
            return {fromMilliseconds(base_reference()->time()),
                    db::TId(base_reference()->featureId())};
        }
    };
};

}  // namespace maps::mrc::fb
