#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/features_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/features_writer.h>
#include <maps/libs/geolib/include/spatial_relation.h>

namespace maps::mrc::fb::tests {

namespace {

const auto TIME = chrono::parseSqlDateTime("2021-07-01 12:00:00+03");
const auto VERSION = makeVersion(TIME);
const auto DUMP_DIRECTORY = std::string{"."};
const auto LAST_TXN_ID = db::TId{123};
const auto FEATURE_SCHEMA_VERSION = TSchemaVersion(321);

db::Features makePhotosOrderedByTime(const std::string& sourceId, size_t count)
{
    static auto featureId = db::TId{};
    auto result = db::Features{};
    for (size_t i = 0; i < count; ++i) {
        result.emplace_back(++featureId)
            .setSourceId(sourceId)
            .setGeodeticPos(geolib3::Point2(i + 0.5, i + 0.5))
            .setHeading(geolib3::Heading(45))
            .setTimestamp(TIME + std::chrono::seconds(i))
            .setSize({9, 12})
            .setMdsKey({"123", "/image/" + std::to_string(i)})
            .setDataset(db::Dataset::Agents)
            .setAutomaticShouldBePublished(true);
    }
    return result;
}

db::TId id(db::TId id)
{
    return id;
}

db::TId id(const db::Feature& feature)
{
    return feature.id();
}

auto equalId = [](const auto& lhs, const auto& rhs) {
    return id(lhs) == id(rhs);
};

}  // anonymous namespace

TEST(features, readWrite)
{
    auto features = makePhotosOrderedByTime("source_id", 10);
    features[0].setOrientation(common::ImageOrientation::fromExif(4));
    features[1].setCameraDeviation(db::CameraDeviation::Right);
    features[2].setPrivacy(db::FeaturePrivacy::Restricted);
    features[3].setGraph(db::GraphType::Pedestrian);
    features[4].setUserId("42").setShowAuthorship(true);
    features[5].setWalkObjectId(100500);
    auto writer = FeaturesWriter{};
    for (const auto& feature : features) {
        writer.add(feature, {});
    }
    writer.dump(
        VERSION, LAST_TXN_ID, FEATURE_SCHEMA_VERSION, DUMP_DIRECTORY);

    auto reader = FeaturesReader{DUMP_DIRECTORY};
    EXPECT_EQ(reader.featuresNumber(), features.size());
    EXPECT_EQ(LAST_TXN_ID, reader.lastTxnId());
    EXPECT_EQ(FEATURE_SCHEMA_VERSION, reader.schemaVersion());
    for (const auto& expect : features) {
        auto test = reader.featureById(expect.id());
        EXPECT_TRUE(test.has_value());
        EXPECT_EQ(test->sourceId(), expect.sourceId());
        EXPECT_EQ(test->geodeticPos(), expect.geodeticPos());
        EXPECT_EQ(test->heading(), expect.heading());
        EXPECT_EQ(test->timestamp(), expect.timestamp());
        EXPECT_EQ(test->size(), expect.size());
        EXPECT_EQ(test->mdsKey(), expect.mdsKey());
        EXPECT_EQ(test->dataset(), expect.dataset());
        EXPECT_EQ(test->orientation(), expect.orientation());
        EXPECT_EQ(test->cameraDeviation(), expect.cameraDeviation());
        EXPECT_EQ(test->privacy(), expect.privacy());
        EXPECT_EQ(test->graph(), expect.graph());
        EXPECT_EQ(test->userId(), expect.userId());
        EXPECT_EQ(test->walkObjectId(), expect.walkObjectId());
    }
    EXPECT_NO_THROW(reader.feature(reader.featuresNumber() - 1));
    EXPECT_THROW(reader.feature(reader.featuresNumber()), maps::Exception);
    EXPECT_NO_THROW(reader.objectsInPhoto(reader.featuresNumber() - 1));
    EXPECT_THROW(reader.objectsInPhoto(reader.featuresNumber()),
                 maps::Exception);
}

TEST(features, objectsInPhotos)
{
    auto features = makePhotosOrderedByTime("source_id", 1);
    auto objectsInPhoto = db::ObjectsInPhoto{};
    objectsInPhoto.emplace_back(
        features[0].id(), db::ObjectInPhotoType::Face, 2, 2, 4, 4, 1.);
    objectsInPhoto.emplace_back(
        features[0].id(), db::ObjectInPhotoType::LicensePlate, 5, 5, 7, 7, 1.);
    auto writer = FeaturesWriter{};
    writer.add(features[0], objectsInPhoto);
    writer.dump(
        VERSION, LAST_TXN_ID, FEATURE_SCHEMA_VERSION, DUMP_DIRECTORY);

    auto reader = FeaturesReader{DUMP_DIRECTORY};
    EXPECT_EQ(reader.featuresNumber(), 1u);
    auto tests = reader.objectsInPhotoByFeatureId(features[0].id());
    EXPECT_EQ(tests.size(), objectsInPhoto.size());
    for (size_t i = 0; i < objectsInPhoto.size(); ++i) {
        const auto& expect = objectsInPhoto[i];
        const auto& test = tests[i];
        EXPECT_EQ(test.featureId(), expect.featureId());
        EXPECT_EQ(test.type(), expect.type());
        EXPECT_EQ(test.imageBox(), expect.imageBox());
    }
}

TEST(features, photoTimes)
{
    const size_t COUNT = 5;
    const auto SOURCE_A = std::string{"source_a"};
    const auto SOURCE_B = std::string{"source_b"};
    auto featuresA = makePhotosOrderedByTime(SOURCE_A, COUNT);
    auto featuresB = makePhotosOrderedByTime(SOURCE_B, COUNT);
    auto writer = FeaturesWriter{};
    // shuffle
    for (size_t i = 0; i < COUNT; ++i) {
        writer.add(featuresA[COUNT - i - 1], {});
        writer.add(featuresB[i], {});
    }
    writer.dump(
        VERSION, LAST_TXN_ID, FEATURE_SCHEMA_VERSION, DUMP_DIRECTORY);

    auto reader = FeaturesReader{DUMP_DIRECTORY};
    auto range = reader.photoTimes(SOURCE_A);
    EXPECT_EQ(range.size(), COUNT);
    // ordered by time
    for (size_t i = 0; i < COUNT; ++i) {
        const auto test = range[i];
        const auto& expect = featuresA[i];
        EXPECT_EQ(test.time, expect.timestamp());
        EXPECT_EQ(test.featureId, expect.id());
    }

    auto pos = COUNT / 2;
    auto it = std::lower_bound(range.begin(),
                               range.end(),
                               TPhotoTime{.time = featuresA[pos].timestamp()});
    EXPECT_NE(it, range.end());
    EXPECT_EQ(it->featureId, featuresA[pos].id());
    EXPECT_NE(it, range.begin());
    EXPECT_EQ(std::prev(it)->featureId, featuresA[pos - 1].id());
    EXPECT_NE(std::next(it), range.end());
    EXPECT_EQ(std::next(it)->featureId, featuresA[pos + 1].id());

    range = reader.photoTimes(db::feature::NO_SOURCE_ID);
    EXPECT_TRUE(range.empty());
}

TEST(features, rtree)
{
    auto features = makePhotosOrderedByTime("source_id", 10);
    auto writer = FeaturesWriter{};
    for (const auto& feature : features) {
        writer.add(feature, {});
    }
    writer.dump(
        VERSION, LAST_TXN_ID, FEATURE_SCHEMA_VERSION, DUMP_DIRECTORY);

    auto reader = FeaturesReader{DUMP_DIRECTORY};
    auto intersects = [&](const geolib3::BoundingBox& box, fb_rtree::Id id) {
        auto feature = reader.featureById(id);
        EXPECT_TRUE(feature.has_value());
        return spatialRelation(
            box, feature->geodeticPos(), geolib3::Intersects);
    };
    for (const auto& expect : features) {
        auto tests = reader.rtree().allIdsInWindow(
            expect.geodeticPos().boundingBox(), intersects);
        EXPECT_EQ(std::distance(tests.begin(), tests.end()), 1);
        EXPECT_EQ(db::TId(*tests.begin()), expect.id());
    }
}

TEST(features, walkObjectPhotos)
{
    const size_t COUNT = 5;
    auto featuresA = makePhotosOrderedByTime("source_a", COUNT);
    auto featuresB = makePhotosOrderedByTime("source_b", COUNT);
    auto featuresC = makePhotosOrderedByTime("source_c", COUNT);

    const auto WALK_OBJECT_ID_A = db::TId(100500);
    for (auto& feature : featuresA) {
        feature.setWalkObjectId(WALK_OBJECT_ID_A);
    }

    const auto WALK_OBJECT_ID_B = db::TId(12345);
    for (auto& feature : featuresB) {
        feature.setWalkObjectId(WALK_OBJECT_ID_B);
    }

    const auto WALK_OBJECT_ID_WITHOUT_PHOTOS = db::TId(1546);

    // shuffle
    auto writer = FeaturesWriter{};
    for (size_t i = 0; i < COUNT; ++i) {
        writer.add(featuresA[COUNT - i - 1], {});
        writer.add(featuresB[i], {});
        writer.add(featuresC[i], {});
    }
    writer.dump(
        VERSION, LAST_TXN_ID, FEATURE_SCHEMA_VERSION, DUMP_DIRECTORY);

    auto reader = FeaturesReader{DUMP_DIRECTORY};
    {
        auto featureIds =
            reader.lookupFeatureIdsByWalkObjectId(WALK_OBJECT_ID_A);
        EXPECT_TRUE(std::is_permutation(featuresA.begin(),
                                        featuresA.end(),
                                        featureIds.begin(),
                                        featureIds.end(),
                                        equalId));
    }
    {
        auto featureIds =
            reader.lookupFeatureIdsByWalkObjectId(WALK_OBJECT_ID_B);
        EXPECT_TRUE(std::is_permutation(featuresB.begin(),
                                        featuresB.end(),
                                        featureIds.begin(),
                                        featureIds.end(),
                                        equalId));
    }
    {
        auto featureIds = reader.lookupFeatureIdsByWalkObjectId(
            WALK_OBJECT_ID_WITHOUT_PHOTOS);
        EXPECT_TRUE(featureIds.empty());
    }
}

}  // namespace maps::mrc::fb::tests
