#include <library/cpp/testing/gtest/gtest.h>

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/heading.h>
#include "maps/libs/geolib/include/units.h"
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/services/mrc/eye/lib/location/include/rotation.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/eye/url_context.h>
#include "maps/wikimap/mapspro/services/mrc/libs/fb/impl/utility.h"
#include "yandex/maps/mrc/traffic_signs/signs.h"
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/objects_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/objects_writer.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/traffic_sign_groups_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/traffic_sign_groups_writer.h>


namespace maps::mrc::fb::tests {

namespace {

const auto DUMMY_ID = db::TId{1};
const auto FEATURE_ID_VISIBLE = db::TId{100};
const auto FEATURE_ID_MISSING = db::TId{200};
const auto HEADING_1 = geolib3::Heading(10);
const auto HEADING_2 = geolib3::Heading(20);
const auto ID_1 = db::TId{1};
const auto ID_2 = db::TId{2};
const auto MDS_GROUP_ID = std::string("group_id");
const auto MDS_PATH = std::string("path");
const auto PANO_OID = std::string("oid");
const auto SIZE = common::Size(800, 600);

const auto TIME = chrono::parseSqlDateTime("2021-07-01 12:00:00+03");
const auto DISAPPEARED_TIME = TIME + std::chrono::days{1};
const auto VERSION = makeVersion(TIME);

const auto DUMP_DIRECTORY = std::string{"."};

common::ImageOrientation orientation() {
    return common::ImageOrientation::fromExif(1);
}


db::eye::Object makeTrafficSignObject(
    db::TId id,
    traffic_signs::TrafficSign trafficSign)
{
    maps::json::Builder builder;
    builder << [&](maps::json::ObjectBuilder builder) {
        builder["type"] = traffic_signs::toString(trafficSign);
        builder["temporary"] = false;
    };
    auto jsonAttrs = json::Value::fromString(builder.str());
    auto attrs = db::eye::SignAttrs::fromJson(jsonAttrs);
    return db::eye::Object(id, DUMMY_ID, attrs);
}

template <typename T>
T makeLocation(
    db::TId objectId,
    const geolib3::Point2& geodeticPoint,
    geolib3::Heading heading)
{
    return T(
        objectId,
        geolib3::convertGeodeticToMercator(geodeticPoint),
        eye::toRotation(heading, orientation()));
}

db::eye::Frame makeFrame(
    chrono::TimePoint time,
    const db::eye::UrlContext& urlContext)
{
    return db::eye::Frame(DUMMY_ID, orientation(), urlContext, SIZE, time);
}

db::eye::Detection makeDetection(
    traffic_signs::TrafficSign trafficSign,
    const common::ImageBox& box)
{
    json::Builder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder["box"] << [&](json::ArrayBuilder b) {
            b << [&](json::ArrayBuilder b) { b << box.minX() << box.minY(); }
              << [&](json::ArrayBuilder b) { b << box.maxX() << box.maxY(); };
        };
        builder["type"] = traffic_signs::toString(trafficSign);
        builder["temporary"] = false;
        builder["type_confidence"] = 0.99;
        builder["temporary_confidence"] = 0.99;
    };
    auto jsonAttrs = json::Value::fromString(builder.str());
    auto attrs = db::eye::DetectedSign::fromJson(jsonAttrs);
    return db::eye::Detection(DUMMY_ID, attrs);
}

}  // anonymous namespace

TEST(Objects, read_write_objects)
{
    ObjectsWriter writer;

    // Add SpeedLimit 40 sign
    {
        auto signType = traffic_signs::TrafficSign::ProhibitoryMaxSpeed40;
        auto object = makeTrafficSignObject(ID_1, signType);
        object.setDisappearedAt(DISAPPEARED_TIME);

        auto objectLocation = makeLocation<db::eye::ObjectLocation>(
            object.id(), geolib3::Point2(1.0, 1.0), HEADING_1);

        std::vector<FrameWithOptDetection> detections {
            FrameWithOptDetection {
                .frame = makeFrame(TIME, db::eye::MrcUrlContext{
                    .featureId = FEATURE_ID_VISIBLE,
                    .mdsGroupId = MDS_GROUP_ID,
                    .mdsPath = MDS_PATH
                }),
                .frameLocation = makeLocation<db::eye::FrameLocation>(
                    FEATURE_ID_VISIBLE, geolib3::Point2(1.1, 1.1), HEADING_1),
                .detection = makeDetection(
                    signType, common::ImageBox(10, 10, 20, 20))
            },
            FrameWithOptDetection {
                .frame = makeFrame(DISAPPEARED_TIME, db::eye::MrcUrlContext{
                    .featureId = FEATURE_ID_MISSING,
                    .mdsGroupId = MDS_GROUP_ID,
                    .mdsPath = MDS_PATH
                }),
                .frameLocation = makeLocation<db::eye::FrameLocation>(
                    FEATURE_ID_MISSING, geolib3::Point2(1.1, 1.1), HEADING_1)
                // detection is missing on frame
            }
        };
        std::vector<traffic_signs::TrafficSign> infoTables{};
        writer.add(object, objectLocation, detections, infoTables, DUMMY_ID);
    }

    // Add Parking sign
    {
        auto signType = traffic_signs::TrafficSign::InformationParking;
        auto object = makeTrafficSignObject(ID_2, signType);

        auto objectLocation = makeLocation<db::eye::ObjectLocation>(
            object.id(), geolib3::Point2(2.0, 2.0), HEADING_2);

        std::vector<FrameWithOptDetection> detections {
            // Feature frame
            FrameWithOptDetection {
                .frame = makeFrame(TIME, db::eye::MrcUrlContext{
                    .featureId = FEATURE_ID_VISIBLE,
                    .mdsGroupId = MDS_GROUP_ID,
                    .mdsPath = MDS_PATH
                }),
                .frameLocation = makeLocation<db::eye::FrameLocation>(
                    FEATURE_ID_VISIBLE, geolib3::Point2(2.1, 2.1), HEADING_2),
                .detection = makeDetection(
                    signType, common::ImageBox(20, 20, 30, 30))
            },
            // Panorama frame
            FrameWithOptDetection {
                .frame = makeFrame(TIME, db::eye::PanoramaUrlContext{
                    .oid = PANO_OID,
                    .heading = HEADING_2,
                    .tilt = geolib3::Degrees(0),
                    .horizontalFOV = geolib3::Degrees(90),
                    .size = SIZE}),
                .frameLocation = makeLocation<db::eye::FrameLocation>(
                    DUMMY_ID, geolib3::Point2(2.2, 2.2), HEADING_2),
                .detection = makeDetection(
                    signType, common::ImageBox(20, 20, 30, 30))
            }
        };
        std::vector<traffic_signs::TrafficSign> infoTables{
            traffic_signs::TrafficSign::InformationPaidServices
        };
        writer.add(object, objectLocation, detections, infoTables, {});
    }

    writer.dump(VERSION, DUMP_DIRECTORY);

    ObjectsReader reader(DUMP_DIRECTORY);
    EXPECT_EQ(reader.objectsNumber(), 2u);

    // Read objects by id
    auto object = reader.objectById(ID_1);
    EXPECT_TRUE(object);
    EXPECT_EQ(db::TId(object->id()), ID_1);
    EXPECT_EQ(object->type(), ObjectType::ObjectType_TrafficSign);
    EXPECT_EQ(
        geolib3::Point2(object->posX(), object->posY()),
        geolib3::Point2(1.0, 1.0));
    EXPECT_EQ(object->heading(), 10);
    EXPECT_EQ(object->detectedAt(), toMilliseconds(TIME));
    EXPECT_EQ(object->disappearedAt(), toMilliseconds(DISAPPEARED_TIME));
    EXPECT_EQ(object->attrs_type(), ObjectAttrs_TrafficSignAttrs);

    auto attrs = static_cast<const TrafficSignAttrs*>(object->attrs());
    EXPECT_EQ(
        attrs->trafficSignType(),
        TrafficSignType::TrafficSignType_ProhibitoryMaxSpeed);
    EXPECT_EQ(attrs->trafficSignText()->str(), "40");
    EXPECT_FALSE(attrs->isTemporary());

    auto visibleOnFrames = object->visibleOnFrames();
    EXPECT_TRUE(visibleOnFrames);
    EXPECT_EQ(visibleOnFrames->size(), 1u);
    auto frame = visibleOnFrames->Get(0);
    EXPECT_EQ(frame->time(), toMilliseconds(TIME));
    EXPECT_EQ(frame->source_type(), FrameSource_SourceFeature);
    auto sourceFeature = static_cast<const SourceFeature*>(frame->source());
    EXPECT_EQ(db::TId(sourceFeature->featureId()), FEATURE_ID_VISIBLE);
    EXPECT_EQ(sourceFeature->orientation(), 1);
    auto box = frame->bbox();
    EXPECT_EQ(box->minX(), 10);
    EXPECT_EQ(box->minY(), 10);
    EXPECT_EQ(box->maxX(), 20);
    EXPECT_EQ(box->maxY(), 20);

    auto missingOnFrames = object->missingOnFrames();
    EXPECT_TRUE(missingOnFrames);
    EXPECT_EQ(missingOnFrames->size(), 1u);
    auto missingOnFrame = missingOnFrames->Get(0);
    EXPECT_EQ(missingOnFrame->time(), toMilliseconds(DISAPPEARED_TIME));
    EXPECT_EQ(missingOnFrame->source_type(), FrameSource_SourceFeature);
    sourceFeature = static_cast<const SourceFeature*>(missingOnFrame->source());
    EXPECT_EQ(db::TId(sourceFeature->featureId()), FEATURE_ID_MISSING);

    EXPECT_FALSE(object->informationTables());
    EXPECT_EQ(db::TId(object->signsGroupId()), DUMMY_ID);

    // Get objects by spatial index
    auto objects = reader.objectsInBbox(
        geolib3::BoundingBox{{1.5, 1.5}, {2.5, 2.5}});
    EXPECT_EQ(objects.size(), 1u);

    object = objects.front();
    EXPECT_TRUE(object);
    EXPECT_EQ(db::TId(object->id()), ID_2);
    EXPECT_EQ(object->type(), ObjectType::ObjectType_TrafficSign);
    EXPECT_EQ(
        geolib3::Point2(object->posX(), object->posY()),
        geolib3::Point2(2.0, 2.0));
    EXPECT_EQ(object->heading(), 20);
    EXPECT_EQ(object->detectedAt(), toMilliseconds(TIME));
    EXPECT_EQ(object->disappearedAt(), 0u);
    EXPECT_EQ(object->attrs_type(), ObjectAttrs_TrafficSignAttrs);

    attrs = static_cast<const TrafficSignAttrs*>(object->attrs());
    EXPECT_EQ(
        attrs->trafficSignType(),
        TrafficSignType::TrafficSignType_InformationParking);
    EXPECT_FALSE(attrs->isTemporary());

    visibleOnFrames = object->visibleOnFrames();
    EXPECT_TRUE(visibleOnFrames);
    EXPECT_EQ(visibleOnFrames->size(), 2u);

    frame = visibleOnFrames->Get(0);
    EXPECT_EQ(frame->time(), toMilliseconds(TIME));
    EXPECT_EQ(frame->source_type(), FrameSource_SourceFeature);
    sourceFeature = static_cast<const SourceFeature*>(frame->source());
    EXPECT_EQ(db::TId(sourceFeature->featureId()), FEATURE_ID_VISIBLE);
    EXPECT_EQ(sourceFeature->orientation(), 1);
    box = frame->bbox();
    EXPECT_EQ(box->minX(), 20);
    EXPECT_EQ(box->minY(), 20);
    EXPECT_EQ(box->maxX(), 30);
    EXPECT_EQ(box->maxY(), 30);

    frame = visibleOnFrames->Get(1);
    EXPECT_EQ(frame->time(), toMilliseconds(TIME));
    EXPECT_EQ(frame->source_type(), FrameSource_SourcePanorama);
    auto sourcePano = static_cast<const SourcePanorama*>(frame->source());
    EXPECT_EQ(sourcePano->oid()->str(), PANO_OID);
    EXPECT_EQ(sourcePano->heading(), 20);
    EXPECT_EQ(sourcePano->tilt(), 0);
    EXPECT_EQ(sourcePano->horizontalFov(), 90);
    EXPECT_EQ(sourcePano->width(), SIZE.width);
    EXPECT_EQ(sourcePano->height(), SIZE.height);
    EXPECT_EQ(
        geolib3::Point2(sourcePano->posX(), sourcePano->posY()),
        geolib3::Point2(2.2, 2.2));

    box = frame->bbox();
    EXPECT_EQ(box->minX(), 20);
    EXPECT_EQ(box->minY(), 20);
    EXPECT_EQ(box->maxX(), 30);
    EXPECT_EQ(box->maxY(), 30);

    EXPECT_FALSE(object->missingOnFrames());

    auto informationTables = object->informationTables();
    EXPECT_TRUE(informationTables);
    EXPECT_EQ(informationTables->size(), 1u);
    EXPECT_EQ(informationTables->Get(0),
        TrafficSignType::TrafficSignType_InformationPaidServices);
}

TEST(Objects, read_write_object_groups)
{
    const auto GROUP_ID_1 = db::TId{10};
    const auto GROUP_ID_2 = db::TId{20};

    TrafficSignGroupsWriter writer;
    writer.add(GROUP_ID_1, {ID_1, ID_2},
        GroupOrientation::GroupOrientation_Horizontal);
    writer.add(GROUP_ID_2, {3, 4},
        GroupOrientation::GroupOrientation_Vertical);
    writer.dump(VERSION, DUMP_DIRECTORY);

    TrafficSignGroupsReader reader(DUMP_DIRECTORY);
    EXPECT_EQ(reader.groupsNumber(), 2u);
    auto group = reader.groupById(GROUP_ID_1);
    EXPECT_TRUE(group);
    EXPECT_EQ(db::TId(group->groupId()), GROUP_ID_1);
    auto objectIds = group->objectIds();
    EXPECT_EQ(objectIds->size(), 2u);
    EXPECT_EQ(objectIds->Get(0), ID_1);
    EXPECT_EQ(objectIds->Get(1), ID_2);
    EXPECT_EQ(group->orientation(), GroupOrientation::GroupOrientation_Horizontal);
}

}  // namespace maps::mrc::fb::tests
