#include <library/cpp/testing/gtest/gtest.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/photo_to_edge_pairs_reader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb/include/photo_to_edge_pairs_writer.h>

namespace maps::mrc::fb::tests {

namespace {

const auto GRAPH_VERSION = std::string{"21.11.01-1"};
const auto TIME = chrono::parseSqlDateTime("2021-11-02 11:03:00+03");
const auto MRC_VERSION = makeVersion(TIME);
const auto PHOTO_TO_EDGE_SCHEMA_VERSION = TSchemaVersion(100500);

struct EqualTo {
    bool operator()(const PolylinePosition& lhs,
                    const PolylinePosition& rhs) const
    {
        return lhs.segmentIdx() == rhs.segmentIdx() &&
               lhs.segmentRelPosition() == rhs.segmentRelPosition();
    }

    bool operator()(const PhotoToEdgePair& lhs,
                    const PhotoToEdgePair& rhs) const
    {
        return lhs.featureId() == rhs.featureId() &&
               lhs.edgeId() == rhs.edgeId() &&
               (*this)(lhs.begin(), rhs.begin()) &&
               (*this)(lhs.end(), rhs.end());
    }

    bool operator()(const std::vector<PhotoToEdgePair>& lhs,
                    const std::vector<PhotoToEdgePair>& rhs) const
    {
        return std::is_permutation(
            lhs.begin(), lhs.end(), rhs.begin(), rhs.end(), *this);
    }
};

}  // anonymous namespace

TEST(photoToEdge, readWrite)
{
    auto featureId1 = 314;
    auto featureId2 = 27;
    auto edgeId1 = 1;
    auto edgeId2 = 618;
    auto photoToEdgePairs =
        std::vector<PhotoToEdgePair>{PhotoToEdgePair(featureId1,
                                                     edgeId1,
                                                     PolylinePosition(1, 0.3),
                                                     PolylinePosition(1, 1.0),
                                                     0 /*order*/),
                                     PhotoToEdgePair(featureId1,
                                                     edgeId2,
                                                     PolylinePosition(0, 0.0),
                                                     PolylinePosition(0, 0.7),
                                                     1 /*order*/),
                                     PhotoToEdgePair(featureId2,
                                                     edgeId1,
                                                     PolylinePosition(0, 0.4),
                                                     PolylinePosition(1, 0.9),
                                                     0 /*order*/)};

    auto testFile = "./photo_to_edge_pairs.fb";
    writePhotoToEdgePairsToFile(GRAPH_VERSION,
                                MRC_VERSION,
                                PHOTO_TO_EDGE_SCHEMA_VERSION,
                                photoToEdgePairs,
                                testFile);
    auto reader = PhotoToEdgePairsReader{testFile};
    EXPECT_EQ(reader.graphVersion(), GRAPH_VERSION);
    EXPECT_EQ(reader.mrcVersion(), MRC_VERSION);
    EXPECT_EQ(reader.schemaVersion(), PHOTO_TO_EDGE_SCHEMA_VERSION);
    EXPECT_EQ(reader.photoToEdgePairNumber(), photoToEdgePairs.size());
    auto result = std::vector<PhotoToEdgePair>{};
    for (size_t i = 0; i < reader.photoToEdgePairNumber(); ++i) {
        result.push_back(reader.photoToEdgePair(i));
    }
    EXPECT_TRUE(EqualTo{}(result, photoToEdgePairs));
    EXPECT_THROW(reader.photoToEdgePair(reader.photoToEdgePairNumber()),
                 maps::Exception);
    auto featureIdToPairsMap =
        std::unordered_map<db::TId, std::vector<PhotoToEdgePair>>{};
    for (const auto& pair : photoToEdgePairs) {
        featureIdToPairsMap[pair.featureId()].push_back(pair);
    }
    for (const auto& [featureId, pairs] : featureIdToPairsMap) {
        result = reader.lookupByFeatureId(featureId);
        EXPECT_TRUE(EqualTo{}(result, pairs));
        EXPECT_TRUE(std::is_sorted(
            result.begin(), result.end(), [](const auto& lhs, const auto& rhs) {
                return lhs.order() < rhs.order();
            }));
    }
}

}  // namespace maps::mrc::fb::tests
