#include <contrib/libs/benchmark/include/benchmark/benchmark.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb_rtree/include/packer.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <cmath>

namespace maps::mrc::fb_rtree::bench {

namespace {

const int LEAVES_NUMBER = 10'000'000;
const int COORD_RANGE = 100'000;

geolib3::Point2 randomCoord()
{
    return geolib3::Point2(rand() % COORD_RANGE, rand() % COORD_RANGE);
}

using IdToPointMap = std::unordered_map<Id, geolib3::Point2>;

const IdToPointMap& getIdToPointMap()
{
    static auto singleton = [] {
        auto result = IdToPointMap{};
        result.reserve(LEAVES_NUMBER);
        for (int i = 0; i < LEAVES_NUMBER; ++i) {
            result.insert({Id(i), randomCoord()});
        }
        return result;
    }();
    return singleton;
}

const Rtree& getRtree()
{
    static auto singleton = [] {
        auto leaves = LeafNodes{};
        leaves.reserve(LEAVES_NUMBER);
        for (auto& [id, point] : getIdToPointMap()) {
            leaves.push_back({id, point.boundingBox()});
        }
        return buildRtree("version", std::move(leaves));
    }();
    return *singleton;
}

void inWindow(benchmark::State& state)
{
    const int SIDE = std::sqrt(100. / LEAVES_NUMBER) * COORD_RANGE;  // ~100 IDs
    auto& rtree = getRtree();
    auto intersects = [&](const geolib3::BoundingBox& box, Id id) {
        return spatialRelation(
            box, getIdToPointMap().at(id), geolib3::Intersects);
    };
    for (auto _ : state) {
        auto bbox = geolib3::BoundingBox(randomCoord(), SIDE, SIDE);
        auto rng = rtree.allIdsInWindow(bbox, intersects);
        for (auto id : rng) {
            benchmark::DoNotOptimize(id);
        }
    }
}

void nearest(benchmark::State& state)
{
    const int LIMIT = 100;
    auto& rtree = getRtree();
    auto distance = SquaredDistance{};
    auto distanceToId = [&](const geolib3::Point2& point, Id id) {
        return distance(point, getIdToPointMap().at(id));
    };
    for (auto _ : state) {
        auto ids =
            nearestIds(rtree, randomCoord(), distance, distanceToId, LIMIT);
        benchmark::DoNotOptimize(ids);
    }
}

}  // anonymous namespace

BENCHMARK(inWindow);

BENCHMARK(nearest);

}  // namespace maps::mrc::fb_rtree::bench
