#include "compact_bbox.h"

#include <maps/libs/common/include/exception.h>

#include <algorithm>
#include <cmath>
#include <cstdint>
#include <limits>

namespace maps::mrc::fb_rtree::impl {

namespace {

constexpr int MAX_COMPACT_BBOX_COORD = std::numeric_limits<uint8_t>::max();

uint8_t clampToByteRange(int number)
{
    return std::clamp(number, 0, MAX_COMPACT_BBOX_COORD);
}

}  // namespace

fb::CompactBoundingBox compact(const geolib3::BoundingBox& bbox,
                               const geolib3::BoundingBox& parentBbox)
{
    ASSERT(geolib3::sign(bbox.minX() - parentBbox.minX()) >= 0);
    ASSERT(geolib3::sign(bbox.minY() - parentBbox.minY()) >= 0);
    ASSERT(geolib3::sign(bbox.maxX() - parentBbox.maxX()) <= 0);
    ASSERT(geolib3::sign(bbox.maxY() - parentBbox.maxY()) <= 0);
    double horizontalStepRecip = MAX_COMPACT_BBOX_COORD / parentBbox.width();
    double verticalStepRecip = MAX_COMPACT_BBOX_COORD / parentBbox.height();
    const auto minX = clampToByteRange(
        std::floor(horizontalStepRecip * (bbox.minX() - parentBbox.minX())));
    const auto maxX = clampToByteRange(
        std::ceil(horizontalStepRecip * (bbox.maxX() - parentBbox.minX())));
    const auto minY = clampToByteRange(
        std::floor(verticalStepRecip * (bbox.minY() - parentBbox.minY())));
    const auto maxY = clampToByteRange(
        std::ceil(verticalStepRecip * (bbox.maxY() - parentBbox.minY())));
    return {minX, maxX, minY, maxY};
}

geolib3::BoundingBox uncompact(fb::CompactBoundingBox bbox,
                               const geolib3::BoundingBox& parentBbox)
{
    double horizontalStep = parentBbox.width() / MAX_COMPACT_BBOX_COORD;
    double verticalStep = parentBbox.height() / MAX_COMPACT_BBOX_COORD;
    return {{parentBbox.minX() + horizontalStep * bbox.minX(),
             parentBbox.minY() + verticalStep * bbox.minY()},
            {parentBbox.minX() + horizontalStep * bbox.maxX(),
             parentBbox.minY() + verticalStep * bbox.maxY()}};
}

}  // namespace maps::mrc::fb_rtree::impl
