#pragma once

#include "bbox_iterator.h"
#include "nearest_iterator.h"
#include "node.h"

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/memory_map_option.h>
#include <maps/wikimap/mapspro/services/mrc/libs/fb_rtree/storage.fbs64.h>
#include <maps/libs/geolib/include/bounding_box.h>

#include <util/generic/buffer.h>
#include <util/generic/iterator_range.h>
#include <util/memory/blob.h>

namespace maps::mrc::fb_rtree {

class Rtree {
public:
    Rtree(TBlob&&);
    Rtree(TBuffer);
    Rtree(std::string_view filePath, EMappingMode);

    std::string_view version() const;

    // Returns range of IDs
    auto allIdsInWindow(const geolib3::BoundingBox& bbox,
                        IntersectsWithId intersectsWithId) const
    {
        return MakeIteratorRange(
            impl::BboxIterator{this, bbox, intersectsWithId},
            impl::BboxIterator{});
    }

    auto allIds() const
    {
        IntersectsWithId alwaysTrue =
            [](const geolib3::BoundingBox&, Id) { return true; };

        return MakeIteratorRange(
            impl::BboxIterator{this, root().bbox, alwaysTrue},
            impl::BboxIterator{});
    }

    // Returns range of IDs (nodes with least distance go first)
    auto nearestIds(const geolib3::Point2& point,
                    DistanceToBox distanceToBox,
                    DistanceToId distanceToId) const
    {
        return MakeIteratorRange(
            impl::NearestIterator{this, point, distanceToBox, distanceToId},
            impl::NearestIterator{});
    }

    // Low-level interface that allows you to enumerate rtree nodes
    Node root() const;
    void forEachDirectChild(const Node&,
                            std::function<void(const Node&)>) const;

private:
    const TBlob blob_;
    const fb::Rtree* ptr_;

    size_t branching_;
    const flatbuffers64::Vector<const fb::CompactBoundingBox*>* nodes_;
    VarWidthVectorReader levelBegin_;
    FixedWidthVectorReader ids_;
};

// Returns limited number of IDs (nodes with least distance go first)
Ids nearestIds(const Rtree&,
               const geolib3::Point2&,
               DistanceToBox,
               DistanceToId,
               size_t limit);

}  // namespace maps::mrc::fb_rtree
