#include <maps/wikimap/mapspro/services/mrc/libs/graph_matcher_adapter/impl/utility.h>

#include <maps/wikimap/mapspro/services/mrc/libs/graph_matcher_adapter/include/compact_graph_matcher_adapter.h>

#include <maps/analyzer/libs/graphmatching/include/match_signals.h>
#include <maps/analyzer/libs/graphmatching/include/stream_matcher.h>
#include <maps/analyzer/libs/mapmatching/include/matcher.h>

#include <maps/libs/config/include/config.h>
#include <maps/libs/log8/include/log8.h>


#include <boost/range/algorithm/copy.hpp>

#include <algorithm>
#include <chrono>
#include <map>
#include <optional>


namespace graphmatching = maps::analyzer::graphmatching;

namespace maps::mrc::adapters {

CompactGraphMatcherAdapter::CompactGraphMatcherAdapter(
    const std::string& graphFolder,
    const std::string& matcherConfigFile,
    EMappingMode mappingMode)
    : graph_(graphFolder + "/road_graph.fb", mappingMode)
    , rtree_(graphFolder + "/rtree.fb", graph_, mappingMode)
    , config_(analyzer::graphmatching::MatcherConfig::fromString(
          config::readConfigFile(matcherConfigFile)))
{
    REQUIRE(graph_.version() == rtree_.version(),
            "versions of graph(" << graph_.version() << "), rtree("
                                 << rtree_.version() << ") are not equal");
}


TrackSegments CompactGraphMatcherAdapter::match(
    db::TrackPoints trackPoints) const
{
    TrackSegments trackSegments;
    if (trackPoints.empty()) {
        return trackSegments;
    }
    auto onPathMatched = [&](graphmatching::Signal& srcSignal,
                       graphmatching::Signal& destSignal,
                       graphmatching::Candidate&,
                       graphmatching::Candidate& destCandidate,
                       uint64_t,
                       uint64_t) {
        const auto& path = destCandidate.pathFromPrevious;
        if (!path) {
            return;
        }
        TrackSegments segments;
        auto it = analyzer::shortest_path::iteratePath(*path, graph_);
        for (decltype(it) last; it != last; ++it) {
            if (it->start == it->finish) {
                continue;
            }
            auto segment = road_graph::segmentGeometry(graph_, it->segmentId);
            segments.emplace_back(
                geolib3::Segment2{segment.pointByPosition(it->start),
                                  segment.pointByPosition(it->finish)},
                chrono::TimePoint{},
                chrono::TimePoint{},
                it->segmentId.edgeId
            );
        }

        if (!segments.empty()) {
            boost::copy(makePath(fromBoost(srcSignal.gpsSignal.time()),
                                 fromBoost(destSignal.gpsSignal.time()),
                                 segments),
                        std::back_inserter(trackSegments));
        }
    };

    graphmatching::StreamSignalsMatcher matcher(
        [](auto&&... ){} /*onPointMatched*/,
        [](auto&&... ){} /*onSegmentMatched*/,
        [](auto&&... ){} /* onSignalProcessed */,
        onPathMatched,
        graph_, rtree_, config_,
        graphmatching::FilterConfig{5. /* standingRaduis */, std::nullopt, std::chrono::seconds(300)},
        {} /* default SignalsMatchOptions */
    );

    sortUniqueByTime(trackPoints);
    for (const auto& trackPoint : trackPoints) {
        matcher.push(toGpsSignal(trackPoint));
    }
    matcher.done();
    return trackSegments;
}

std::string CompactGraphMatcherAdapter::graphVersion() const
{
    return std::string{graph_.version()};
}

}  // namespace maps::mrc::adapters
