#include <maps/wikimap/mapspro/services/mrc/libs/graph_matcher_adapter/impl/utility.h>

#include <maps/libs/common/include/math.h>

#include <boost/date_time/posix_time/conversion.hpp>

namespace maps::mrc::adapters {
namespace {

chrono::TimePoint::duration operator*(chrono::TimePoint::duration duration,
                                      double times)
{
    return chrono::TimePoint::duration{(chrono::TimePoint::duration::rep)(
        static_cast<double>(duration.count()) * times)};
}

}  // anonymous namespace

chrono::TimePoint fromBoost(const boost::posix_time::ptime& time)
{
    return chrono::TimePoint::clock::from_time_t(
        boost::posix_time::to_time_t(time));
}

boost::posix_time::ptime toBoost(const chrono::TimePoint& time)
{
    using clock = std::chrono::system_clock;
    return boost::posix_time::from_time_t(clock::to_time_t(
        std::chrono::time_point_cast<clock::time_point::duration>(time)));
}

TrackSegments makePath(chrono::TimePoint startTime,
                       chrono::TimePoint endTime,
                       const TrackSegments& segments)
{
    std::vector<double> lengths(segments.size());
    auto sum = maps::common::partialSumKahan(
        segments.begin(),
        segments.end(),
        lengths.begin(),
        [](const auto& item) { return geolib3::geoLength(item.segment); });
    TrackSegments result;
    if (sum > geolib3::EPS) {  // to avoid division by zero
        result.reserve(segments.size());
        auto start = startTime;
        for (size_t i = 0; i < segments.size(); ++i) {
            auto end = startTime + (endTime - startTime) * (lengths[i] / sum);
            result.emplace_back(
                segments[i].segment,
                start,
                end,
                segments[i].edgeId);
            start = end;
        }
    }
    return result;
}

analyzer::data::GpsSignal toGpsSignal(const db::TrackPoint& trackPoint)
{
    analyzer::data::GpsSignal result;
    result.setLon(trackPoint.geodeticPos().x());
    result.setLat(trackPoint.geodeticPos().y());
    result.setTime(toBoost(trackPoint.timestamp()));
    if (trackPoint.accuracyMeters()) {
        result.setAccuracy(*trackPoint.accuracyMeters());
    }
    if (trackPoint.heading()) {
        result.setDirection(trackPoint.heading()->value());
    }
    if (trackPoint.speedMetersPerSec()) {
        result.setAverageSpeed(*trackPoint.speedMetersPerSec());
    }
    return result;
}

void sortUniqueByTime(db::TrackPoints& trackPoints)
{
    // by timestamp asc, isAugmented desc
    std::sort(trackPoints.begin(),
              trackPoints.end(),
              [](const auto& lhs, const auto& rhs) {
                  return std::make_tuple(toBoost(lhs.timestamp()),
                                         !lhs.isAugmented()) <
                         std::make_tuple(toBoost(rhs.timestamp()),
                                         !rhs.isAugmented());
              });
    // isAugmented aren't removed because they are first
    auto it = std::unique(trackPoints.begin(),
                          trackPoints.end(),
                          [](const auto& lhs, const auto& rhs) {
                              return toBoost(rhs.timestamp()) ==
                                     toBoost(lhs.timestamp());
                          });
    trackPoints.erase(it, trackPoints.end());
}

}  // namespace maps::mrc::adapters
