#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/json/include/value_repr.h>
#include <maps/libs/json/include/std.h>

#include <maps/libs/http/include/http.h>
#include <maps/wikimap/mapspro/services/mrc/libs/nirvana/include/nirvana_workflow.h>


namespace maps::mrc::nirvana {

WorkflowInstance::WorkflowInstance(
    const std::string& _oauthToken,
    const std::string& _workflowId,
    const std::string& _workflowInstanceId,
    const std::string& _quotaID,
    const std::string& _comment)
    : oauthToken(_oauthToken)
    , workflowId(_workflowId)
    , workflowInstanceId(_workflowInstanceId)
    , quotaID(_quotaID)
    , comment(_comment)
    , commandCallID(0) {
    if (workflowInstanceId.empty())
        createWorkflowInstance();
}

std::string WorkflowInstance::getWorkflowInstanceId() const
{
    return workflowInstanceId;
}

BlockPattern WorkflowInstance::addDataBlock(const std::string& dataId)
{
    static const std::string ADD_DATA_BLOCK_COMMAND = "addDataBlocks";

    REQUIRE(!workflowInstanceId.empty(),
        "Unable to add data block because workflow instance does not exist");

    maps::json::Value answer = launchCommand(ADD_DATA_BLOCK_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["blocks"] << [&](maps::json::ArrayBuilder builder) {
                builder << [&](maps::json::ObjectBuilder builder) {
                    builder["storedDataId"] = dataId;
                };
            };
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
    maps::json::Value blockData = answer["result"][0];
    return {blockData["blockGuid"].as<std::string>(), blockData["blockCode"].as<std::string>()};
}

BlockPattern WorkflowInstance::addOperationBlock(const std::string& operationId)
{
    static const std::string ADD_OPERATION_BLOCK_COMMAND = "addOperationBlocks";

    REQUIRE(!workflowInstanceId.empty(),
        "Unable to add operation block because workflow instance does not exist");

    maps::json::Value answer = launchCommand(ADD_OPERATION_BLOCK_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["blocks"] << [&](maps::json::ArrayBuilder builder) {
                builder << [&](maps::json::ObjectBuilder builder) {
                    builder["operationId"] = operationId;
                };
            };
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
    maps::json::Value blockData = answer["result"][0];
    return {blockData["blockGuid"].as<std::string>(), blockData["blockCode"].as<std::string>()};
}

void WorkflowInstance::setBlockParameters(const BlockPattern& blockPattern,
    std::function<void(maps::json::ArrayBuilder)> blockParams)
{
    static const std::string SET_BLOCK_PARAMETERS_COMMAND = "setBlockParameters";

    REQUIRE(!blockPattern.GUID.empty() || !blockPattern.code.empty(),
        "Block code or GUID must be specified");
    REQUIRE(!workflowInstanceId.empty(),
        "Unable to set parameters of block because workflow instance does not exist");

    maps::json::Value answer = launchCommand(SET_BLOCK_PARAMETERS_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["blocks"] << [&](maps::json::ArrayBuilder builder) {
                builder << [&](maps::json::ObjectBuilder builder) {
                    if (!blockPattern.GUID.empty())
                        builder["guid"] = blockPattern.GUID;
                    if (!blockPattern.code.empty())
                        builder["code"] = blockPattern.code;
                };
            };
            builder["params"] << blockParams;
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
}

void WorkflowInstance::connectDataBlockToOperations(const BlockPattern& dataBlockPattern,
    const std::vector<BlockPattern>& operationBlocksPattern,
    const std::string& operationInputName)
{
    static const std::string CONNECT_DATA_BLOCKS_COMMAND = "connectDataBlocks";
    REQUIRE(!workflowInstanceId.empty(),
        "Unable to connect blocks because workflow instance does not exist");

    maps::json::Value answer = launchCommand(CONNECT_DATA_BLOCKS_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["sourceBlocks"] << [&](maps::json::ArrayBuilder builder) {
                builder << [&](maps::json::ObjectBuilder builder) {
                    if (!dataBlockPattern.GUID.empty())
                        builder["guid"] = dataBlockPattern.GUID;
                    if (!dataBlockPattern.code.empty())
                        builder["code"] = dataBlockPattern.code;
                };
            };
            builder["destBlocks"] << [&](maps::json::ArrayBuilder builder) {
                for (size_t i = 0; i < operationBlocksPattern.size(); i++) {
                    builder << [&](maps::json::ObjectBuilder builder) {
                        if (!operationBlocksPattern[i].GUID.empty())
                            builder["guid"] = operationBlocksPattern[i].GUID;
                        if (!operationBlocksPattern[i].code.empty())
                            builder["code"] = operationBlocksPattern[i].code;
                    };
                };
            };
            builder["destInputs"] << [&](maps::json::ArrayBuilder builder) {
                builder << operationInputName;
            };
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
}

void WorkflowInstance::connectOperationBlocks(const std::vector<BlockPattern>& srcBlocksPattern,
    const std::string& srcOutputName,
    const std::vector<BlockPattern>& dstBlocksPattern,
    const std::string& dstInputName)
{
    static const std::string CONNECT_OPERATION_BLOCKS_COMMAND = "connectOperationBlocks";

    REQUIRE(!workflowInstanceId.empty(),
        "Unable to connect blocks because workflow instance does not exist");

    maps::json::Value answer = launchCommand(CONNECT_OPERATION_BLOCKS_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["sourceBlocks"] << [&](maps::json::ArrayBuilder builder) {
                for (size_t i = 0; i < srcBlocksPattern.size(); i++) {
                    builder << [&](maps::json::ObjectBuilder builder) {
                        if (!srcBlocksPattern[i].GUID.empty())
                            builder["guid"] = srcBlocksPattern[i].GUID;
                        if (!srcBlocksPattern[i].code.empty())
                            builder["code"] = srcBlocksPattern[i].code;
                    };
                };
            };
            builder["destBlocks"] << [&](maps::json::ArrayBuilder builder) {
                for (size_t i = 0; i < dstBlocksPattern.size(); i++) {
                    builder << [&](maps::json::ObjectBuilder builder) {
                        if (!dstBlocksPattern[i].GUID.empty())
                            builder["guid"] = dstBlocksPattern[i].GUID;
                        if (!dstBlocksPattern[i].code.empty())
                            builder["code"] = dstBlocksPattern[i].code;
                    };
                };
            };
            builder["sourceOutputs"] << [&](maps::json::ArrayBuilder builder) {
                builder << srcOutputName;
            };
            builder["destInputs"] << [&](maps::json::ArrayBuilder builder) {
                builder << dstInputName;
            };
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
}

std::string WorkflowInstance::getBlockResultUrl(const BlockPattern& blockPattern,
    const std::string& outputName)
{
    static const std::string GET_BLOCK_RESULTS_COMMAND = "getBlockResults";

    REQUIRE(!workflowInstanceId.empty(),
        "Unable to block results because workflow instance does not exist");

    maps::json::Value answer = launchCommand(GET_BLOCK_RESULTS_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
            builder["blocks"] << [&](maps::json::ArrayBuilder builder) {
                builder << [&](maps::json::ObjectBuilder builder) {
                    if (!blockPattern.GUID.empty())
                        builder["guid"] = blockPattern.GUID;
                    if (!blockPattern.code.empty())
                        builder["code"] = blockPattern.code;
                };
            };
            builder["outputs"] << [&](maps::json::ArrayBuilder builder) {
                builder << outputName;
            };

        }
    );
    maps::json::Value data = answer["result"];
    REQUIRE(data.isArray(), "Invalid result format");
    size_t blockIdx = 0;
    for (; blockIdx < data.size(); blockIdx++) {
        const maps::json::Value& val = data[blockIdx];
        if ((!blockPattern.GUID.empty() && blockPattern.GUID == val["blockGuid"].as<std::string>()) ||
            (!blockPattern.code.empty() && blockPattern.code == val["blockCode"].as<std::string>())) {
            break;
        }
    }
    REQUIRE(blockIdx < data.size(), "Unable to found block results");
    const maps::json::Value& blockResult = data[blockIdx]["results"];
    size_t resultIdx = 0;
    for (; resultIdx < blockResult.size(); resultIdx++) {
        const maps::json::Value& val = blockResult[resultIdx];
        if (val["endpoint"].as<std::string>() == outputName) {
            return val["directStoragePath"].as<std::string>();
        }
    }
    return "";
}

bool WorkflowInstance::validateWorkflow()
{
    static const std::string VALIDATE_WORKFLOW_COMMAND = "validateWorkflow";

    REQUIRE(!workflowInstanceId.empty(),
        "Unable to validate workflow because workflow instance does not exist");

    maps::json::Value answer = launchCommand(VALIDATE_WORKFLOW_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
    return answer["result"].as<std::string>() == "ok";
}

bool WorkflowInstance::startWorkflow()
{
    static const std::string START_WORKFLOW_COMMAND = "startWorkflow";

    REQUIRE(!workflowInstanceId.empty(),
        "Unable to start workflow because workflow instance does not exist");

    maps::json::Value answer = launchCommand(START_WORKFLOW_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
    return !answer["result"].as<std::string>().empty();
}

ExecutionState WorkflowInstance::getExecutionState()
{
    static const std::string GET_EXECUTION_STATE_COMMAND = "getExecutionState";

    REQUIRE(!workflowInstanceId.empty(),
        "Unable to get execution state because workflow instance does not exist");

    maps::json::Value answer = launchCommand(GET_EXECUTION_STATE_COMMAND,
        [&](maps::json::ObjectBuilder builder) {
            builder["workflowId"] = workflowId;
            builder["workflowInstanceId"] = workflowInstanceId;
        }
    );
    if (answer["result"]["status"].as<std::string>() != "completed") {
        return ExecutionState::Running;
    }
    if (answer["result"]["result"].as<std::string>() == "success") {
        return ExecutionState::Success;
    }
    return ExecutionState::Failed;
}

void WorkflowInstance::createWorkflowInstance()
{
    static const std::string CREATE_WORKFLOW_INSTANCE = "createWorkflowInstance";

    REQUIRE(workflowInstanceId.empty(),
        "Workflow instance already created with ID: " << workflowInstanceId);
    workflowInstanceId.clear();

    maps::json::Value answer = launchCommand(CREATE_WORKFLOW_INSTANCE,
        [&](maps::json::ObjectBuilder builder) {
            builder["workflowInstance"] << [&](maps::json::ObjectBuilder builder) {
                builder["meta"] << [&](maps::json::ObjectBuilder builder) {
                    builder["quotaProjectId"] = quotaID;
                    builder["comment"] = comment;
                };
            };
            builder["workflowId"] = workflowId;
        }
    );
    workflowInstanceId = answer["result"].as<std::string>();
}

maps::json::Value WorkflowInstance::launchCommand(
    const std::string& commandName,
    std::function<void(maps::json::ObjectBuilder)> params)
{
    static const std::string NIRVANA_URL = "https://nirvana.yandex-team.ru/api/public/v1/";

    maps::common::RetryPolicy retryPolicy;
    retryPolicy.setTryNumber(5)
        .setInitialCooldown(std::chrono::seconds(1))
        .setCooldownBackoff(2);

    auto validateResponse = [&](const auto& httpResponse) {
        if (httpResponse.get().status() != 200) {
            WARN() << "Nirvana request for command: " << commandName << " "
                   << "returned invalid status: " << httpResponse.get().status();
            return false;
        }
        return true;
    };

    maps::json::Builder request;
    request << [&](maps::json::ObjectBuilder builder) {
        builder["jsonrpc"] = "2.0";
        builder["id"] = ++commandCallID;
        builder["method"] = commandName;
        builder["params"] << params;
    };
    maps::http::Response httpResponse = maps::common::retry(
        [&]() {
            maps::http::Request httpRequest(httpClient, maps::http::POST, NIRVANA_URL + commandName);
            httpRequest.addHeader("Authorization", "OAuth " + oauthToken);
            httpRequest.addHeader("Content-Type", "application/json; charset=utf-8");
            httpRequest.setContent(request.str());
            return httpRequest.perform();
        },
        retryPolicy,
        validateResponse
    );
    maps::json::Value answer = maps::json::Value(httpResponse.body());
    REQUIRE(!answer.hasField("error"),
        "Nirvana unable to perfom command " << commandName <<
        "Error: " << answer["error"]["message"].as<std::string>());
    return answer;
}

} // namespace maps::mrc::nirvana
