#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/unittest/include/yandex/maps/wiki/unittest/arcadia.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/revision_loader.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <yandex/maps/shell_cmd.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revisionapi/revisionapi.h>
#include <yandex/maps/wiki/unittest/config.h>
#include <yandex/maps/wiki/unittest/localdb.h>

#include <boost/format.hpp>

#include <fstream>
#include <sstream>
#include <string>
#include <vector>

namespace maps::mrc::object::tests {

namespace {

using MapsproDbFixture = wiki::unittest::MapsproDbFixture;
using ConfigFileHolder = wiki::unittest::ConfigFileHolder;

geolib3::Point2 mercatorPointFromGeoCoords(double lon, double lat)
{
    return geolib3::convertGeodeticToMercator(geolib3::Point2(lon, lat));
}

LoaderHolder makeRevisionLoader(pqxx::transaction_base& txn)
{
    wiki::revision::RevisionsGateway gateway(txn);

    return object::makeRevisionLoader(
        gateway.snapshot(gateway.headCommitId())
    );
}


class Fixture: public wiki::unittest::ArcadiaDbFixture {

public:
    Fixture();

private:
    ConfigFileHolder config_;
};

Fixture::Fixture()
    : config_(database().host(), database().port(), database().dbname(),
              database().user(), database().password())
{
    constexpr wiki::revision::UserID USER_ID = 1;

    static const std::string IMPORT_PATH(
        ArcadiaSourceRoot() + "/maps/wikimap/mapspro/services/mrc/libs/object/tests/objects.json"
    );

    wiki::revisionapi::RevisionAPI revisionAPI(pool());
    std::ifstream in(IMPORT_PATH);

    revisionAPI.importData(USER_ID, wiki::revisionapi::IdMode::PreserveId, in);
}

template<typename TObject>
std::map<TId, TObject> groupById(const std::vector<TObject>& objects)
{
    std::map<TId, TObject> objectsById;

    for (const auto& object: objects) {
        objectsById.emplace(object.id(), object);
    }

    return objectsById;
}


std::set<std::string> ADDRESS_POINT_NAME_SET = {
    "21",
    "23\u043a1",
    "21\u043a1",
    "23\u043a2",
    "23\u043a3",
    "19\u043a6",
    "19\u043a5",
    "19\u043a4",
    "23"
};


} // namespace

TEST(revision_loader, load_road_elements_by_box)
{
    Fixture fixture;

    constexpr double radiusMeter = 30;
    const geolib3::Point2 center = mercatorPointFromGeoCoords(35.1227697, 56.2716454);
    const geolib3::BoundingBox box(center, 2 * radiusMeter, 2 * radiusMeter);

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const auto elements = loader->loadRoadElements(box);
    const auto elementById = groupById(elements);
    ASSERT_EQ(elementById.size(), 2u);

    const TId firstId = 2138599182;
    const auto firstIt = elementById.find(2138599182);
    ASSERT_TRUE(firstIt != elementById.end());
    const auto& first = firstIt->second;

    ASSERT_EQ(first.id(), firstId);
    ASSERT_EQ(first.startJunctionId(), 2138599179);
    ASSERT_EQ(first.endJunctionId(), 2138599180);
    ASSERT_EQ(first.startZLevel(), 0);
    ASSERT_EQ(first.endZLevel(), 0);
    ASSERT_EQ(first.accessId(), RoadElement::AccessId::All);
    ASSERT_EQ(first.speedLimit(), 90);
    ASSERT_EQ(first.speedLimitF(), 90);
    ASSERT_EQ(first.speedLimitT(), 60);
    ASSERT_EQ(first.speedLimitTruckF(), -1);
    ASSERT_EQ(first.speedLimitTruckT(), -1);
    ASSERT_EQ(first.fc(), RoadElement::FunctionalClass::MinorLocalRoad);
    ASSERT_EQ(first.fow(), RoadElement::FormOfWay::None);
    ASSERT_EQ(first.backTransportLane(), false);
    ASSERT_EQ(first.underConstruction(), false);
    ASSERT_EQ(first.trafficLightIds(), TIds({3371233456, 3371233457}));
    ASSERT_EQ(first.speedBumpIds(), TIds({3371233462, 3371233463}));

    const TId secondId = 2138599185;
    const auto secondIt = elementById.find(secondId);
    ASSERT_TRUE(secondIt != elementById.end());
    const auto& second = secondIt->second;

    ASSERT_EQ(second.id(), secondId);
    ASSERT_EQ(second.startJunctionId(), 2138599180);
    ASSERT_EQ(second.endJunctionId(), 2138599181);
    ASSERT_EQ(second.startZLevel(), 0);
    ASSERT_EQ(second.endZLevel(), 0);
    ASSERT_EQ(second.accessId(), RoadElement::AccessId::All);
    ASSERT_EQ(second.speedLimit(), 60);
    ASSERT_EQ(second.speedLimitF(), 60);
    ASSERT_EQ(second.speedLimitT(), 40);
    ASSERT_EQ(second.speedLimitTruckF(), 40);
    ASSERT_EQ(second.speedLimitTruckT(), 40);
    ASSERT_EQ(second.fc(), RoadElement::FunctionalClass::LocalRoad);
    ASSERT_EQ(second.fow(), RoadElement::FormOfWay::None);
    ASSERT_EQ(second.backTransportLane(), true);
    ASSERT_EQ(second.underConstruction(), false);
    ASSERT_EQ(second.trafficLightIds(), TIds({3371233456}));
    ASSERT_EQ(second.speedBumpIds(), TIds({3371233462}));
}

TEST(revision_loader, load_road_element_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const TId id = 2138599185;

    const RoadElements elements = loader->loadRoadElements({id});
    ASSERT_EQ(elements.size(), 1u);
    const RoadElement& element = elements.front();

    ASSERT_EQ(element.id(), id);
    ASSERT_EQ(element.fLanes().size(), 3u);
    ASSERT_EQ(element.tLanes().size(), 2u);
}

TEST(revision_loader, load_parking_lot_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();

    const TId id = 2138599219;

    LoaderHolder loader = makeRevisionLoader(*txn);
    const auto lots = loader->loadParkingLots({id});
    ASSERT_EQ(lots.size(), 1u);

    const ParkingLot& lot = lots.front();
    ASSERT_EQ(lot.id(), id);
}

TEST(revision_loader, load_parking_lot_by_box)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();

    constexpr double radiusMeter = 300;
    const geolib3::Point2 center = mercatorPointFromGeoCoords(35.1226088, 56.2650663);
    const geolib3::BoundingBox box(center, 2 * radiusMeter, 2 * radiusMeter);

    LoaderHolder loader = makeRevisionLoader(*txn);
    const auto lots = loader->loadParkingLots(box);
    ASSERT_EQ(lots.size(), 1u);

    const ParkingLot& lot = lots.front();
    ASSERT_EQ(lot.id(), 2138599219);
}

TEST(revision_loader, load_linear_parking_lot_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();

    const TId id = 2138599229;

    LoaderHolder loader = makeRevisionLoader(*txn);
    const auto lots = loader->loadLinearParkingLots({id});
    ASSERT_EQ(lots.size(), 1u);

    const LinearParkingLot& lot = lots.front();
    ASSERT_EQ(lot.id(), 2138599229);
    ASSERT_EQ(lot.type(), LinearParkingLot::Type::UrbanRoadnetParkingToll);
}

TEST(revision_loader, load_linear_parking_lot_by_box)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();

    constexpr double radiusMeter = 5;
    const geolib3::Point2 center = mercatorPointFromGeoCoords(35.1154360, 56.2648476);
    const geolib3::BoundingBox box(center, 2 * radiusMeter, 2 * radiusMeter);

    LoaderHolder loader = makeRevisionLoader(*txn);
    const auto lots = loader->loadLinearParkingLots(box);
    ASSERT_EQ(lots.size(), 1u);

    const LinearParkingLot& lot = lots.front();
    ASSERT_EQ(lot.id(), 2138599229);
}

TEST(revision_loader, load_mrc_regions_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();

    const TId id = 3371233458;

    LoaderHolder loader = makeRevisionLoader(*txn);
    const auto regions = loader->loadMrcRegions({id});
    ASSERT_EQ(regions.size(), 1u);

    const MrcRegion& region = regions.front();
    ASSERT_EQ(region.id(), id);
    ASSERT_EQ(region.type(), MrcRegion::Type::Task);
}

TEST(revision_loader, load_mrc_regions_by_box)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();

    constexpr double radiusMeter = 5000;
    const geolib3::Point2 center = mercatorPointFromGeoCoords(37.580209, 55.613540);
    const geolib3::BoundingBox box(center, 2 * radiusMeter, 2 * radiusMeter);

    LoaderHolder loader = makeRevisionLoader(*txn);
    const auto regions = loader->loadMrcRegions(box);
    ASSERT_EQ(regions.size(), 3u);

    TIds ids;
    std::vector<MrcRegion::Type> types;
    for (const auto& region : regions) {
        ids.push_back(region.id());
        types.push_back(region.type());
    }

    EXPECT_THAT(ids,
        ::testing::UnorderedElementsAre(3371233458, 3371233459, 3371233460)
    );

    EXPECT_THAT(types,
        ::testing::UnorderedElementsAre(MrcRegion::Type::Task, MrcRegion::Type::Residential, MrcRegion::Type::Restricted)
    );
}

TEST(revision_loader, load_all_mrc_regions)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();

    LoaderHolder loader = makeRevisionLoader(*txn);
    const auto regions = loader->loadAllMrcRegions();
    ASSERT_EQ(regions.size(), 4u);

    TIds ids;
    for (const auto& region : regions) {
        ids.push_back(region.id());
    }

    EXPECT_THAT(ids,
        ::testing::UnorderedElementsAre(3371233458, 3371233459, 3371233460, 3371233461)
    );
}


TEST(revision_loader, load_road_junciton_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const TId id = 2138599180;
    const RoadJunctions junctions = loader->loadRoadJunctions({id});
    ASSERT_EQ(junctions.size(), 1u);

    const RoadJunction& junction = junctions.front();
    ASSERT_EQ(junction.id(), id);
    ASSERT_EQ(junction.conditionIds(), TIds({2454828612}));
    ASSERT_EQ(junction.elementIds(), TIds({2138599182, 2138599185}));
    ASSERT_EQ(junction.trafficLightId(), 3371233456);
    ASSERT_EQ(junction.speedBumpId(), 3371233462);
}

TEST(revision_loader, load_road_junciton_by_box)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    constexpr double radiusMeter = 5;
    const geolib3::Point2 center = mercatorPointFromGeoCoords(35.12276974007883,56.271645431084195);
    const geolib3::BoundingBox box(center, 2 * radiusMeter, 2 * radiusMeter);

    const RoadJunctions junctions = loader->loadRoadJunctions(box);
    ASSERT_EQ(junctions.size(), 1u);

    const RoadJunction& junction = junctions.front();
    ASSERT_EQ(junction.id(), 2138599180);
}

TEST(revision_loader, load_condition_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const TId id = 2454828612;

    const Conditions conditions = loader->loadConditions({id});
    ASSERT_EQ(conditions.size(), 1u);

    const Condition& condition = conditions.front();
    ASSERT_EQ(condition.id(), id);
    ASSERT_EQ(condition.fromElementId(), 2138599182);
    ASSERT_EQ(condition.viaJunctionId(), 2138599180);
    ASSERT_EQ(condition.toElementIds(), TIds({2138599185}));
    ASSERT_EQ(condition.type(), Condition::Type::Barrier);
    ASSERT_EQ(condition.accessId(), Condition::AccessId::Vehicles);
}

TEST(revision_loader, load_traffic_light_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const TId firstId = 3371233456;
    const TId secondId = 3371233457;

    const TrafficLights trafficLights = loader->loadTrafficLights({firstId, secondId});
    const auto trafficLightById = groupById(trafficLights);
    ASSERT_EQ(trafficLightById.size(), 2u);

    const auto firstIt = trafficLightById.find(firstId);
    ASSERT_TRUE(firstIt != trafficLightById.end());
    const auto& first = firstIt->second;
    ASSERT_EQ(first.id(), firstId);
    ASSERT_EQ(first.junctionId(), 2138599180);
    ASSERT_EQ(first.elementIds(), TIds({2138599182, 2138599185}));

    const auto secondIt = trafficLightById.find(secondId);
    ASSERT_TRUE(secondIt != trafficLightById.end());
    const auto& second = secondIt->second;
    ASSERT_EQ(second.id(), secondId);
    ASSERT_EQ(second.junctionId(), 2138599179);
    ASSERT_EQ(second.elementIds(), TIds({2138599182}));
}

TEST(revision_loader, load_speed_bump_by_id)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const TId firstId = 3371233462;
    const TId secondId = 3371233463;

    const SpeedBumps speedBumps = loader->loadSpeedBumps({firstId, secondId});
    const auto speedBumpById = groupById(speedBumps);
    ASSERT_EQ(speedBumpById.size(), 2u);

    const auto firstIt = speedBumpById.find(firstId);
    ASSERT_TRUE(firstIt != speedBumpById.end());
    const auto& first = firstIt->second;
    ASSERT_EQ(first.id(), firstId);
    ASSERT_EQ(first.junctionId(), 2138599180);
    ASSERT_EQ(first.elementIds(), TIds({2138599182, 2138599185}));

    const auto secondIt = speedBumpById.find(secondId);
    ASSERT_TRUE(secondIt != speedBumpById.end());
    const auto& second = secondIt->second;
    ASSERT_EQ(second.id(), secondId);
    ASSERT_EQ(second.junctionId(), 2138599179);
    ASSERT_EQ(second.elementIds(), TIds({2138599182}));
}

TEST(revision_loader, load_address_point_with_names)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const maps::geolib3::Point2 pt1 = maps::geolib3::convertGeodeticToMercator(maps::geolib3::Point2(43.95, 56.293));
    const maps::geolib3::Point2 pt2 = maps::geolib3::convertGeodeticToMercator(maps::geolib3::Point2(43.96, 56.290));

    maps::geolib3::BoundingBox bbox(pt1, pt2);
    maps::mrc::object::AddressPointWithNames addrPointWithNames = loader->loadAddressPointWithNames(bbox);
    ASSERT_EQ(addrPointWithNames.size(), 9u);

    for (size_t i = 0; i < addrPointWithNames.size(); i++) {
        const maps::geolib3::Point2 &pt = addrPointWithNames[i].geom();
        EXPECT_TRUE(maps::geolib3::spatialRelation(pt, bbox, maps::geolib3::SpatialRelation::Within));
        EXPECT_EQ(ADDRESS_POINT_NAME_SET.count(addrPointWithNames[i].name()),  1u);
    }
}

TEST(revision_loader, load_buildings)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const maps::geolib3::Point2 pt1 = maps::geolib3::convertGeodeticToMercator(maps::geolib3::Point2(43.948644, 56.292832));
    const maps::geolib3::Point2 pt2 = maps::geolib3::convertGeodeticToMercator(maps::geolib3::Point2(43.952721, 56.290573));

    maps::geolib3::BoundingBox bbox(pt1, pt2);
    maps::mrc::object::Buildings buildings = loader->loadBuildings(bbox);
    ASSERT_EQ(buildings.size(), 13u);
}

TEST(revision_loader, load_entrances)
{
    Fixture fixture;

    auto txn = fixture.pool().masterReadOnlyTransaction();
    LoaderHolder loader = makeRevisionLoader(*txn);

    const maps::geolib3::Point2 pt1 = maps::geolib3::convertGeodeticToMercator(maps::geolib3::Point2(43.959851032, 52.226032107));
    const maps::geolib3::Point2 pt2 = maps::geolib3::convertGeodeticToMercator(maps::geolib3::Point2(44.959851032, 54.226032107));

    maps::geolib3::BoundingBox bbox(pt1, pt2);
    maps::mrc::object::Entrances entrances = loader->loadEntrances(bbox);
    ASSERT_EQ(entrances.size(), 10u);
}

} // namespace maps::mrc::object::tests
