#include "graph_matching.h"

#include "geom.h"

#include <maps/libs/geolib/include/closest_point.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/vector.h>
#include <maps/libs/log8/include/log8.h>

#include <cmath>

using maps::geolib3::Radians;
using maps::geolib3::sin;
using maps::geolib3::cos;

namespace maps::mrc::pos_improvment {

namespace {

const Seconds NEIGHBORING_RANGE(10);
const Radians MAX_ANGLE_DIFF(5 * M_PI / 180.0);
const Meters MAX_DISTANCE(40);
const Meters MAX_SHIFT(1);
const MetersPerSec MIN_SPEED(3);

size_t getIndexByTime(const GpsSegments& matchedTrack, Time time)
{
    GpsEvent eventToSearch(time, {0,0}, 0.0_m, std::nullopt, std::nullopt);
    GpsSegment segmentToSearch{eventToSearch, eventToSearch};
    auto it = std::upper_bound(
        matchedTrack.begin(), matchedTrack.end(), segmentToSearch,
        [](const GpsSegment& lhs, const GpsSegment& rhs) {
            return lhs.first.time < rhs.first.time;
        });

    return it - matchedTrack.begin();
}

} // anonymous namespace

GpsEvents getProjectionOnMatchedTrack(const GpsEvents& gpsEvents,
                                      const GpsSegments& matchedTrack,
                                      const CarGroundDirectionEvents& directionEvents)
{
    REQUIRE(!directionEvents.empty(), "no direction events");
    size_t curDirectionIndex = 0;

    GpsEvents matchedGpsEvents;

    size_t carMovingEvents = 0;
    size_t matchedMovingEvents = 0;

    for (const GpsEvent& gpsEvent : gpsEvents) {
        while(directionEvents[curDirectionIndex].time < gpsEvent.time
              && curDirectionIndex < directionEvents.size() - 1) {
            curDirectionIndex++;
        }
        if (gpsEvent.speed > MIN_SPEED) {
            carMovingEvents++;
        }
        double bestScore = 1000000;
        int bestIndex = -1;
        int leftIndex = getIndexByTime(matchedTrack,
                                       gpsEvent.time - NEIGHBORING_RANGE);
        int rightIndex = std::min(getIndexByTime(matchedTrack,
                                                 gpsEvent.time + NEIGHBORING_RANGE),
                                  matchedTrack.size() - 1);

        for (int i = leftIndex; i <= rightIndex; i++) {
            const GpsEvent& segmentStart = matchedTrack[i].first;
            const GpsEvent& segmentEnd = matchedTrack[i].second;
            geolib3::Segment2 segment(segmentStart.mercatorPos,
                                      segmentEnd.mercatorPos);

            Radians angleDiff = angleBetween(
                geolib3::Direction2(directionEvents[curDirectionIndex].angle),
                geolib3::Direction2(segment));
            if (angleDiff > MAX_ANGLE_DIFF) {
                continue;
            }

            Meters distanceToSegment = mercatorDistanceToMeters(
                geolib3::distance(gpsEvent.mercatorPos, segment),
                gpsEvent.mercatorPos);

            if (distanceToSegment > MAX_DISTANCE) {
                continue;
            }

            // 0.01 rad difference is equal to 4 meters difference
            double score = distanceToSegment.value() + angleDiff.value() * 400;
            if (score < bestScore) {
                bestScore = score;
                bestIndex = i;
            }
        }

        if (bestIndex != -1) {
            if (gpsEvent.speed > MIN_SPEED) {
                matchedMovingEvents++;
            }

            geolib3::Segment2 segment(matchedTrack[bestIndex].first.mercatorPos,
                                      matchedTrack[bestIndex].second.mercatorPos);
            geolib3::Point2 proj = geolib3::closestPoint(segment, gpsEvent.mercatorPos);
            geolib3::Vector2 shift = gpsEvent.mercatorPos - proj;
            Meters distanceToSegment = mercatorDistanceToMeters(
                geolib3::length(shift),
                gpsEvent.mercatorPos);

            if (distanceToSegment > MAX_SHIFT) {
                shift = shift / distanceToSegment.value() * MAX_SHIFT.value();
            }

            matchedGpsEvents.emplace_back(
                gpsEvent.time,
                proj + shift,
                gpsEvent.accuracy,
                gpsEvent.speed,
                gpsEvent.direction);
        }
    }

    INFO() << "getProjectionOnMatchedTrack "
        << "matchedMovingEvents / carMovingEvents = "
        << double(matchedMovingEvents) / carMovingEvents;

    INFO() << matchedMovingEvents << " good matched points from "
           << carMovingEvents << " points";

    REQUIRE(double(matchedMovingEvents) / carMovingEvents > 0.85,
            "can't project track on matched track");

    return matchedGpsEvents;
}

} // namespace maps::mrc::pos_improvment
