#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/include/position_improvment.h>

#include "azimuth.h"
#include "rodrigues.h"
#include "gyroscope.h"
#include "gravity.h"
#include "gps.h"
#include "graph_matching.h"
#include "phone_orientation.h"
#include "position.h"
#include "speed.h"
#include "utils.h"

#include <maps/libs/log8/include/log8.h>


using maps::geolib3::cos;
using maps::geolib3::sin;

namespace maps::mrc::pos_improvment {

namespace {

// Doubles events frequency using linear interpolation
template<typename Vector3EventType, typename Vector3Type>
std::vector<Vector3EventType> increaseFrequency(const std::vector<Vector3EventType>& events)
{
    REQUIRE(!events.empty(), "empty vector of events");
    // Event.values is a vector of derivative of some Value1
    // For gyroscope it is a rotation speed, for accelerometer it is
    // an acceleration (a speed of a velocity changing)
    std::vector<Vector3EventType> newEvents = {events[0]};
    newEvents.reserve(events.size() * 2 - 1);

    for (size_t i = 1; i < events.size(); i++) {
        geolib3::Vector3 middleValues = (events[i - 1].values() + events[i].values()) / 2.0;
        newEvents.emplace_back(events[i - 1].time + (events[i].time-events[i - 1].time) / 2.0,
                               static_cast<Vector3Type&>(middleValues));
        newEvents.push_back(events[i]);
    }
    return newEvents;
}

// Returns vector in the provided coordinate system
// The new coordinate system has the same zero point
geolib3::Vector3 vectorInNewCoordinateSystem(
    const geolib3::Vector3& vector,
    const UnitVector3& newCsX,
    const UnitVector3& newCsY,
    const UnitVector3& newCsZ) {
    return geolib3::Vector3 {
        geolib3::innerProduct(vector, newCsX),
        geolib3::innerProduct(vector, newCsY),
        geolib3::innerProduct(vector, newCsZ)};
}

ImprovedGpsEvent constructImprovedGpsEvent(
    const GpsEvent& gpsEvent,
    const geolib3::Point2& odometerGpsPosition,
    const UnitVector3& gravity,
    const UnitVector3& carHorizFrontVector)
{
    UnitVector3 carRightVector(
        geolib3::crossProduct(gravity, carHorizFrontVector));
    UnitVector3 upVector(-gravity);

    // north direction in phone coordinate system
    // (car front direction rotated azimuth_degrees around gravity vector)
    UnitVector3 north(carHorizFrontVector * sin(*gpsEvent.direction)
                      - carRightVector * cos(*gpsEvent.direction));
    // east direction in phone coordinate system
    UnitVector3 east(geolib3::crossProduct(gravity, north));

    UnitVector3 cameraFrontDirection(
        vectorInNewCoordinateSystem({0, 0, -1},
                                    east,
                                    north,
                                    upVector));
    UnitVector3 cameraRightDirection(
        vectorInNewCoordinateSystem({1, 0, 0},
                                    east,
                                    north,
                                    upVector));
    UnitVector3 cameraUpDirection(
        vectorInNewCoordinateSystem({0, 1, 0},
                                    east,
                                    north,
                                    upVector));
    return ImprovedGpsEvent(gpsEvent,
                            odometerGpsPosition,
                            cameraFrontDirection,
                            cameraRightDirection,
                            cameraUpDirection);
}

ImprovedGpsEvents constructImprovedGpsEvents(
    const PreciseTrack& preciseTrack,
    GravityEvents gravityEvents,
    CarHorizontalFrontVecEvents horizontalFrontVecEvents)
{
    TrackEventPtrs events;
    events.reserve(preciseTrack.gpsEvents.size()
                   + gravityEvents.size()
                   + horizontalFrontVecEvents.size());
    insert(events, preciseTrack.gpsEvents);
    insert(events, gravityEvents);
    insert(events, horizontalFrontVecEvents);

    ImprovedGpsEvents improvedGpsEvents;
    improvedGpsEvents.reserve(preciseTrack.gpsEvents.size());

    UnitVector3 gravity = gravityEvents[0].values();
    UnitVector3 carHorizFrontVector = horizontalFrontVecEvents[0].values();
    size_t gpsEventsIndex = 0;

    for (auto trackEvent : events) {
        if (trackEvent->type == EventType::Gravity) {
            auto& event = *static_cast<GravityEvent const *>(trackEvent);
            gravity = event.values();
        }
        if (trackEvent->type == EventType::CarHorizontalFrontVector) {
            auto& event = *static_cast<CarHorizontalFrontVecEvent const *>(trackEvent);
            carHorizFrontVector = event.values();
        }

        if (trackEvent->type == EventType::Gps) {
            auto& event = *static_cast<GpsEvent const *>(trackEvent);
            improvedGpsEvents.push_back(
                constructImprovedGpsEvent(event,
                                          preciseTrack.odometerTrack[gpsEventsIndex],
                                          gravity,
                                          carHorizFrontVector));
            gpsEventsIndex++;
        }
    }
    return improvedGpsEvents;
}

// makes accEvents and gyroEvents time more precise
std::pair<AccelerometerEvents, GyroscopeEvents> fixSensorsTimeLag(
    const GpsEvents& gpsEvents,
    const UnitVector3& carUpVector,
    AccelerometerEvents accEvents,
    GyroscopeEvents gyroEvents)
{
    UnitVector3 carFrontVector = findCarFrontDirection(
        accEvents, gyroEvents, gpsEvents, carUpVector);

    auto [gravityEvents, horizontalFrontVecEvents]
        = findGravityAndHorizontalFrontVector(
            accEvents, gyroEvents, gpsEvents, carUpVector, carFrontVector);

    CarGroundDirectionEvents carGroundDirectionEvents = calculateAzimuthEvents(
        gpsEvents, gravityEvents, gyroEvents);

    return fixSensorsAndGpsTimeLag(
        gpsEvents, accEvents, gyroEvents, carGroundDirectionEvents);
}

// using car direction makes speed more precise
GpsEvents improveSpeedUsingDirection(
    GpsEvents gpsEvents,
    const UnitVector3& carUpVector,
    const UnitVector3& carFrontVector,
    const AccelerometerEvents& accEvents,
    const GyroscopeEvents& gyroEvents)
{
    auto [gravityEvents, horizontalFrontVecEvents]
        = findGravityAndHorizontalFrontVector(
            accEvents, gyroEvents, gpsEvents, carUpVector, carFrontVector);

    CarGroundDirectionEvents carGroundDirectionEvents = calculateAzimuthEvents(
        gpsEvents, gravityEvents, gyroEvents);

    return use2PointsGpsSpeed(gpsEvents, carGroundDirectionEvents);
}

} // anonymous namespace

ImprovedGpsEvent::ImprovedGpsEvent(const chrono::TimePoint& time,
                                   const geolib3::Point2& geodeticPos,
                                   const geolib3::Point2& odometerMercatorPos,
                                   geolib3::Heading carHeading,
                                   const std::vector<double>& cameraRodrigues,
                                   Meters accuracy,
                                   MetersPerSec speed)
    : GpsEvent(createGpsEvent(time,
                              geodeticPos,
                              accuracy,
                              carHeading,
                              speed))
    , cameraFrontDirection_(geolib3::Vector3(1, 0, 0)) // will be evaluated
    , cameraRightDirection_(geolib3::Vector3(1, 0, 0)) // in the
    , cameraUpDirection_(geolib3::Vector3(1, 0, 0))    // constructor
    , geodeticPos_(geodeticPos)
    , odometerMercatorPos_(odometerMercatorPos)
{
    auto cameraOrientation = fromRodrigues(cameraRodrigues);
    cameraRightDirection_ = cameraOrientation.cameraRightDirection;
    cameraFrontDirection_ = cameraOrientation.cameraFrontDirection;
    cameraUpDirection_ = cameraOrientation.cameraUpDirection;
}

std::vector<double> ImprovedGpsEvent::cameraRodrigues() const
{
    return toRodrigues({cameraRightDirection_,
                        cameraFrontDirection_,
                        cameraUpDirection_});
}

ImprovedGpsEvents calculatePreciseGpsTrack(GpsEvents gpsEvents,
                                           AccelerometerEvents accEvents,
                                           GyroscopeEvents gyroEvents,
                                           std::optional<GpsSegments> matchedTrack)
{
    gyroEvents = calibrateGyroscope(std::move(gyroEvents));
    UnitVector3 carUpVector = findCarUpDirection(accEvents);

    gpsEvents = use2PointsGpsAzimuth(std::move(gpsEvents));
    gpsEvents = smoothAzimuth(std::move(gpsEvents), gyroEvents, carUpVector);
    gpsEvents = use2PointsGpsSpeed(std::move(gpsEvents));

    // Increase sensor events frequency because the current algorithm
    // doesn't do interpolation between events
    accEvents = increaseFrequency<AccelerometerEvent, AccelerationVector>(std::move(accEvents));
    gyroEvents = increaseFrequency<GyroscopeEvent, RotationSpeedVector>(std::move(gyroEvents));

    std::tie(accEvents, gyroEvents) = fixSensorsTimeLag(
        gpsEvents, carUpVector, std::move(accEvents), std::move(gyroEvents));

    UnitVector3 carFrontVector = findCarFrontDirection(
        accEvents, gyroEvents, gpsEvents, carUpVector);

    gpsEvents = improveSpeedUsingDirection(
        std::move(gpsEvents), carUpVector, carFrontVector, accEvents, gyroEvents);

    auto [gravityEvents, horizontalFrontVecEvents]
        = findGravityAndHorizontalFrontVector(
            accEvents, gyroEvents, gpsEvents, carUpVector, carFrontVector);

    CarGroundDirectionEvents carGroundDirectionEvents = calculateAzimuthEvents(
        gpsEvents, gravityEvents, gyroEvents);
    CarGroundSpeedEvents carGroundSpeedEvents = calculateSpeedEvents(
        gpsEvents, horizontalFrontVecEvents, accEvents);

    PreciseTrack preciseGpsTrack;

    if (matchedTrack) {
        GpsEvents matchedTrackGoodPoints = getProjectionOnMatchedTrack(
            gpsEvents,
            *matchedTrack,
            carGroundDirectionEvents);

        preciseGpsTrack = calculatePreciseTrack(
            matchedTrackGoodPoints, carGroundDirectionEvents, carGroundSpeedEvents);
    } else {
        preciseGpsTrack = calculatePreciseTrack(
            gpsEvents, carGroundDirectionEvents, carGroundSpeedEvents);
    }

    return constructImprovedGpsEvents(preciseGpsTrack,
                                      gravityEvents,
                                      horizontalFrontVecEvents);
}

ImprovedGpsEvent getInterpolatedPositionByTime(
    const ImprovedGpsEvents& track,
    chrono::TimePoint time)
{
    REQUIRE(track.size() > 1, "empty track was provided");
    REQUIRE(time >= track.front().timestamp() && time <= track.back().timestamp(),
            "time is outside the track time length");
    Time targetTime = std::chrono::time_point_cast<Seconds>(time);
    auto it = std::upper_bound(
        track.begin(), track.end(), targetTime,
        [](Time lhs, const ImprovedGpsEvent& rhs) {
            return lhs < rhs.time;
        });
    // time is inside the track time, but upper bound returns value in [1, track.size()]
    int index = std::max((long)1, std::min(it - track.begin(), (long)(track.size() - 1)));

    if (track[index].time == track[index-1].time) {
        return track[index];
    }

    double d = (targetTime - track[index-1].time)
             / (track[index].time - track[index-1].time);

    double mercatorX = lerp(track[index].mercatorPosition().x(),
                            track[index-1].mercatorPosition().x(),
                            d);
    double mercatorY = lerp(track[index].mercatorPosition().y(),
                            track[index-1].mercatorPosition().y(),
                            d);
    double odoMercatorX = lerp(track[index].odometerMercatorPosition().x(),
                               track[index-1].odometerMercatorPosition().x(),
                               d);
    double odoMercatorY = lerp(track[index].odometerMercatorPosition().y(),
                               track[index-1].odometerMercatorPosition().y(),
                               d);

    double speed = lerp(track[index].speedMetersPerSec(),
                        track[index-1].speedMetersPerSec(),
                        d);

    auto azimuth1 = geolib3::Direction2(track[index - 1].carHeading());
    auto azimuth2 = geolib3::Direction2(track[index].carHeading());
    auto azimuth = geolib3::Direction2(
        azimuth1.radians()
        + d * geolib3::orientedAngleBetween(azimuth1, azimuth2));

    UnitVector3 cameraFrontDirection
        = UnitVector3(lerp<geolib3::Vector3>(
                              track[index].cameraFrontDirection(),
                              track[index-1].cameraFrontDirection(),
                              d));
    UnitVector3 cameraRightDirection
        = UnitVector3(lerp<geolib3::Vector3>(
                              track[index].cameraRightDirection(),
                              track[index-1].cameraRightDirection(),
                              d));
    UnitVector3 cameraUpDirection
        = UnitVector3(geolib3::crossProduct(cameraRightDirection,
                                            cameraFrontDirection));
    return ImprovedGpsEvent(
        createGpsEvent(time,
                       geolib3::mercator2GeoPoint({mercatorX, mercatorY}),
                       Meters(0.0), // accuracy is not used
                       azimuth.heading(),
                       MetersPerSec(speed)),
        {odoMercatorX, odoMercatorY},
        cameraFrontDirection,
        cameraRightDirection,
        cameraUpDirection);
}


GpsEvent createGpsEvent(chrono::TimePoint time,
                        geolib3::Point2 geodeticPos,
                        Meters accuracy,
                        std::optional<geolib3::Heading> heading,
                        std::optional<MetersPerSec> speed)
{
    std::optional<geolib3::Radians> direction;
    if (heading) {
        direction = geolib3::Direction2(*heading).radians();
    }
    return GpsEvent(std::chrono::time_point_cast<Seconds>(time),
                    geolib3::geoPoint2Mercator(geodeticPos),
                    accuracy,
                    speed,
                    direction);
}

AccelerometerEvent createAccelerometerEvent(chrono::TimePoint time,
                                            double x,
                                            double y,
                                            double z)
{
    return AccelerometerEvent(std::chrono::time_point_cast<Seconds>(time),
                              {x, y, z});
}

GyroscopeEvent createGyroscopeEvent(chrono::TimePoint time,
                                    double x,
                                    double y,
                                    double z)
{
    return GyroscopeEvent(std::chrono::time_point_cast<Seconds>(time),
                          {x, y, z});
}

} // namespace maps::mrc::pos_improvment
