#pragma once

#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/include/events.h>

#include <vector>
#include <algorithm>

namespace maps::mrc::pos_improvment {

struct MostPopularDirection {
    UnitVector3 vector;
    double confidence;
};

// Finds the most popular direction.
MostPopularDirection findMostPopularDirection(
    const std::vector<geolib3::Vector3>& vectors,
    size_t histogramDimension);

// For each event calculates average value of the neighboring events in
// the provided filter time range.
// Returns events with averaged values.
// The frequency of the result events is almost the same as in input events.
//
// This function works when each event value represents a speed of
// something(rotating speed, speed of velocity changing).
// To calculate the average speed of several events we need to calculate the
// overal distance and the overal time and divide the distance by the time
template<EventType eventType, typename Vector3Type>
TVector3Events<eventType, Vector3Type> getFilteredValues(
    const TVector3Events<eventType, Vector3Type>& events, Seconds filterTime)
{
    TVector3Events<eventType, Vector3Type> filteredEvents;
    if (events.size() < 2) {
        return filteredEvents;
    }
    size_t index1 = 0;
    size_t index2 = 0;
    geolib3::Vector3 curDistance(0, 0, 0); // if the event value is a speed
    while(index2 < events.size() - 2
          && events[index2 + 1].time - events[index1].time < filterTime)
    {
        index2++;
        Seconds dt = events[index2].time - events[index2-1].time;
        curDistance = curDistance + events[index2].values() * dt.count();
    }
    if (events[index2 + 1].time - events[index1].time < filterTime) {
        // too big filter time
        return filteredEvents;
    }
    for (index2 = index2 + 1; index2 < events.size(); index2++) {
        Seconds dt = events[index2].time - events[index2-1].time;
        curDistance = curDistance + events[index2].values() * dt.count();

        auto newEvent = events[index2];
        dt = (events[index2].time - events[index1].time);
        if (dt.count() > 0) {
            auto filteredValues = curDistance / dt.count();
            newEvent.values() = static_cast<Vector3Type&>(filteredValues);
        }
        newEvent.time = events[index1].time + dt / 2;
        filteredEvents.push_back(newEvent);

        while(events[index2].time - events[index1].time > filterTime) {
            index1++;
            dt = events[index1].time - events[index1 - 1].time;
            curDistance = curDistance - events[index1].values() * dt.count();
        }
    }
    return filteredEvents;
}

template <typename Value>
Value lerp(Value end, Value start, double fraction) {
    return start + fraction * (end - start);
}

} // namespace maps::mrc::pos_improvment
