#pragma once

#include "events.h"

#include <maps/libs/chrono/include/time_point.h>

namespace maps{
namespace mrc {
namespace pos_improvment {

GpsEvent createGpsEvent(chrono::TimePoint time,
                        geolib3::Point2 geodeticPos,
                        Meters accuracy,
                        std::optional<geolib3::Heading> heading,
                        std::optional<MetersPerSec> speed);

// 3d coordinates are in the Android phone coordinate system
// https://developer.android.com/images/axis_device.png
// https://developer.android.com/guide/topics/sensors/sensors_overview

// accelerometer value should be m/s/s
AccelerometerEvent createAccelerometerEvent(chrono::TimePoint time,
                                            double x,
                                            double y,
                                            double z);

// gyroscope value should be rad/s
GyroscopeEvent createGyroscopeEvent(chrono::TimePoint time,
                                    double x,
                                    double y,
                                    double z);


// See description of odometerMercatorPos, cameraFrontDirection,
// cameraRightDirection, cameraUpDirection in the corresponding class methods
class ImprovedGpsEvent : public GpsEvent {
public:
    ImprovedGpsEvent(const GpsEvent& gpsEvent,
                     const geolib3::Point2& odometerMercatorPos,
                     const UnitVector3& cameraFrontDirection,
                     const UnitVector3& cameraRightDirection,
                     const UnitVector3& cameraUpDirection)
            : GpsEvent(gpsEvent)
            , cameraFrontDirection_(cameraFrontDirection)
            , cameraRightDirection_(cameraRightDirection)
            , cameraUpDirection_(cameraUpDirection)
            , geodeticPos_(geolib3::mercator2GeoPoint(mercatorPos))
            , odometerMercatorPos_(odometerMercatorPos)
    {
        REQUIRE(gpsEvent.speed, "can't create ImprovedGpsEvent without speed");
        REQUIRE(gpsEvent.direction, "can't create ImprovedGpsEvent without direction");
    }

    ImprovedGpsEvent(const chrono::TimePoint& time,
                     const geolib3::Point2& geodeticPos,
                     const geolib3::Point2& odometerMercatorPos,
                     geolib3::Heading carHeading,
                     const std::vector<double>& cameraRodrigues, // camera orientation as 3 numbers
                     Meters accuracy = 0.0_m,
                     MetersPerSec speed = 0.0_mps);

    chrono::TimePoint timestamp() const {
        return std::chrono::time_point_cast<chrono::TimePoint::duration>(time);
    }

    geolib3::Point2 geodeticPosition() const { return geodeticPos_; }
    geolib3::Point2 mercatorPosition() const { return mercatorPos; }

    // this position is not accurate, but x and y mercator distances
    // between neighboring points are very precise
    geolib3::Point2 odometerMercatorPosition() const { return odometerMercatorPos_; }

    double speedMetersPerSec() const { return speed->value(); }
    geolib3::Heading carHeading() const { return geolib3::Direction2(*direction).heading(); }

    // phone orientation in Earth ground coordinate system, where:
    // x -> East
    // y -> North
    // z -> Up

    // phone main camera direction
    const UnitVector3& cameraFrontDirection() const { return cameraFrontDirection_; }
    // phone right side in usual portrait mode
    const UnitVector3& cameraRightDirection() const { return cameraRightDirection_; }
    // phone up side in usual portrait mode
    const UnitVector3& cameraUpDirection() const { return cameraUpDirection_; }

    // returns camera orientation as 3 numbers
    std::vector<double> cameraRodrigues() const;

private:
    UnitVector3 cameraFrontDirection_;
    UnitVector3 cameraRightDirection_;
    UnitVector3 cameraUpDirection_;
    geolib3::Point2 geodeticPos_;
    geolib3::Point2 odometerMercatorPos_;
};

using ImprovedGpsEvents = std::vector<ImprovedGpsEvent>;

// Returns high rate gps track with precise azimuth and better coordinates
// @param matchedTrack - matched to graph track. If provided, the
// result track will be softly attached to the graph
ImprovedGpsEvents calculatePreciseGpsTrack(
    GpsEvents gpsEvents,
    AccelerometerEvents accEvents,
    GyroscopeEvents gyroEvents,
    std::optional<GpsSegments> matchedTrack = std::nullopt);

ImprovedGpsEvent getInterpolatedPositionByTime(
    const ImprovedGpsEvents& track,
    chrono::TimePoint time);

} // namespace pos_improvment
} // namespace mrc
} // namespace maps
