#include "test_utils.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/impl/gyroscope.h>

namespace maps::mrc::pos_improvment::tests {

TEST(gyroscope_calibration_tests, test_calibrated_gyroscope)
{
    GyroscopeEvents events;
    events.emplace_back(1.0_ts, RotationSpeedVector{1, 2, 3});
    events.emplace_back(2.0_ts, RotationSpeedVector{0, 0, 0});
    events.emplace_back(3.0_ts, RotationSpeedVector{0, 0, 0});
    EXPECT_THAT(calibrateGyroscope(events), testing::Eq(events));
}

TEST(gyroscope_calibration_tests, test_calibrated_gyroscope2)
{
    GyroscopeEvents events;
    events.emplace_back(10.0_ts, RotationSpeedVector{0, 0, 0});
    events.emplace_back(20.0_ts, RotationSpeedVector{0, 0, 0});
    events.emplace_back(30.0_ts, RotationSpeedVector{-5, 10, -35});
    events.emplace_back(40.0_ts, RotationSpeedVector{0, 0, 0});
    events.emplace_back(50.0_ts, RotationSpeedVector{10, -25, -60});
    EXPECT_THAT(calibrateGyroscope(events), testing::Eq(events));
}

TEST(gyroscope_calibration_tests, test_uncalibrated_gyroscope)
{
    GyroscopeEvents events;
    events.emplace_back(1.0_ts, RotationSpeedVector{11, 22, 33});
    events.emplace_back(2.0_ts, RotationSpeedVector{1, 2, 3});
    events.emplace_back(3.0_ts, RotationSpeedVector{1, 2, 3});

    GyroscopeEvents calibratedEvents;
    calibratedEvents.emplace_back(1.0_ts, RotationSpeedVector{10, 20, 30});
    calibratedEvents.emplace_back(2.0_ts, RotationSpeedVector{0, 0, 0});
    calibratedEvents.emplace_back(3.0_ts, RotationSpeedVector{0, 0, 0});

    EXPECT_THAT(calibrateGyroscope(events), testing::Eq(calibratedEvents));
}

TEST(gyroscope_calibration_tests, test_uncalibrated_gyroscope2)
{
    GyroscopeEvents events;
    events.emplace_back(1.0_ts, RotationSpeedVector{-1, -2, -3});
    events.emplace_back(2.0_ts, RotationSpeedVector{11, 22, 33});
    events.emplace_back(3.0_ts, RotationSpeedVector{-1, -2, -3});

    GyroscopeEvents calibratedEvents;
    calibratedEvents.emplace_back(1.0_ts, RotationSpeedVector{0, 0, 0});
    calibratedEvents.emplace_back(2.0_ts, RotationSpeedVector{12, 24, 36});
    calibratedEvents.emplace_back(3.0_ts, RotationSpeedVector{0, 0, 0});

    EXPECT_THAT(calibrateGyroscope(events), testing::Eq(calibratedEvents));
}

TEST(gyroscope_calibration_tests, test_fix_sensors_lag)
{
    GpsEvents gpsEvents {
        {0.5_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.0},
        {0.6_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.0},
        {0.7_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.0},
        {0.8_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.0},
        {0.9_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.0},
        {1.0_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.0},

        {1.1_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.1},
        {1.2_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.2},
        {1.3_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.3},
        {1.4_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.4},
        {1.5_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.5},

        {1.6_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.5},
        {1.7_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.5},
        {1.8_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.5},
        {1.9_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.5},
        {2.0_ts, {0, 0}, 1.0_m, 5.0_mps, PI * 0.5}
    };

    // the shift should be calculated using gps and direction events
    // the shift is 0.4_s
    CarGroundDirectionEvents directionEvents {
        {EventType::CarGroundDirection, 0.6_ts, PI * 0.0},
        {EventType::CarGroundDirection, 0.7_ts, PI * 0.1}, // should be 1.1_ts
        {EventType::CarGroundDirection, 0.8_ts, PI * 0.2},
        {EventType::CarGroundDirection, 0.9_ts, PI * 0.3},
        {EventType::CarGroundDirection, 1.0_ts, PI * 0.4},
        {EventType::CarGroundDirection, 1.1_ts, PI * 0.5},
        {EventType::CarGroundDirection, 1.2_ts, PI * 0.5}
    };

    GyroscopeEvents gyroEvents {
        {1.0_ts, RotationSpeedVector{1, 2, 3}},
        {2.0_ts, RotationSpeedVector{4, 5, 6}}
    };

    AccelerometerEvents accEvents {
        {3.0_ts, AccelerationVector{7, 8, 9}},
        {4.0_ts, AccelerationVector{10, 11, 12}}
    };

    auto [newAccEvents, newGyroEvents] = fixSensorsAndGpsTimeLag(
        gpsEvents, accEvents, gyroEvents, directionEvents);

    EXPECT_EQ(newGyroEvents.size(), gyroEvents.size());
    EXPECT_EQ(newGyroEvents[0].values(), gyroEvents[0].values());
    EXPECT_EQ(newGyroEvents[1].values(), gyroEvents[1].values());
    EXPECT_NEAR(newGyroEvents[0].time.time_since_epoch().count(), 1.4, 0.1);
    EXPECT_NEAR(newGyroEvents[1].time.time_since_epoch().count(), 2.4, 0.1);

    EXPECT_EQ(newAccEvents.size(), accEvents.size());
    EXPECT_EQ(newAccEvents[0].values(), accEvents[0].values());
    EXPECT_EQ(newAccEvents[1].values(), accEvents[1].values());
    EXPECT_NEAR(newAccEvents[0].time.time_since_epoch().count(), 3.4, 0.1);
    EXPECT_NEAR(newAccEvents[1].time.time_since_epoch().count(), 4.4, 0.1);
}

} // namespace maps::mrc::pos_improvment::tests
