#include "test_utils.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/include/position_improvment.h>

using maps::geolib3::Point2;
using maps::geolib3::Vector2;
using maps::geolib3::Vector3;

namespace maps::mrc::pos_improvment::tests {

    // at this point 2 mercator meters = 1 real meter
    const Point2 P0 = geolib3::geoPoint2Mercator(Point2{40, 60});

TEST(position_improvment_tests, test_position_improvment)
{
    // car goes to north
    // the first 2 seconds speed is increasing
    // on the third second speed is not changing
    // the last 2 seconds speed is decreasing

    AccelerometerEvents accEvents {
        // speed is increasing
        {1.0_ts, AccelerationVector{9.8, 5, 0}},
        {1.33_ts, AccelerationVector{9.8, 5, 0}},
        {1.66_ts, AccelerationVector{9.8, 5, 0}},
        {2.0_ts, AccelerationVector{9.8, 5, 0}},
        {2.33_ts, AccelerationVector{9.8, 5, 0}},
        {2.33_ts, AccelerationVector{9.8, 5, 0}},
        {2.66_ts, AccelerationVector{9.8, 5, 0}},
        {3.0_ts, AccelerationVector{9.8, 5, 0}},
        {3.33_ts, AccelerationVector{9.8, 5, 0}},
        {3.6599_ts, AccelerationVector{9.8, 5, 0}},
        {3.999999_ts, AccelerationVector{9.8, 5, 0}},

        // speed is decreasing
        {4.000001_ts, AccelerationVector{9.8, -5, 0}},
        {4.33_ts, AccelerationVector{9.8, -5, 0}},
        {4.66_ts, AccelerationVector{9.8, -5, 0}},
        {5.0_ts, AccelerationVector{9.8, -5, 0}},
        {5.33_ts, AccelerationVector{9.8, -5, 0}},
        {5.66_ts, AccelerationVector{9.8, -5, 0}},
        {5.999999_ts, AccelerationVector{9.8, -5, 0}},
        {6.000001_ts, AccelerationVector{9.95, 0, 0}}
    };
    // unused accelerometer events for good gravity detection
    for (int i = 7; i < 29; i++) {
        accEvents.emplace_back(Time(Seconds(i)) , AccelerationVector{9.65, 0, 0});
        accEvents.emplace_back(Time(Seconds(i)) , AccelerationVector{9.95, 0, 0});
    }

    GyroscopeEvents gyroEvents {
    // no rotation
        {1.0_ts, RotationSpeedVector{0, 0, 0}},
        {1.33_ts, RotationSpeedVector{0, 0, 0}},
        {1.66_ts, RotationSpeedVector{0, 0, 0}},
        {2.0_ts, RotationSpeedVector{0, 0, 0}},
        {2.33_ts, RotationSpeedVector{0, 0, 0}},
        {2.66_ts, RotationSpeedVector{0, 0, 0}},
        {3.0_ts, RotationSpeedVector{0, 0, 0}},
        {3.33_ts, RotationSpeedVector{0, 0, 0}},
        {3.66_ts, RotationSpeedVector{0, 0, 0}},
        {4.0_ts, RotationSpeedVector{0, 0, 0}},
        {4.33_ts, RotationSpeedVector{0, 0, 0}},
        {4.66_ts, RotationSpeedVector{0, 0, 0}},
        {5.0_ts, RotationSpeedVector{0, 0, 0}},
        {5.33_ts, RotationSpeedVector{0, 0, 0}},
        {5.66_ts, RotationSpeedVector{0, 0, 0}},
        {6.0_ts, RotationSpeedVector{0, 0, 0}}
    };

    // There are 4 marked errors in gps events. They should be ignored.
    GpsEvents gpsEvents {
        // gpsY[i] = gpsY[i-1] + distance = gpsY[i-1] + avg_mercator_speed * 1_sec
        // = gpsY[i-1] + (gpsY[i].speed + gpsY[i-1].speed) / 2 * 2_meter_to_mercactor
        // = gpsY[i-1] + (gpsY[i].speed + gpsY[i-1].speed)
        {0.9999999_ts, P0 + Vector2{0, 0}, 0.0_m, 5.0_mps, PI/2},
        {2.0_ts, P0 + Vector2{0, 15}, 0.0_m, 125.0_mps/*10.0_mps*/, PI/2},
        {3.0_ts, P0 + Vector2{0, 40}, 0.0_m, 15.0_mps, PI/2},
        {4.0_ts, P0 + Vector2{0, 75/*70*/}, 0.0_m, 20.0_mps, PI/2},
        {5.0_ts, P0 + Vector2{0, 110}, 0.0_m, 15.0_mps, PI/2},
        {6.0000001_ts, P0 + Vector2{0*456/*0*/, 135}, 0.0_m, 10.0_mps, 0_rad/*PI/2*/}
    };

    ImprovedGpsEvents improvedGpsEvents
        = calculatePreciseGpsTrack(gpsEvents, accEvents, gyroEvents);

    EXPECT_EQ(improvedGpsEvents.size(), gyroEvents.size() * 2 - 1);

    EXPECT_EQ(improvedGpsEvents[0].timestamp(), chrono::TimePoint(std::chrono::seconds(1)));
    EXPECT_NEAR(improvedGpsEvents[0].speedMetersPerSec(), 5.0, 0.01);
    EXPECT_NEAR(improvedGpsEvents[0].mercatorPosition().y(), P0.y(), 1.0);
    EXPECT_NEAR(improvedGpsEvents[0].odometerMercatorPosition().y(), P0.y(), 1.0);

    EXPECT_EQ(improvedGpsEvents.back().timestamp(), chrono::TimePoint(std::chrono::seconds(6)));
    EXPECT_NEAR(improvedGpsEvents.back().speedMetersPerSec(), 10.0, 0.01);
    EXPECT_NEAR(improvedGpsEvents.back().mercatorPosition().y(), P0.y() + 135, 1.0);
    EXPECT_NEAR(improvedGpsEvents.back().odometerMercatorPosition().y(), P0.y() + 135, 1.0);

    const auto& middleGpsPoint = improvedGpsEvents[improvedGpsEvents.size() / 2];

    EXPECT_TRUE(middleGpsPoint.timestamp() >
                chrono::TimePoint(std::chrono::milliseconds(3490)));
    EXPECT_TRUE(middleGpsPoint.timestamp() <
                chrono::TimePoint(std::chrono::milliseconds(3510)));
    EXPECT_NEAR(middleGpsPoint.speedMetersPerSec(), 17.5, 0.1);
    EXPECT_NEAR(middleGpsPoint.mercatorPosition().y(), P0.y() + 55, 1.0);
    EXPECT_NEAR(middleGpsPoint.odometerMercatorPosition().y(), P0.y() + 55, 1.0);

    for (const auto& improvedGpsEvent : improvedGpsEvents) {
        EXPECT_NEAR(improvedGpsEvent.carHeading().value(), 0, 0.01);
        EXPECT_NEAR(improvedGpsEvent.mercatorPosition().x(), P0.x(), 0.00001);
        EXPECT_NEAR(improvedGpsEvent.odometerMercatorPosition().x(), P0.x(), 0.00001);

        // https://developer.android.com/images/axis_device.png
        // Car goes to north, phone is in landscape mode (button is on
        // the right side), camera is directed to east

        // camera is directed to east
        EXPECT_NEAR(improvedGpsEvent.cameraFrontDirection().x(), 1, 0.002);
        EXPECT_NEAR(improvedGpsEvent.cameraFrontDirection().y(), 0, 0.002);
        EXPECT_NEAR(improvedGpsEvent.cameraFrontDirection().z(), 0, 0.002);

        // phone right side (of portrait mode) is directed up
        EXPECT_NEAR(improvedGpsEvent.cameraRightDirection().x(), 0, 0.002);
        EXPECT_NEAR(improvedGpsEvent.cameraRightDirection().y(), 0, 0.002);
        EXPECT_NEAR(improvedGpsEvent.cameraRightDirection().z(), 1, 0.002);

        // phone up side (of portrait mode) is directed to north
        EXPECT_NEAR(improvedGpsEvent.cameraUpDirection().x(), 0, 0.002);
        EXPECT_NEAR(improvedGpsEvent.cameraUpDirection().y(), 1, 0.002);
        EXPECT_NEAR(improvedGpsEvent.cameraUpDirection().z(), 0, 0.002);
    }
}

TEST(position_improvment_tests, test_interpolate_position)
{
    ImprovedGpsEvents events {
        {GpsEvent{1.0_ts, P0 + Vector2{10, 20}, 0.0_m, 10.0_mps, PI * 0.1},
         P0 + Vector2{110, 120},
         UnitVector3({0, 1, 0}),
         UnitVector3({1, 0, 0}),
         UnitVector3({0, 0, 1})},
        {GpsEvent{2.0_ts, P0 + Vector2{20, 40}, 0.0_m, 20.0_mps, PI * 0.2},
         P0 + Vector2{120, 140},
         UnitVector3({1, 0, 0}),
         UnitVector3({0, -1, 0}),
         UnitVector3({0, 0, 1})},
        {GpsEvent{3.0_ts, P0 + Vector2{30, 60}, 0.0_m, 30.0_mps, PI * 0.3},
         P0 + Vector2{130, 160},
         UnitVector3({0, -1, 0}),
         UnitVector3({-1, 0, 0}),
         UnitVector3({0, 0, 1})}
    };

    ImprovedGpsEvent event = getInterpolatedPositionByTime(
        events, chrono::TimePoint(std::chrono::milliseconds(1000)));

    EXPECT_EQ(event.timestamp(), chrono::TimePoint(std::chrono::milliseconds(1000)));
    EXPECT_NEAR(event.speedMetersPerSec(), 10.0, 0.0001);
    EXPECT_NEAR(geolib3::Direction2(event.carHeading()).radians().value(), M_PI * 0.1, 0.00001);

    EXPECT_NEAR(event.mercatorPosition().x(), P0.x() + 10, 0.0001);
    EXPECT_NEAR(event.mercatorPosition().y(), P0.y() + 20, 0.0001);
    EXPECT_NEAR(event.odometerMercatorPosition().x(), P0.x() + 110, 0.0001);
    EXPECT_NEAR(event.odometerMercatorPosition().y(), P0.y() + 120, 0.0001);

    EXPECT_NEAR(event.cameraFrontDirection().x(), 0.0, 0.0001);
    EXPECT_NEAR(event.cameraFrontDirection().y(), 1.0, 0.0001);
    EXPECT_NEAR(event.cameraFrontDirection().z(), 0.0, 0.0001);

    EXPECT_NEAR(event.cameraRightDirection().x(), 1.0, 0.0001);
    EXPECT_NEAR(event.cameraRightDirection().y(), 0.0, 0.0001);
    EXPECT_NEAR(event.cameraRightDirection().z(), 0.0, 0.0001);

    EXPECT_NEAR(event.cameraUpDirection().x(), 0.0, 0.0001);
    EXPECT_NEAR(event.cameraUpDirection().y(), 0.0, 0.0001);
    EXPECT_NEAR(event.cameraUpDirection().z(), 1.0, 0.0001);

    event = getInterpolatedPositionByTime(
        events, chrono::TimePoint(std::chrono::milliseconds(2500)));

    EXPECT_EQ(event.timestamp(), chrono::TimePoint(std::chrono::milliseconds(2500)));
    EXPECT_NEAR(event.speedMetersPerSec(), 25.0, 0.0001);
    EXPECT_NEAR(geolib3::Direction2(event.carHeading()).radians().value(), M_PI * 0.25, 0.00001);
    EXPECT_NEAR(event.mercatorPosition().x(), P0.x() + 25, 0.0001);
    EXPECT_NEAR(event.mercatorPosition().y(), P0.y() + 50, 0.0001);
    EXPECT_NEAR(event.odometerMercatorPosition().x(), P0.x() + 125, 0.0001);
    EXPECT_NEAR(event.odometerMercatorPosition().y(), P0.y() + 150, 0.0001);

    EXPECT_NEAR(event.cameraFrontDirection().x(), 0.707106, 0.0001);
    EXPECT_NEAR(event.cameraFrontDirection().y(), -0.707106, 0.0001);
    EXPECT_NEAR(event.cameraFrontDirection().z(), 0.0, 0.0001);

    EXPECT_NEAR(event.cameraRightDirection().x(), -0.707106, 0.0001);
    EXPECT_NEAR(event.cameraRightDirection().y(), -0.707106, 0.0001);
    EXPECT_NEAR(event.cameraRightDirection().z(), -0.0, 0.0001);

    EXPECT_NEAR(event.cameraUpDirection().x(), 0.0, 0.0001);
    EXPECT_NEAR(event.cameraUpDirection().y(), 0.0, 0.0001);
    EXPECT_NEAR(event.cameraUpDirection().z(), 1.0, 0.0001);
}

} // namespace maps::mrc::pos_improvment::tests
