#include "test_utils.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/wikimap/mapspro/services/mrc/libs/position_improvment/impl/position.h>

using maps::geolib3::Point2;
using maps::geolib3::Vector2;
using maps::geolib3::Vector3;

namespace maps::mrc::pos_improvment::tests {

    // at this point 2 mercator meters = 1 real meter
    const Point2 P0 = geolib3::geoPoint2Mercator(Point2{40, 60});

TEST(position_tests, test_position_perfect_accuracy)
{
    // gps speed and direction should be ignored
    GpsEvents gpsEvents {
        {1.0_ts, P0 + Vector2{0, 0}, 1.0_m, 5.0_mps, PI},
        {2.0_ts, P0 + Vector2{0, 40}, 1.0_m, 5.0_mps, PI},
        {3.0_ts, P0 + Vector2{0, 80}, 1.0_m, 5.0_mps, PI}
    };
    CarGroundDirectionEvents directionEvents {
        {EventType::CarGroundDirection, 1.5_ts, PI / 2},
        {EventType::CarGroundDirection, 2.5_ts, PI / 2}
    };
    CarGroundSpeedEvents speedEvents {
        {EventType::CarGroundSpeed, 1.499999_ts, 20.0_mps},
        {EventType::CarGroundSpeed, 2.499999_ts, 20.0_mps}
    };

    GpsEvents newGpsEvents
        = calculatePreciseTrack(gpsEvents, directionEvents, speedEvents).gpsEvents;

    EXPECT_EQ(newGpsEvents.size(), 2u);

    EXPECT_EQ(newGpsEvents[0].time, 1.5_ts);
    EXPECT_NEAR(newGpsEvents[0].speed->value(), 20.0, 0.001);
    EXPECT_NEAR(newGpsEvents[0].direction->value(), M_PI / 2, 0.001);
    EXPECT_NEAR(newGpsEvents[0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(newGpsEvents[0].mercatorPos.y(), P0.y() + 20, 0.001);

    EXPECT_EQ(newGpsEvents[1].time, 2.5_ts);
    EXPECT_NEAR(newGpsEvents[1].speed->value(), 20.0, 0.001);
    EXPECT_NEAR(newGpsEvents[1].direction->value(), M_PI / 2, 0.001);
    EXPECT_NEAR(newGpsEvents[1].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(newGpsEvents[1].mercatorPos.y(), P0.y() + 60, 0.001);
}

TEST(position_tests, test_position_error_in_gps)
{
    // the same as test_position_perfect_accuracy but with error in gps

    // gps speed and direction should be ignored
    GpsEvents gpsEvents {
        {1.0_ts, P0 + Vector2{0, 0}, 1.0_m, 5.0_mps, PI},
        {2.0_ts, P0 + Vector2{0, 40}, 1.0_m, 5.0_mps, PI},
        {3.0_ts, P0 + Vector2{0, 70}, 1.0_m, 5.0_mps, PI} // should be {0, 80}
    };
    CarGroundDirectionEvents directionEvents {
        {EventType::CarGroundDirection, 1.5_ts, PI / 2},
        {EventType::CarGroundDirection, 2.5_ts, PI / 2}
    };
    CarGroundSpeedEvents speedEvents {
        {EventType::CarGroundSpeed, 1.499999_ts, 20.0_mps},
        {EventType::CarGroundSpeed, 2.499999_ts, 20.0_mps}
    };

    GpsEvents newGpsEvents
        = calculatePreciseTrack(gpsEvents, directionEvents, speedEvents).gpsEvents;

    EXPECT_EQ(newGpsEvents.size(), 2u);

    EXPECT_EQ(newGpsEvents[0].time, 1.5_ts);
    EXPECT_NEAR(newGpsEvents[0].speed->value(), 20.0, 0.001);
    EXPECT_NEAR(newGpsEvents[0].direction->value(), M_PI / 2, 0.001);
    EXPECT_NEAR(newGpsEvents[0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(newGpsEvents[0].mercatorPos.y(), P0.y() + 20, 0.001);

    EXPECT_EQ(newGpsEvents[1].time, 2.5_ts);
    EXPECT_NEAR(newGpsEvents[1].speed->value(), 20.0, 0.001);
    EXPECT_NEAR(newGpsEvents[1].direction->value(), M_PI / 2, 0.001);
    EXPECT_NEAR(newGpsEvents[1].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(newGpsEvents[1].mercatorPos.y(), P0.y() + 60, 0.001);
}

TEST(position_tests, test_position_turn)
{
    // gps speed and direction should be ignored
    GpsEvents gpsEvents {
        {1.0_ts, P0 + Vector2{0, 0}, 1.0_m, 5.0_mps, PI},
        {2.0_ts, P0 + Vector2{35, 0}, 1.0_m, 5.0_mps, PI}, // should be {40, 0}
        {3.0_ts, P0 + Vector2{40, 70}, 1.0_m, 5.0_mps, PI}
    };
    CarGroundDirectionEvents directionEvents {
        {EventType::CarGroundDirection, 1.5_ts, 0.0_rad},
        {EventType::CarGroundDirection, 2.5_ts, PI / 2}
    };
    CarGroundSpeedEvents speedEvents {
        {EventType::CarGroundSpeed, 1.499999_ts, 20.0_mps},
        {EventType::CarGroundSpeed, 2.500001_ts, 50.0_mps}
    };

    GpsEvents newGpsEvents
        = calculatePreciseTrack(gpsEvents, directionEvents, speedEvents).gpsEvents;

    EXPECT_EQ(newGpsEvents.size(), 2u);

    EXPECT_EQ(newGpsEvents[0].time, 1.5_ts);
    EXPECT_NEAR(newGpsEvents[0].speed->value(), 20.0, 0.001);
    EXPECT_NEAR(newGpsEvents[0].direction->value(), 0, 0.001);
    EXPECT_NEAR(newGpsEvents[0].mercatorPos.x(), P0.x() + 20, 0.001);
    EXPECT_NEAR(newGpsEvents[0].mercatorPos.y(), P0.y(), 0.001);

    EXPECT_EQ(newGpsEvents[1].time, 2.5_ts);
    EXPECT_NEAR(newGpsEvents[1].speed->value(), 20.0, 0.001);
    EXPECT_NEAR(newGpsEvents[1].direction->value(), M_PI / 2, 0.001);
    EXPECT_NEAR(newGpsEvents[1].mercatorPos.x(), P0.x() + 40, 0.001);
    EXPECT_NEAR(newGpsEvents[1].mercatorPos.y(), P0.y() + 20, 0.001);
}

} // namespace maps::mrc::pos_improvment::tests
