#include "common.h"
#include "feature.h"
#include "maps/libs/geolib/include/multipolygon.h"
#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/libs/sql_chemistry/include/filter.h>
#include <maps/wikimap/mapspro/services/mrc/libs/object/include/mock_loader.h>
#include <maps/wikimap/mapspro/services/mrc/libs/privacy/impl/caching_region_privacy.h>
#include <maps/wikimap/mapspro/services/mrc/libs/privacy/tests/mocks/geo_id_provider.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/point.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>
#include <yandex/maps/mrc/unittest/unittest_config.h>

#include <chrono>
#include <memory>
#include <string>

namespace maps::mrc::privacy::tests {

using namespace std::chrono_literals;

using ::testing::_;
using ::testing::Return;
using ::testing::Matcher;

namespace {

class ForbiddenAreaGuard {
public:
    ForbiddenAreaGuard(
            object::MockLoader& objectLoader,
            const geolib3::Polygon2& geoPolygon)
    {

        objectLoader.add(
            object::MrcRegions{
                object::MrcRegion(
                    object::RevisionID{1, 1},
                    geolib3::convertGeodeticToMercator(geoPolygon)
                ).type(object::MrcRegion::Type::Restricted)
            }
        );
    }
};

struct Fixture:  public testing::Test {

    inline auto makeCachingRegionPrivacy(GeoIdProviderPtr geoIdProvider)
    {
        return CachingRegionPrivacy(
            {
                {PUBLIC_REGION_ID, db::FeaturePrivacy::Public},
                {RESTRICTED_REGION_ID, db::FeaturePrivacy::Restricted},
                {SECRET_REGION_ID, db::FeaturePrivacy::Secret},
            },
            objectLoader,
            std::move(geoIdProvider)
        );
    }


    static const db::TId PUBLIC_REGION_ID = 1;
    static const db::TId RESTRICTED_REGION_ID = 2;
    static const db::TId SECRET_REGION_ID = 3;
    static const db::TId UNCLASSIFIED_REGION_ID = 4;

    object::MockLoader objectLoader;
};

geolib3::MultiPolygon2 makePolygon(const geolib3::PointsVector& points)
{
    return geolib3::MultiPolygon2(
        {geolib3::Polygon2(points)}
    );
}

MATCHER_P(BoxEq_, box, "") {
    return box.lowerCorner() == arg.lowerCorner() && box.upperCorner() == arg.upperCorner();
}

inline auto BoxEq(const geolib3::BoundingBox& box)
{
    return Matcher<const geolib3::BoundingBox&>(BoxEq_(box));
}

} // namespace

const geolib3::Point2 GEO_POINT(5.5, 5.5);

const geolib3::BoundingBox GEO_BOX(
    geolib3::Point2(5.0, 5.0),
    geolib3::Point2(6.0, 6.0)
);

const geolib3::MultiPolygon2 PARTIALLY_INTERSECTED_GEOM =
    makePolygon({
            {3.5, 3.5},
            {3.5, 4.5},
            {4.5, 4.5},
            {4.5, 3.5},
            {3.5, 3.5},
        });

const geolib3::MultiPolygon2 COVERING_GEOM =
    makePolygon({
            {4.0, 4.0},
            {4.0, 7.0},
            {7.0, 7.0},
            {7.0, 4.0},
            {4.0, 4.0},
        });

const geolib3::MultiPolygon2 CONTAINED_GEOM =
    makePolygon({
            {5.3, 5.3},
            {5.7, 5.3},
            {5.7, 5.7},
            {5.3, 5.7},
            {5.3, 5.3},
        });

TEST_F(Fixture, eval_feature_privacy_should)
{
    struct TestData {
        db::TIds geoIds;
        db::FeaturePrivacy privacy;
    };

    std::vector<TestData> testData {
        {{PUBLIC_REGION_ID}, db::FeaturePrivacy::Public},
        {{PUBLIC_REGION_ID, SECRET_REGION_ID}, db::FeaturePrivacy::Secret},
        {{PUBLIC_REGION_ID, RESTRICTED_REGION_ID}, db::FeaturePrivacy::Restricted},
        {{RESTRICTED_REGION_ID, PUBLIC_REGION_ID}, db::FeaturePrivacy::Restricted},
        {{RESTRICTED_REGION_ID, SECRET_REGION_ID}, db::FeaturePrivacy::Secret},
        {{PUBLIC_REGION_ID, RESTRICTED_REGION_ID, SECRET_REGION_ID}, db::FeaturePrivacy::Secret},
        {{UNCLASSIFIED_REGION_ID}, db::FeaturePrivacy::Restricted},
        {{PUBLIC_REGION_ID, UNCLASSIFIED_REGION_ID}, db::FeaturePrivacy::Public},
        {{SECRET_REGION_ID, UNCLASSIFIED_REGION_ID}, db::FeaturePrivacy::Secret},
    };

    for (const auto& [geoIds, expectedPrivacy] : testData) {
        auto geoIdProvider = makeMockGeoIdProvider();

        EXPECT_CALL(*geoIdProvider, load(GEO_POINT))
            .WillOnce(Return(geoIds));

        auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
        EXPECT_EQ(
            regionPrivacy.evalFeaturePrivacy(GEO_POINT),
            expectedPrivacy);
    }
}

TEST_F(Fixture, eval_min_max_feature_privacy_default_case)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(BoxEq(GEO_BOX)))
        .WillRepeatedly(Return(db::TIds{UNCLASSIFIED_REGION_ID}));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Restricted);
    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Restricted);
}

TEST_F(Fixture, eval_min_max_feature_privacy_public_region_fully_covers)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(BoxEq(GEO_BOX)))
        .WillRepeatedly(Return(db::TIds{PUBLIC_REGION_ID, UNCLASSIFIED_REGION_ID}));

    EXPECT_CALL(*geoIdProvider, geomById(PUBLIC_REGION_ID))
        .WillRepeatedly(Return(COVERING_GEOM));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Public);

    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Public);
}

TEST_F(Fixture, eval_min_max_feature_privacy_public_region_partially_covers)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(BoxEq(GEO_BOX)))
        .WillRepeatedly(Return(db::TIds{PUBLIC_REGION_ID, UNCLASSIFIED_REGION_ID}));

    EXPECT_CALL(*geoIdProvider, geomById(PUBLIC_REGION_ID))
        .WillRepeatedly(Return(PARTIALLY_INTERSECTED_GEOM));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Public);

    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Restricted);
}


TEST_F(Fixture, eval_min_max_feature_privacy_secret_region_fully_covers)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(BoxEq(GEO_BOX)))
        .WillRepeatedly(Return(db::TIds{SECRET_REGION_ID, UNCLASSIFIED_REGION_ID}));

    EXPECT_CALL(*geoIdProvider, geomById(SECRET_REGION_ID))
        .WillRepeatedly(Return(COVERING_GEOM));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Secret);

    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Secret);
}

TEST_F(Fixture, eval_min_max_feature_privacy_secret_region_partially_covers)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(BoxEq(GEO_BOX)))
        .WillRepeatedly(Return(db::TIds{SECRET_REGION_ID, UNCLASSIFIED_REGION_ID}));

    EXPECT_CALL(*geoIdProvider, geomById(SECRET_REGION_ID))
        .WillRepeatedly(Return(PARTIALLY_INTERSECTED_GEOM));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Restricted);

    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Secret);
}

TEST_F(Fixture, eval_min_max_feature_privacy_public_secret_region_fully_covers)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(BoxEq(GEO_BOX)))
        .WillRepeatedly(Return(db::TIds{SECRET_REGION_ID, PUBLIC_REGION_ID, UNCLASSIFIED_REGION_ID}));

    EXPECT_CALL(*geoIdProvider, geomById(SECRET_REGION_ID))
        .WillRepeatedly(Return(COVERING_GEOM));
    EXPECT_CALL(*geoIdProvider, geomById(PUBLIC_REGION_ID))
        .WillRepeatedly(Return(COVERING_GEOM));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Secret);

    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Secret);
}

TEST_F(Fixture, eval_min_max_feature_privacy_public_secret_region_partially_covers)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(BoxEq(GEO_BOX)))
        .WillRepeatedly(Return(db::TIds{SECRET_REGION_ID, PUBLIC_REGION_ID, UNCLASSIFIED_REGION_ID}));

    EXPECT_CALL(*geoIdProvider, geomById(SECRET_REGION_ID))
        .WillRepeatedly(Return(PARTIALLY_INTERSECTED_GEOM));
    EXPECT_CALL(*geoIdProvider, geomById(PUBLIC_REGION_ID))
        .WillRepeatedly(Return(PARTIALLY_INTERSECTED_GEOM));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Public);

    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_BOX),
        db::FeaturePrivacy::Secret);
}


TEST_F(Fixture, eval_min_max_feature_privacy_for_secret_area)
{
    ForbiddenAreaGuard guard{objectLoader, COVERING_GEOM.polygonAt(0)};

    auto regionPrivacy = makeCachingRegionPrivacy(makeMockGeoIdProvider());
    EXPECT_EQ(regionPrivacy.evalMinFeaturePrivacy(
                  CONTAINED_GEOM.boundingBox()),
              db::FeaturePrivacy::Secret);
    EXPECT_EQ(regionPrivacy.evalMaxFeaturePrivacy(
                  CONTAINED_GEOM.boundingBox()),
              db::FeaturePrivacy::Secret);

    EXPECT_EQ(regionPrivacy.evalMinFeaturePrivacy(
                  COVERING_GEOM.boundingBox()),
              db::FeaturePrivacy::Secret);

    EXPECT_EQ(regionPrivacy.evalMaxFeaturePrivacy(
                  COVERING_GEOM.boundingBox()),
              db::FeaturePrivacy::Secret);
}


TEST_F(Fixture, eval_min_max_feature_privacy_degenerate_box)
{
    auto geoIdProvider = makeMockGeoIdProvider();

    EXPECT_CALL(*geoIdProvider, load(GEO_POINT))
        .WillRepeatedly(Return(db::TIds{PUBLIC_REGION_ID}));

    auto regionPrivacy = makeCachingRegionPrivacy(std::move(geoIdProvider));
    EXPECT_EQ(
        regionPrivacy.evalMinFeaturePrivacy(GEO_POINT.boundingBox()),
        db::FeaturePrivacy::Public);

    EXPECT_EQ(
        regionPrivacy.evalMaxFeaturePrivacy(GEO_POINT.boundingBox()),
        db::FeaturePrivacy::Public);
}

} // namespace maps::mrc::browser::tests
