#include <maps/wikimap/mapspro/services/mrc/libs/sensors_feature_positioner/include/sensors_feature_positioner.h>

#include <string>

namespace pos_im = maps::mrc::pos_improvment;

using Meters = pos_im::Meters;
using MetersPerSec = pos_im::MetersPerSec;

namespace maps::mrc::sensors_feature_positioner {

namespace {

pos_im::GpsEvents toGpsEvents(const db::TrackPoints& trackPoints) {
    constexpr Meters DEFAULT_ACCURACY(10);
    pos_im::GpsEvents gpsEvents;
    for (const auto& trackPoint : trackPoints) {
        std::optional<MetersPerSec> speed;
        if (trackPoint.speedMetersPerSec()) {
            speed = MetersPerSec(*trackPoint.speedMetersPerSec());
        }
        Meters accuracy
            = trackPoint.accuracyMeters()
            ? Meters(*trackPoint.accuracyMeters())
            : DEFAULT_ACCURACY;
        gpsEvents.push_back(pos_im::createGpsEvent(
                                trackPoint.timestamp(),
                                trackPoint.geodeticPos(),
                                accuracy,
                                trackPoint.heading(),
                                speed));
    }
    return gpsEvents;
}

pos_im::GpsSegments toGpsSegments(const mrc::adapters::TrackSegments& matchedTrack) {
    constexpr Meters DEFAULT_ACCURACY(10);
    pos_im::GpsSegments matchedTrackSegments;
    matchedTrackSegments.reserve(matchedTrack.size());
    for (const auto& segment : matchedTrack) {
        matchedTrackSegments.push_back({
                pos_im::createGpsEvent(
                    segment.startTime,
                    segment.segment.start(),
                    DEFAULT_ACCURACY,
                    std::nullopt,
                    std::nullopt),
                pos_im::createGpsEvent(
                    segment.endTime,
                    segment.segment.end(),
                    DEFAULT_ACCURACY,
                    std::nullopt,
                    std::nullopt)});
    }
    return matchedTrackSegments;
}

} // anonymous namespace

SensorsFeaturePositioner::SensorsFeaturePositioner(
    const adapters::Matcher& graphMatcher,
    const db::TrackPoints& trackPoints,
    const SensorEvents& sensorEvents)
{
    auto matchedTrack = toGpsSegments(graphMatcher.match(trackPoints));
    REQUIRE(matchedTrack.size(), "matcher returns empty track");

    improvedGpsTrack_ = pos_improvment::calculatePreciseGpsTrack(
        toGpsEvents(trackPoints),
        sensorEvents.accEvents,
        sensorEvents.gyroEvents,
        matchedTrack);
}

std::optional<pos_im::ImprovedGpsEvent> SensorsFeaturePositioner::getPositionByTime(
    chrono::TimePoint time) const
{
    if (improvedGpsTrack_.empty()
        || time < improvedGpsTrack_.front().timestamp()
        || time > improvedGpsTrack_.back().timestamp())
    {
        return std::nullopt;
    }

    return pos_im::getInterpolatedPositionByTime(improvedGpsTrack_, time);
}

} // namespace maps::mrc::sensors_feature_positioner
