#include "camera_azimuth_calibration.h"

#include "object_rays_intersection.h"
#include "utils.h"

#include <maps/libs/geolib/include/distance.h>

namespace maps::mrc::sensors_feature_positioner {

namespace {

const double SIMILAR_CAMERAS_DISTANCE_THRESHOLD = 0.5;

// Returns rotated ray
Ray rotateCameraAzimuth(Ray ray, double angle)
{
    geolib3::Vector2 groundDirection(ray.directionToObject.x(),
                                     ray.directionToObject.y());
    geolib3::Vector2 newGroundDirection = geolib3::rotate(groundDirection, angle);
    ray.directionToObject = pos_improvment::UnitVector3(geolib3::Vector3(
        newGroundDirection.x(),
        newGroundDirection.y(),
        ray.directionToObject.z()));
    return ray;
}

// Each group contains all the rays of all the photos from one camera
// position
// @param rays - rays of one ride sorted by time
std::vector<Rays> groupRaysByCameraPosition(const Rays& rays)
{
    if (rays.empty()) {
        return {};
    }
    std::vector<Rays> groupedRays;
    groupedRays.push_back({rays[0]});
    for (size_t i = 1; i < rays.size(); i++) {
        if (metersBetweenCameras(groupedRays.back()[0], rays[i])
            < SIMILAR_CAMERAS_DISTANCE_THRESHOLD) {
            groupedRays.back().push_back(rays[i]);
        } else {
            groupedRays.push_back({rays[i]});
        }
    }
    return groupedRays;
}

// Tryes to intersect ray with otherRays and return the intersection
// with the best accuracy (if good intersections exist)
std::optional<ObjectIntersection> getBestIntersection(
    const Ray& ray,
    const Rays& otherRays,
    double angleShift)
{
    double bestAccuracy = std::numeric_limits<double>::max();
    std::optional<ObjectIntersection> bestIntersection;

    Ray rotatedRay = rotateCameraAzimuth(ray, angleShift);
    IdGenerator idGenerator;

    for (const Ray& r : otherRays) {
        auto intersection = createObjectFrom2Photos(rotatedRay,
                                                    rotateCameraAzimuth(r, angleShift),
                                                    idGenerator);
        if (!intersection) {
            continue;
        }
        double distanceToObjectError = std::abs(
            intersection->metersFromCamera1 // distance beetween the camera and the intersection
            - rotatedRay.metersToObject); // expected distance from the camera
                                        // to the object calculated using
                                        // bbox size

        // Calculating intersection accuracy:
        // 1 meter distance between rays is equal to 0.1rad angle error
        // and is equal to 0.25m distanceToObject calculating error
        double accuracy = intersection->errorMeters
            + intersection->errorRadians.value() * 10
            + distanceToObjectError / 4;
        if (accuracy < bestAccuracy) {
            bestAccuracy = accuracy;
            bestIntersection = intersection;
        }
    }
    return bestIntersection;
}

// Returns min distance in meters from point to the nearest otherPoint.
// the point and the otherPoints should be mercator 3d points
double distanceToNearestPoint(geolib3::Point3 point,
                              const std::vector<geolib3::Point3>& otherPoints,
                              geolib3::Point2 approximateMercatorPos)
{
    double minDistance = std::numeric_limits<double>::max();
    for (const geolib3::Point3 p : otherPoints) {
        double distance = geolib3::toMeters(geolib3::length(point - p),
                                            approximateMercatorPos);
        minDistance = std::min(minDistance, distance);
    }
    return minDistance;
}

// Try different azimuth shifts and select the azimuth shift with the
// best rays intersections
std::vector<Rays> calibrateRaysAzimuth(std::vector<Rays>& groupedRays)
{
    std::vector<double> angles;
    for (double angleShift = -0.5; angleShift < 0.5; angleShift += 0.01) {
        angles.push_back(angleShift);
    }

    // scores for each photo and for each possible angle(scores[photoIndex][shiftIndex])
    // score ≈ number_of_intersection - avg_intersections_error
    // + number_of_close_intersection_points
    std::vector<std::vector<double>> scores(groupedRays.size());

    for (size_t photoIndex = 0; photoIndex < groupedRays.size(); photoIndex++) {
        const Rays& curPhotoRays = groupedRays[photoIndex];
        for (double angleShift : angles) {
            double curAngleScore = 0;
            for (const Ray& curPhotoRay : curPhotoRays) {
                std::vector<geolib3::Point3> intersectionPoints;
                for (size_t anotherPhotoIndex = photoIndex + 1;
                     anotherPhotoIndex < groupedRays.size();
                     anotherPhotoIndex++)
                {
                    const Rays& anotherPhotoRays = groupedRays[anotherPhotoIndex];
                    if (metersBetweenCameras(curPhotoRay, anotherPhotoRays[0])
                        > MAX_METERS_BETWEEN_CAMERAS)
                    {
                        break;
                    }
                    std::optional<ObjectIntersection> objectIntersection
                        = getBestIntersection(curPhotoRay, anotherPhotoRays, angleShift);
                    if (!objectIntersection) {
                        continue;
                    }
                    // if the distance between two rays > 2 meters, we
                    // don't consider them as an intersections
                    curAngleScore += std::max(0.0, 2 - objectIntersection->errorMeters);
                    if (intersectionPoints.size()) {
                        double minDistanceToAnotherIntersection
                            = distanceToNearestPoint(
                                objectIntersection->odoMercatorPos,
                                intersectionPoints,
                                curPhotoRay.cameraPos.mercatorPosition());
                        // if the distance between two intersections >
                        // 3 meters, we don't consider them
                        // as close intersections
                        curAngleScore += std::max(0.0, 3 - minDistanceToAnotherIntersection);
                    }
                    intersectionPoints.push_back(objectIntersection->odoMercatorPos);
                }
            }
            // the score for the photo is the avg score of all the
            // rays of the photo
            scores[photoIndex].push_back(curAngleScore / curPhotoRays.size());
        }
    }

    const int FRAME = 20; // for each photo we look at 40 neighboring
                          // photos to evaluate the best azimuth shift.
                          // (photos without objects are ignored)
    for (size_t photoIndex = 0; photoIndex < groupedRays.size(); photoIndex++) {
        double bestAngle = 0;
        double bestScore = 0;

        for (size_t angleIndex = 0; angleIndex < angles.size(); angleIndex++) {
            double score = 0;
            for (int j = std::max(int(0), int(photoIndex) - FRAME);
                 j < std::min(int(groupedRays.size()), int(photoIndex) + FRAME + 1);
                 j++)
            {
                score += scores[j][angleIndex];
            }
            if (score > bestScore) {
                bestScore = score;
                bestAngle = angles[angleIndex];
            }
        }
        const double MIN_RELIABLE_SCORE = 5;
        if (bestScore > MIN_RELIABLE_SCORE) {
            for (auto& ray : groupedRays[photoIndex]) {
                ray = rotateCameraAzimuth(ray, bestAngle);
            }
        }
    }
    return groupedRays;
}

} // anonymous namespace

Rays calibrateRaysAzimuth(Rays rays)
{
    auto raysGroups = groupRaysByCameraPosition(rays);
    raysGroups = calibrateRaysAzimuth(raysGroups);
    rays.clear();
    for (const auto& raysGroup : raysGroups) {
        rays.insert(rays.end(), raysGroup.begin(), raysGroup.end());
    }
    return rays;
}

} // namespace maps::mrc::sensors_feature_positioner
