#include <maps/wikimap/mapspro/services/mrc/libs/sensors_feature_positioner/include/object_positioner.h>

#include "object_intersections_positioner.h"
#include "object_single_ray_positioner.h"
#include "utils.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/distance.h>

#include <vector>

namespace maps::mrc::sensors_feature_positioner {

namespace {

ResultObject toResultObject(const ObjectIntersections& objectIntersections) {
    ResultObject resultObject;
    std::unordered_set<size_t> savedRayIds;

    for (const ObjectIntersection& objectIntersection: objectIntersections) {
        if (savedRayIds.insert(objectIntersection.ray1.rayId).second) {
            resultObject.rays.push_back(objectIntersection.ray1);
        }
        if (savedRayIds.insert(objectIntersection.ray2.rayId).second) {
            resultObject.rays.push_back(objectIntersection.ray2);
        }
    }

    resultObject.mercatorPos = aggregatedMercatorPosition(objectIntersections);
    return resultObject;
}

ResultObject toResultObject(const ObjectPositions& objectPositions) {
    ResultObject resultObject;
    resultObject.rays.reserve(objectPositions.size());

    geolib3::Point3 avgMercatorPos(0, 0, 0);
    for (const ObjectPosition& objectPosition: objectPositions) {
        avgMercatorPos +=
            (objectPosition.mercatorPos - geolib3::Point3{0, 0, 0})
            / objectPositions.size();
        resultObject.rays.push_back(objectPosition.ray);
    }

    resultObject.mercatorPos = geolib3::Point2{avgMercatorPos.x(), avgMercatorPos.y()};
    return resultObject;
}

} // anonymous namespace

std::vector<Rays> separateBySourceIdAndSort(Rays rays)
{
    if (rays.empty()) {
        return {};
    }

    std::sort(rays.begin(), rays.end(),
              [](const Ray& lhs, const Ray& rhs) {
                  if (lhs.sourceId != rhs.sourceId) {
                      return lhs.sourceId < rhs.sourceId;
                  } else {
                      return lhs.cameraPos.timestamp() < rhs.cameraPos.timestamp();
                  }
              });

    std::vector<Rays> resultRaysGroups;
    resultRaysGroups.push_back({rays[0]});

    for (size_t i = 1; i < rays.size(); i++) {
        if (rays[i].sourceId != rays[i - 1].sourceId) {
            resultRaysGroups.push_back({rays[i]});
        } else {
            resultRaysGroups.back().push_back(rays[i]);
        }
    }

    return resultRaysGroups;
}

ResultObjects calculateObjectsPositionsFromOneRide(Rays rays)
{
    // locate objects using rays intersection
    std::vector<ObjectIntersections> objectsByIntersections
        = locateObjectsUsingIntersections(rays);

    //  Locate all objects using bbox size
    std::vector<ObjectPositions> objectsByBboxSize = calculateObjects(rays);

    // Merge result objects
    ResultObjects resultObjects;
    std::unordered_map<RayId, ResultObject*> rayToObject;
    resultObjects.reserve(objectsByIntersections.size() + objectsByBboxSize.size());

    // add all objects groups located by intersections
    for (const auto& objectIntersections : objectsByIntersections) {
        resultObjects.push_back(toResultObject(objectIntersections));
        for (const auto& ray : resultObjects.back().rays) {
            rayToObject[ray.rayId] = &resultObjects.back();
        }
    }

    // for objects located by bboxes add only groups that doesn't
    // contain object rays used in intersections
    for (const auto& objectByBboxPositions : objectsByBboxSize) {
        // check if this object is connected with some intersection object
        ResultObject* alreadyAddedObject = nullptr;
        for (const auto& objectPosition : objectByBboxPositions) {
            if (rayToObject.count(objectPosition.ray.rayId)) {
                alreadyAddedObject = rayToObject[objectPosition.ray.rayId];
                break;
            }
        }

        if (alreadyAddedObject) {
            // add rays to the existing intersection object group
            for (const auto& objectPosition : objectByBboxPositions) {
                if (!rayToObject.count(objectPosition.ray.rayId)) {
                    alreadyAddedObject->rays.push_back(objectPosition.ray);
                }
            }
        } else {
            // add new object group
            resultObjects.push_back(toResultObject(objectByBboxPositions));
        }
    }
    return resultObjects;
}

ResultObjects calculateObjectsPositions(const Rays& rays) {
    std::vector<Rays> rides = separateBySourceIdAndSort(rays);
    ResultObjects resultObjects;
    for (const auto& ride : rides) {
        ResultObjects objects = calculateObjectsPositionsFromOneRide(ride);
        resultObjects.insert(resultObjects.end(), objects.begin(), objects.end());
    }

    return resultObjects;
}

geolib3::Heading calculateObjectHeading(const ResultObject& resultObject)
{
    REQUIRE(!resultObject.rays.empty(), "ResultObject without rays");

    geolib3::Vector2 avgCarDirection(0, 0);
    for (const auto& ray : resultObject.rays) {
        geolib3::Vector2 carDirection
            = geolib3::Direction2(ray.cameraPos.carHeading()).vector();
        avgCarDirection += carDirection;
    }

    return geolib3::reverse(geolib3::Direction2(avgCarDirection).heading());
}

} // namespace maps::mrc::sensors_feature_positioner
