#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/wikimap/mapspro/services/mrc/libs/sensors_feature_positioner/object_positioner/points_clustering.h>

namespace maps::mrc::sensors_feature_positioner::tests {

using maps::geolib3::Vector2;
using maps::geolib3::Point2;

namespace {

// clusters don't have order, lets sort them for easier testing
std::vector<Cluster> sortClusters(std::vector<Cluster> clusters) {
    for (auto& cluster: clusters) {
        std::sort(cluster.begin(), cluster.end());
    }
    std::sort(clusters.begin(), clusters.end(),
              [&](const Cluster& lhs, const Cluster& rhs) {
                  return lhs[0] < rhs[0];
              });
    return clusters;
}

} // anonymous namespace

    // at this point 2 mercator meters = 1 real meter
    const Point2 P0 = geolib3::geoPoint2Mercator(Point2{40, 60});

TEST(points_clustering_tests, test_1_point)
{
    Points points{
        {1, Point{1, P0 + Vector2{0, 0}, {}, {}}}};

    auto clusters = getClusters(points, 10);
    clusters = sortClusters(std::move(clusters));
    EXPECT_EQ(clusters.size(), 1u);
    EXPECT_THAT(clusters[0], testing::UnorderedElementsAre(1));
}

TEST(points_clustering_tests, test_2_separate_points)
{
    Points points{
        {1, Point{1, P0 + Vector2{0, 0}, {}, {}}},
        {2, Point{2, P0 + Vector2{0, 100}, {}, {}}}};

    auto clusters = getClusters(points, 10);
    clusters = sortClusters(std::move(clusters));
    EXPECT_EQ(clusters.size(), 2u);
    EXPECT_THAT(clusters[0], testing::UnorderedElementsAre(1));
    EXPECT_THAT(clusters[1], testing::UnorderedElementsAre(2));
}

TEST(points_clustering_tests, test_2_neighboring_points)
{
    Points points{
        {1, Point{1, P0 + Vector2{0, 0}, {2}, {}}},
        {2, Point{2, P0 + Vector2{0, 10}, {1}, {}}}};

    auto clusters = getClusters(points, 10);
    clusters = sortClusters(std::move(clusters));
    EXPECT_EQ(clusters.size(), 1u);
    EXPECT_THAT(clusters[0], testing::UnorderedElementsAre(1, 2));
}

TEST(points_clustering_tests, test_2_clusters)
{
    // points are in 2 groups, but some points are close to points
    // from other groups
    Points points{
        {1, Point{1, P0 + Vector2{0, 5}, {2, 3}, {}}},
        {2, Point{2, P0 + Vector2{0, 10}, {1, 3, 4}, {}}},
        {3, Point{3, P0 + Vector2{0, 15}, {1, 2, 4}, {}}},

        {4, Point{4, P0 + Vector2{0, 25}, {3, 5, 6}, {}}},
        {5, Point{5, P0 + Vector2{0, 30}, {3, 4, 6}, {}}},
        {6, Point{6, P0 + Vector2{0, 35}, {4, 5}, {}}}};

    auto clusters = getClusters(points, 15);
    clusters = sortClusters(std::move(clusters));
    EXPECT_EQ(clusters.size(), 2u);
    EXPECT_THAT(clusters[0], testing::UnorderedElementsAre(1, 2, 3));
    EXPECT_THAT(clusters[1], testing::UnorderedElementsAre(4, 5, 6));
}

TEST(points_clustering_tests, test_3_clusters)
{
    // points are in 3 groups, but some points are close to points
    // from other groups
    Points points{
        {1, Point{1, P0 + Vector2{0, 5}, {2, 3, 7}, {}}},
        {2, Point{2, P0 + Vector2{0, 10}, {1, 3, 4}, {}}},
        {3, Point{3, P0 + Vector2{0, 15}, {1, 2, 4}, {}}},

        {4, Point{4, P0 + Vector2{0, 25}, {3, 5, 6}, {}}},
        {5, Point{5, P0 + Vector2{0, 30}, {3, 4, 6}, {}}},
        {6, Point{6, P0 + Vector2{0, 35}, {4, 5}, {}}},

        {7, Point{7, P0 + Vector2{15, 0}, {1, 8, 9}, {}}},
        {8, Point{8, P0 + Vector2{25, 0}, {7, 9}, {}}},
        {9, Point{9, P0 + Vector2{30, 0}, {7, 8}, {}}}};


    auto clusters = getClusters(points, 15);
    clusters = sortClusters(std::move(clusters));
    EXPECT_EQ(clusters.size(), 3u);
    EXPECT_THAT(clusters[0], testing::UnorderedElementsAre(1, 2, 3));
    EXPECT_THAT(clusters[1], testing::UnorderedElementsAre(4, 5, 6));
    EXPECT_THAT(clusters[2], testing::UnorderedElementsAre(7, 8, 9));
}

TEST(points_clustering_tests, test_prohibited_neighbors)
{
    Points points{
        {1, Point{1, P0 + Vector2{0, 0}, {2}, {3}}},
        {2, Point{2, P0 + Vector2{0, 0}, {1, 2}, {}}},
        {3, Point{3, P0 + Vector2{0, 10}, {2}, {1}}}};

    auto clusters = getClusters(points, 10);
    clusters = sortClusters(std::move(clusters));
    EXPECT_EQ(clusters.size(), 2u);
    if (clusters[0].size() == 1) {
        EXPECT_THAT(clusters[0], testing::UnorderedElementsAre(1));
        EXPECT_THAT(clusters[1], testing::UnorderedElementsAre(2, 3));
    } else {
        EXPECT_THAT(clusters[0], testing::UnorderedElementsAre(1, 2));
        EXPECT_THAT(clusters[1], testing::UnorderedElementsAre(3));
    }
}

} // namespace maps::mrc::sensors_feature_positioner::tests
