#include "utils.h"

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/env.h>
#include <maps/wikimap/mapspro/services/mrc/libs/sensors_feature_positioner/object_positioner/object_single_ray_positioner.h>

namespace maps::mrc::sensors_feature_positioner::tests {

using namespace maps::mrc::pos_improvment;

using maps::geolib3::Point2;
using maps::geolib3::Vector2;
using maps::geolib3::Vector3;

namespace {

std::vector<ObjectPositions> sortByRayId(std::vector<ObjectPositions> objects) {
    for (auto& objectPositions : objects) {
        std::sort(objectPositions.begin(), objectPositions.end(),
                  [](const ObjectPosition& lhs, const ObjectPosition& rhs) {
                      return lhs.ray.rayId < rhs.ray.rayId;
                  });
    }
    std::sort(objects.begin(), objects.end(),
              [](const ObjectPositions& lhs, const ObjectPositions& rhs) {
                  REQUIRE(lhs.size(), "empty ObjectPositions array");
                  REQUIRE(rhs.size(), "empty ObjectPositions array");
                  return lhs[0].ray.rayId < rhs[0].ray.rayId;
              });

    return objects;
}

} // anonymous namespace

// at this point 2 mercator meters = 1 real meter
const Point2 P0 = geolib3::geoPoint2Mercator(Point2{40, 60});

TEST(single_ray_positioner_tests, test_object_from_one_ray)
{
    auto signType = traffic_signs::TrafficSign::ProhibitoryNoParking;

    geolib3::Point2 cameraMercPos = P0;
    geolib3::Point2 cameraOdoMercPos = P0 + Vector2{5, 15};
    pos_improvment::UnitVector3 directionToObject(Vector3(0, 1, 0)); // to North
    double metersToObject = 20;
    RayId rayId = 45;
    db::TId featureId = 37;

    Ray ray = createRay(signType,
                        cameraMercPos,
                        cameraOdoMercPos,
                        directionToObject,
                        metersToObject,
                        rayId,
                        featureId);

    Rays rays{ray};
    auto objects = calculateObjects(rays);

    ASSERT_EQ(objects.size(), 1u);
    ASSERT_EQ(objects[0].size(), 1u);

    EXPECT_TRUE(objects[0][0].ray.rayId == ray.rayId);
    EXPECT_TRUE(objects[0][0].ray.featureId == ray.featureId);
    EXPECT_NEAR(objects[0][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.y(), P0.y() + 40, 0.001); // metersToObject
                                                                  // = 20
    EXPECT_NEAR(objects[0][0].mercatorPos.z(), 0, 0.01);
}

TEST(single_ray_positioner_tests, test_2_separate_objects)
{
    auto signType = traffic_signs::TrafficSign::ProhibitoryNoParking;

    geolib3::Point2 cameraMercPos = P0;
    geolib3::Point2 cameraOdoMercPos = P0 + Vector2{5, 15};
    pos_improvment::UnitVector3 directionToObject(Vector3(0, 1, 0)); // to North
    double metersToObject = 20;
    RayId rayId = 45;
    db::TId featureId = 37;

    Ray ray1 = createRay(signType,
                         cameraMercPos,
                         cameraOdoMercPos,
                         directionToObject,
                         metersToObject,
                         rayId,
                         featureId);
    Ray ray2 = ray1;
    ray2.rayId = 55;
    ray2.featureId = 39;
    ray2.metersToObject = 50;

    Rays rays{ray1, ray2};
    auto objects = calculateObjects(rays);

    objects = sortByRayId(objects);
    ASSERT_EQ(objects.size(), 2u);

    ASSERT_EQ(objects[0].size(), 1u);
    EXPECT_TRUE(objects[0][0].ray.rayId == ray1.rayId);
    EXPECT_TRUE(objects[0][0].ray.featureId == ray1.featureId);

    EXPECT_NEAR(objects[0][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.y(), P0.y() + 40, 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.z(), 0, 0.01);

    ASSERT_EQ(objects[1].size(), 1u);
    EXPECT_TRUE(objects[1][0].ray.rayId == ray2.rayId);
    EXPECT_TRUE(objects[1][0].ray.featureId == ray2.featureId);

    EXPECT_NEAR(objects[1][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.y(), P0.y() + 100, 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.z(), 0, 0.01);
}

TEST(single_ray_positioner_tests, test_2_close_rays_with_different_object_type)
{
    auto signType = traffic_signs::TrafficSign::ProhibitoryNoParking;

    geolib3::Point2 cameraMercPos = P0;
    geolib3::Point2 cameraOdoMercPos = P0 + Vector2{5, 15};
    pos_improvment::UnitVector3 directionToObject(Vector3(0, 1, 0)); // to North
    double metersToObject = 20;
    RayId rayId = 45;
    db::TId featureId = 37;

    Ray ray1 = createRay(signType,
                         cameraMercPos,
                         cameraOdoMercPos,
                         directionToObject,
                         metersToObject,
                         rayId,
                         featureId);

    Ray ray2 = ray1;
    ray2.rayId = 55;
    ray2.featureId = 39;
    ray2.objectTypeId = static_cast<size_t>(traffic_signs::TrafficSign::ProhibitoryNoEntry);

    Rays rays{ray1, ray2};
    auto objects = calculateObjects(rays);

    objects = sortByRayId(objects);
    ASSERT_EQ(objects.size(), 2u);

    ASSERT_EQ(objects[0].size(), 1u);
    EXPECT_TRUE(objects[0][0].ray.rayId == ray1.rayId);
    EXPECT_TRUE(objects[0][0].ray.featureId == ray1.featureId);

    EXPECT_NEAR(objects[0][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.y(), P0.y() + 40, 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.z(), 0, 0.01);

    ASSERT_EQ(objects[1].size(), 1u);
    EXPECT_TRUE(objects[1][0].ray.rayId == ray2.rayId);
    EXPECT_TRUE(objects[1][0].ray.featureId == ray2.featureId);

    EXPECT_NEAR(objects[1][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.y(), P0.y() + 40, 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.z(), 0, 0.01);
}

TEST(single_ray_positioner_tests, test_2_similar_objects_on_one_photo)
{
    auto signType = traffic_signs::TrafficSign::ProhibitoryNoParking;

    geolib3::Point2 cameraMercPos = P0;
    geolib3::Point2 cameraOdoMercPos = P0 + Vector2{5, 15};
    pos_improvment::UnitVector3 directionToObject(Vector3(0, 1, 0)); // to North
    double metersToObject = 20;
    RayId rayId = 45;
    db::TId featureId = 37;

    Ray ray1 = createRay(signType,
                         cameraMercPos,
                         cameraOdoMercPos,
                         directionToObject,
                         metersToObject,
                         rayId,
                         featureId);

    Ray ray2 = ray1;
    ray2.rayId = 55;

    Rays rays{ray1, ray2};
    auto objects = calculateObjects(rays);

    objects = sortByRayId(objects);
    ASSERT_EQ(objects.size(), 2u);

    ASSERT_EQ(objects[0].size(), 1u);
    EXPECT_TRUE(objects[0][0].ray.rayId == ray1.rayId);
    EXPECT_TRUE(objects[0][0].ray.featureId == ray1.featureId);

    EXPECT_NEAR(objects[0][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.y(), P0.y() + 40, 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.z(), 0, 0.01);

    ASSERT_EQ(objects[1].size(), 1u);
    EXPECT_TRUE(objects[1][0].ray.rayId == ray2.rayId);
    EXPECT_TRUE(objects[1][0].ray.featureId == ray2.featureId);

    EXPECT_NEAR(objects[1][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.y(), P0.y() + 40, 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.z(), 0, 0.01);
}


TEST(single_ray_positioner_tests, test_2_rays_of_one_object)
{
    auto signType = traffic_signs::TrafficSign::ProhibitoryNoParking;

    geolib3::Point2 cameraMercPos = P0;
    geolib3::Point2 cameraOdoMercPos = P0 + Vector2{5, 15};
    pos_improvment::UnitVector3 directionToObject(Vector3(0, 1, 0)); // to North
    double metersToObject = 20;
    RayId rayId = 45;
    db::TId featureId = 37;

    Ray ray1 = createRay(signType,
                         cameraMercPos,
                         cameraOdoMercPos,
                         directionToObject,
                         metersToObject,
                         rayId,
                         featureId);

    geolib3::Point2 cameraMercPos2 = P0 + Vector2{10, 0};
    geolib3::Point2 cameraOdoMercPos2 = P0 + Vector2{5, 15};
    pos_improvment::UnitVector3 directionToObject2(Vector3(0, 1, 0)); // to North
    double metersToObject2 = 20;
    RayId rayId2 = 46;
    db::TId featureId2 = 38;

    Ray ray2 = createRay(signType,
                         cameraMercPos2,
                         cameraOdoMercPos2,
                         directionToObject2,
                         metersToObject2,
                         rayId2,
                         featureId2);

    Ray ray3 = ray1;
    ray3.rayId = 47;
    ray3.featureId = 39;
    ray3.metersToObject = 40;

    Rays rays{ray1, ray2, ray3};
    auto objects = calculateObjects(rays);

    objects = sortByRayId(objects);
    ASSERT_EQ(objects.size(), 2u);

    ASSERT_EQ(objects[0].size(), 2u);
    EXPECT_TRUE(objects[0][0].ray.rayId == ray1.rayId);
    EXPECT_TRUE(objects[0][0].ray.featureId == ray1.featureId);

    EXPECT_NEAR(objects[0][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.y(), P0.y() + 40, 0.001);
    EXPECT_NEAR(objects[0][0].mercatorPos.z(), 0, 0.01);

    EXPECT_TRUE(objects[0][1].ray.rayId == ray2.rayId);
    EXPECT_TRUE(objects[0][1].ray.featureId == ray2.featureId);

    EXPECT_NEAR(objects[0][1].mercatorPos.x(), P0.x() + 10, 0.001);
    EXPECT_NEAR(objects[0][1].mercatorPos.y(), P0.y() + 40, 0.001);
    EXPECT_NEAR(objects[0][1].mercatorPos.z(), 0, 0.01);

    ASSERT_EQ(objects[1].size(), 1u);
    EXPECT_TRUE(objects[1][0].ray.rayId == ray3.rayId);
    EXPECT_TRUE(objects[1][0].ray.featureId == ray3.featureId);

    EXPECT_NEAR(objects[1][0].mercatorPos.x(), P0.x(), 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.y(), P0.y() + 80, 0.001);
    EXPECT_NEAR(objects[1][0].mercatorPos.z(), 0, 0.01);
}

} // namespace maps::mrc::sensors_feature_positioner::tests
