#include <yandex/maps/mrc/toloka_client/client.h>
#include "client_impl.h"
#include "magic_strings.h"

#include <maps/libs/json/include/builder.h>

namespace maps {
namespace mrc {
namespace toloka {
namespace io {

MOVABLE_PIMPL_DEFINITIONS(TolokaClient)

const std::string& TolokaClient::schema() const
{
    return impl_->schema_;
}

TolokaClient& TolokaClient::setSchema(const std::string& schema)
{
    impl_->schema_ = schema;
    return *this;
}

cr::milliseconds TolokaClient::timeout() const
{
    return impl_->timeout_;
}

TolokaClient& TolokaClient::setTimeout(cr::milliseconds timeout)
{
    impl_->timeout_ = timeout;
    return *this;
}

size_t TolokaClient::maxRequestAttempts() const
{
    return impl_->maxRequestAttempts_;
}

TolokaClient& TolokaClient::setMaxRequestAttempts(size_t maxRequestAttempts)
{
    impl_->maxRequestAttempts_ = maxRequestAttempts;
    return *this;
}

cr::milliseconds TolokaClient::retryInitialTimeout() const
{
    return impl_->retryInitialTimeout_;
}

TolokaClient& TolokaClient::setRetryInitialTimeout(cr::milliseconds timeout)
{
    impl_->retryInitialTimeout_ = timeout;
    return *this;
}

double TolokaClient::retryTimeoutBackoff() const
{
    return impl_->retryTimeoutBackoff_;
}

TolokaClient& TolokaClient::setRetryTimeoutBackoff(double retryTimeoutBackoff)
{
    REQUIRE(retryTimeoutBackoff >= 0,
            "Negative retry timeout back-off: " << retryTimeoutBackoff);
    impl_->retryTimeoutBackoff_ = retryTimeoutBackoff;
    return *this;
}


TolokaClient::TolokaClient(const std::string& host,
                           const std::string& authHeader)
    : impl_(new Impl(host, authHeader))
{
}


PoolsResponse TolokaClient::getPools(Filter filter) const
{
    return impl_->getPools(std::move(filter));
}

PoolsResponse TolokaClient::getPools(const Filter& filter, size_t limit) const
{
    return impl_->getPools(filter, limit);
}

Pool TolokaClient::getPool(const std::string& poolId) const
{
    return impl_->getPool(poolId);
}

Pool TolokaClient::createPool(const PoolCreationParams& poolCreationParams) const
{
    return impl_->createPool(poolCreationParams);
}

Operation TolokaClient::openPool(const std::string& poolId) const
{
    return impl_->setPoolStatus(poolId, PoolStatus::Open);
}

Operation TolokaClient::closePool(const std::string& poolId) const
{
    return impl_->setPoolStatus(poolId, PoolStatus::Closed);
}

Operation TolokaClient::archivePool(const std::string& poolId) const
{
    return impl_->setPoolStatus(poolId, PoolStatus::Archived);
}


TasksResponse TolokaClient::getTasks(Filter filter) const
{
    return impl_->getTasks(std::move(filter));
}

TasksResponse TolokaClient::getTasks(const Filter& filter, size_t limit) const
{
    return impl_->getTasks(filter, limit);
}

Task TolokaClient::getTask(const std::string& taskId) const
{
    return impl_->getTask(taskId);
}

Task TolokaClient::postTask(const TaskCreationParams& taskCreationParams) const
{
    return impl_->postTask(taskCreationParams, StringMap{});
}

Task TolokaClient::postTask(const TaskCreationParams& taskCreationParams,
                            const StringMap& params) const
{
    return impl_->postTask(taskCreationParams, params);
}

TasksPostResponse TolokaClient::postTasks(
    const std::vector<TaskCreationParams>& taskCreationParamsVec) const
{
    return impl_->postTasks(taskCreationParamsVec, StringMap{});
}

TasksPostResponse TolokaClient::postTasks(
    const std::vector<TaskCreationParams>& taskCreationParamsVec,
    const StringMap& params) const
{
    return impl_->postTasks(taskCreationParamsVec, params);
}

void TolokaClient::setTaskOverlap(const std::string& taskId,
                                  size_t overlap) const
{
    REQUIRE(overlap > 0, "overlap must be positive");
    json::Builder builder;
    builder << [=](json::ObjectBuilder b) { b[FIELD_OVERLAP] = overlap; };
    impl_->patchTask(taskId, builder.str());
}

void TolokaClient::stopAssigningTask(const std::string& taskId) const
{
    impl_->stopAssigningTask(taskId);
}

TaskSuitesResponse TolokaClient::getTaskSuites(Filter filter) const
{
    return impl_->getTaskSuites(std::move(filter));
}

TaskSuitesResponse TolokaClient::getTaskSuites(const Filter& filter,
                                               size_t limit) const
{
    return impl_->getTaskSuites(filter, limit);
}

TaskSuite TolokaClient::getTaskSuite(const std::string& taskSuiteId) const
{
    return impl_->getTaskSuite(taskSuiteId);
}

TaskSuite TolokaClient::postTaskSuite(
    const TaskSuiteCreationParams& taskSuiteCreationParams) const
{
    return impl_->postTaskSuite(taskSuiteCreationParams, StringMap{});
}

TaskSuitesPostResponse TolokaClient::postTaskSuites(
    const std::vector<TaskSuiteCreationParams>& taskSuiteCreationParamsVec) const
{
    return impl_->postTaskSuites(taskSuiteCreationParamsVec, StringMap{});
}

TaskSuite TolokaClient::postTaskSuite(
    const TaskSuiteCreationParams& taskSuiteCreationParams,
    const StringMap& params) const
{
    return impl_->postTaskSuite(taskSuiteCreationParams, params);
}

TaskSuitesPostResponse TolokaClient::postTaskSuites(
    const std::vector<TaskSuiteCreationParams>& taskSuiteCreationParamsVec,
    const StringMap& params) const
{
    return impl_->postTaskSuites(taskSuiteCreationParamsVec, params);
}

void TolokaClient::setTaskSuiteOverlap(const std::string& taskSuiteId,
                                       size_t overlap) const
{
    REQUIRE(overlap > 0, "overlap must be positive");
    json::Builder builder;
    builder << [=](json::ObjectBuilder b) { b[FIELD_OVERLAP] = overlap; };
    impl_->patchTaskSuite(taskSuiteId, builder.str());
}

void TolokaClient::stopAssigningTaskSuite(const std::string& taskSuiteId) const
{
    impl_->stopAssigningTaskSuite(taskSuiteId);
}

AssignmentsResponse TolokaClient::getAssignments(Filter filter) const
{
    return impl_->getAssignments(std::move(filter));
}

AssignmentsResponse TolokaClient::getAssignments(const Filter& filter,
                                                 size_t limit) const
{
    return impl_->getAssignments(filter, limit);
}

Assignment TolokaClient::getAssignment(const std::string& assignmentId) const
{
    return impl_->getAssignment(assignmentId);
}

void TolokaClient::acceptAssignment(const std::string& assignmentId,
                                    const std::string& comment) const
{
    impl_->setAssignmentStatus(assignmentId,
                               AssignmentStatus::Accepted,
                               comment);
}

void TolokaClient::rejectAssignment(
    const std::string& assignmentId,
    const std::string& comment) const
{
    impl_->setAssignmentStatus(assignmentId,
                               AssignmentStatus::Rejected,
                               comment);
}

std::string TolokaClient::getAttachment(const std::string& attachmentId) const
{
    return impl_->getAttachment(attachmentId);
}

Requester TolokaClient::getRequester() const
{
    return impl_->getRequester();
}

} // namespace io
} // namespace toloka
} // namespace mrc
} // namespace maps
