#include <yandex/maps/mrc/toloka_client/filter.h>

#include <yandex/maps/mrc/toloka_client/pool.h>
#include <yandex/maps/mrc/toloka_client/task.h>
#include <yandex/maps/mrc/toloka_client/task_suite.h>

#include <boost/optional.hpp>
#include <boost/optional/optional_io.hpp>

namespace maps {
namespace mrc {
namespace toloka {
namespace io {

class Filter::Impl
{
public:
    Impl() = default;

    std::string projectId_;
    std::string poolId_;
    boost::optional<PoolStatus> poolStatus_;
    std::string taskId_;
    std::string taskSuiteId_;
    boost::optional<AssignmentStatus> assignmentStatus_;
    std::string userId_;
    std::string idGreaterThan_;
    std::string idLessThan_;
    boost::optional<chrono::TimePoint> createdAfter_;
    boost::optional<chrono::TimePoint> createdBefore_;
    boost::optional<chrono::TimePoint> createdNotEarlierThan_;
    boost::optional<chrono::TimePoint> createdNotLaterThan_;
    boost::optional<chrono::TimePoint> submittedAfter_;
    boost::optional<chrono::TimePoint> submittedBefore_;
    boost::optional<chrono::TimePoint> submittedNotEarlierThan_;
    boost::optional<chrono::TimePoint> submittedNotLaterThan_;

    void applyProjectId(http::Request& request)
    {
        if (!projectId_.empty()) {
            request.addParam("project_id", projectId_);
        }
    }

    void applyPoolId(http::Request& request)
    {
        if (!poolId_.empty()) {
            request.addParam("pool_id", poolId_);
        }
    }

    void applyPoolStatus(http::Request& request)
    {
        if (poolStatus_) {
            request.addParam("status",
                boost::lexical_cast<std::string>(*poolStatus_));
        }
    }

    void applyTaskId(http::Request& request)
    {
        if (!taskId_.empty()) {
            request.addParam("task_id", taskId_);
        }
    }

    void applyTaskSuiteId(http::Request& request)
    {
        if (!taskSuiteId_.empty()) {
            request.addParam("task_suite_id", taskSuiteId_);
        }
    }

    void applyUserId(http::Request& request)
    {
        if (!userId_.empty()) {
            request.addParam("user_id", userId_);
        }
    }

    void applyAssignmentStatus(http::Request& request)
    {
        if (assignmentStatus_) {
            request.addParam("status",
                    boost::lexical_cast<std::string>(assignmentStatus_));
        }
    }

    void applyIdGt(http::Request& request)
    {
        if (!idGreaterThan_.empty()) {
            request.addParam("id_gt", idGreaterThan_);
        }
    }

    void applyIdLt(http::Request& request)
    {
        if (!idLessThan_.empty()) {
            request.addParam("id_lt", idLessThan_);
        }
    }

    void applyCreatedGt(http::Request& request)
    {
        if (createdAfter_) {
            request.addParam("created_gt",
                chrono::formatIsoDateTime(*createdAfter_));
        }
    }

    void applyCreatedLt(http::Request& request)
    {
        if (createdBefore_) {
            request.addParam("created_lt",
                chrono::formatIsoDateTime(*createdBefore_));
        }
    }

    void applyCreatedGte(http::Request& request)
    {
        if (createdNotEarlierThan_) {
            request.addParam("created_gte",
                chrono::formatIsoDateTime(*createdNotEarlierThan_));
        }
    }

    void applyCreatedLte(http::Request& request)
    {
        if (createdNotLaterThan_) {
            request.addParam("created_lte",
                chrono::formatIsoDateTime(*createdNotLaterThan_));
        }
    }

    void applySubmittedGt(http::Request& request)
    {
        if (submittedAfter_) {
            request.addParam("submitted_gt",
                chrono::formatIsoDateTime(*submittedAfter_));
        }
    }

    void applySubmittedLt(http::Request& request)
    {
        if (submittedBefore_) {
            request.addParam("submitted_lt",
                chrono::formatIsoDateTime(*submittedBefore_));
        }
    }

    void applySubmittedGte(http::Request& request)
    {
        if (submittedNotEarlierThan_) {
            request.addParam("submitted_gte",
                chrono::formatIsoDateTime(*submittedNotEarlierThan_));
        }
    }

    void applySubmittedLte(http::Request& request)
    {
        if (submittedNotLaterThan_) {
            request.addParam("submitted_lte",
                chrono::formatIsoDateTime(*submittedNotLaterThan_));
        }
    }

};

COPYABLE_PIMPL_DEFINITIONS(Filter)

Filter::Filter() : impl_(new Impl())
{
}

Filter& Filter::byProjectId(const std::string& projectId)
{
    impl_->projectId_ = projectId;
    return *this;
}

Filter& Filter::byPoolId(const std::string& poolId)
{
    impl_->poolId_ = poolId;
    return *this;
}

Filter& Filter::byPoolStatus(PoolStatus poolStatus)
{
    impl_->poolStatus_ = poolStatus;
    return *this;
}

Filter& Filter::byTaskId(const std::string& taskId)
{
    impl_->taskId_ = taskId;
    return *this;
}

Filter& Filter::byTaskSuiteId(const std::string& taskSuiteId)
{
    impl_->taskSuiteId_ = taskSuiteId;
    return *this;
}

Filter& Filter::byAssignmentStatus(AssignmentStatus assignmentStatus)
{
    impl_->assignmentStatus_ = assignmentStatus;
    return *this;
}

Filter& Filter::byUserId(const std::string& userId)
{
    impl_->userId_ = userId;
    return *this;
}

Filter& Filter::idGreaterThan(const std::string& id)
{
    impl_->idGreaterThan_ = id;
    return *this;
}

Filter& Filter::idLessThan(const std::string& id)
{
    impl_->idLessThan_ = id;
    return *this;
}

Filter& Filter::createdAfter(chrono::TimePoint timePoint)
{
    impl_->createdAfter_ = timePoint;
    return *this;
}

Filter& Filter::createdBefore(chrono::TimePoint timePoint)
{
    impl_->createdBefore_ = timePoint;
    return *this;
}

Filter& Filter::createdNotEarlierThan(chrono::TimePoint timePoint)
{
    impl_->createdNotEarlierThan_ = timePoint;
    return *this;
}

Filter& Filter::createdNotLaterThan(chrono::TimePoint timePoint)
{
    impl_->createdNotLaterThan_ = timePoint;
    return *this;
}

Filter& Filter::submittedAfter(chrono::TimePoint timePoint)
{
    impl_->submittedAfter_ = timePoint;
    return *this;
}

Filter& Filter::submittedBefore(chrono::TimePoint timePoint)
{
    impl_->submittedBefore_ = timePoint;
    return *this;
}

Filter& Filter::submittedNotEarlierThan(chrono::TimePoint timePoint)
{
    impl_->submittedNotEarlierThan_ = timePoint;
    return *this;
}

Filter& Filter::submittedNotLaterThan(chrono::TimePoint timePoint)
{
    impl_->submittedNotLaterThan_ = timePoint;
    return *this;
}

template <>
void Filter::apply<Pool>(http::Request& request) const
{
    impl_->applyPoolStatus(request);
    impl_->applyProjectId(request);
    impl_->applyIdGt(request);
    impl_->applyIdLt(request);
    impl_->applyCreatedGt(request);
    impl_->applyCreatedGte(request);
    impl_->applyCreatedLt(request);
    impl_->applyCreatedLte(request);
}

template <>
void Filter::apply<Task>(http::Request& request) const
{
    impl_->applyPoolId(request);
    impl_->applyIdGt(request);
    impl_->applyIdLt(request);
    impl_->applyCreatedGt(request);
    impl_->applyCreatedGte(request);
    impl_->applyCreatedLt(request);
    impl_->applyCreatedLte(request);
}

template <>
void Filter::apply<TaskSuite>(http::Request& request) const
{
    impl_->applyPoolId(request);
    impl_->applyTaskId(request);
    impl_->applyIdGt(request);
    impl_->applyIdLt(request);
    impl_->applyCreatedGt(request);
    impl_->applyCreatedGte(request);
    impl_->applyCreatedLt(request);
    impl_->applyCreatedLte(request);
}

template <>
void Filter::apply<Assignment>(http::Request& request) const
{
    impl_->applyAssignmentStatus(request);
    impl_->applyPoolId(request);
    impl_->applyTaskSuiteId(request);
    impl_->applyUserId(request);
    impl_->applyIdGt(request);
    impl_->applyIdLt(request);
    impl_->applyCreatedGt(request);
    impl_->applyCreatedGte(request);
    impl_->applyCreatedLt(request);
    impl_->applyCreatedLte(request);
    impl_->applySubmittedGt(request);
    impl_->applySubmittedGte(request);
    impl_->applySubmittedLt(request);
    impl_->applySubmittedLte(request);
}

} // namespace io
} // namespace toloka
} // namespace mrc
} // namespace maps
