#pragma once

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/pimpl_utils.h>
#include <maps/libs/json/include/value.h>

#include <boost/optional.hpp>

#include <string>
#include <vector>

namespace maps {
namespace mrc {
namespace toloka {
namespace io {

enum class PoolStatus
{
    Open,
    Closed,
    Archived
};

enum class PoolCloseReason
{
    Manual,
    Expired,
    Completed,
    NotEnoughBalance,
    AssignmentsLimitExceeded,
    Blocked
};

std::ostream& operator<<(std::ostream& os, PoolStatus status);
std::istream& operator>>(std::istream& is, PoolStatus& status);

std::ostream& operator<<(std::ostream& os, PoolCloseReason reason);
std::istream& operator>>(std::istream& is, PoolCloseReason& reason);

// Forward declaration
class Pool;

class PoolCreationParams
{
public:
    // Create pool parameters from an existing pool
    PoolCreationParams(const Pool& pool);

    PoolCreationParams& setPrivateName(std::string name);
    PoolCreationParams& setProjectId(std::string projectId);
    PoolCreationParams& setExpiresAt(chrono::TimePoint expiresAt);
    PoolCreationParams& setRewardPerAssignment(double reward);
    PoolCreationParams& setAssignmentMaxDurationSec(size_t duration);
    PoolCreationParams& setDefaultOverlapTaskSuites(size_t overlap);
    PoolCreationParams& setHasAdultContent(bool hasAdultContent);
    PoolCreationParams& setAutoAcceptSolutions(bool autoAcceptSolutions);

    PoolCreationParams& setFilter(json::Value filter);
    PoolCreationParams& setQualityControl(json::Value qualityControl);
    PoolCreationParams& setMixerConfig(json::Value mixerConfig);

    friend std::ostream& operator <<(std::ostream&, const PoolCreationParams&);

    COPYABLE_PIMPL_DECLARATIONS(PoolCreationParams)
};

/**
 * Toloka pool.
 */
class Pool
{
public:
    const std::string& id() const;
    const std::string& privateName() const;
    PoolStatus status() const;
    const std::string& projectId() const;
    chrono::TimePoint createdAt() const;
    chrono::TimePoint expiresAt() const;
    boost::optional<chrono::TimePoint> lastStopped() const;
    double rewardPerAssignment() const;
    size_t assignmentMaxDurationSec() const;
    size_t defaultOverlapTaskSuites() const;
    bool hasAdultContent() const;
    bool autoAcceptSolutions() const;
    boost::optional<PoolCloseReason> lastCloseReason() const;

    const boost::optional<json::Value>& filter() const;
    const boost::optional<json::Value>& qualityControl() const;
    const boost::optional<json::Value>& mixerConfig() const;

    COPYABLE_PIMPL_DECLARATIONS(Pool)
};

using Pools = std::vector<Pool>;

class PoolsResponse
{
public:
    const Pools& pools() const;

    bool hasMore() const;

    MOVABLE_PIMPL_DECLARATIONS(PoolsResponse)
};


} // namespace io
} // namespace toloka
} // namespace mrc
} // namespace maps

