#include <yandex/maps/mrc/toloka_client/pool.h>
#include "pool_impl.h"

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>

#include <vector>
#include <utility>

namespace maps {
namespace mrc {
namespace toloka {
namespace io {

namespace {

const std::vector<std::pair<PoolStatus, std::string>> POOL_STATUS_NAMES{
    {PoolStatus::Open, "OPEN"},
    {PoolStatus::Closed, "CLOSED"},
    {PoolStatus::Archived, "ARCHIVED"}};

const std::vector<std::pair<PoolCloseReason, std::string>> CLOSE_REASON_NAMES{
    {PoolCloseReason::Manual, "MANUAL"},
    {PoolCloseReason::Expired, "EXPIRED"},
    {PoolCloseReason::Completed, "COMPLETED"},
    {PoolCloseReason::NotEnoughBalance, "NOT_ENOUGH_BALANCE"},
    {PoolCloseReason::AssignmentsLimitExceeded, "ASSIGNMENTS_LIMIT_EXCEEDED"},
    {PoolCloseReason::Blocked, "BLOCKED"},
};

} // anonymous namespace

std::ostream& operator<<(std::ostream& os, PoolStatus status)
{
    auto itr
        = std::find_if(POOL_STATUS_NAMES.begin(), POOL_STATUS_NAMES.end(),
                       [&](const std::pair<PoolStatus, std::string>& pair) {
                           return pair.first == status;
                       });
    REQUIRE(itr != POOL_STATUS_NAMES.end(),
            "Bad pool status value: " << static_cast<int>(status));
    return os << itr->second;
}

std::istream& operator>>(std::istream& is, PoolStatus& status)
{
    std::string str;
    is >> str;
    auto itr
        = std::find_if(POOL_STATUS_NAMES.begin(), POOL_STATUS_NAMES.end(),
                       [&](const std::pair<PoolStatus, std::string>& pair) {
                           return pair.second == str;
                       });
    REQUIRE(itr != POOL_STATUS_NAMES.end(), "Bad pool status: " << str);
    status = itr->first;
    return is;
}

std::ostream& operator<<(std::ostream& os, PoolCloseReason reason)
{
    auto itr
        = std::find_if(CLOSE_REASON_NAMES.begin(), CLOSE_REASON_NAMES.end(),
                       [&](const std::pair<PoolCloseReason, std::string>& pair) {
                           return pair.first == reason;
                       });
    REQUIRE(itr != CLOSE_REASON_NAMES.end(),
            "Bad close reason value: " << static_cast<int>(reason));
    return os << itr->second;
}

std::istream& operator>>(std::istream& is, PoolCloseReason& reason)
{
    std::string str;
    is >> str;
    auto itr
        = std::find_if(CLOSE_REASON_NAMES.begin(), CLOSE_REASON_NAMES.end(),
                       [&](const std::pair<PoolCloseReason, std::string>& pair) {
                           return pair.second == str;
                       });
    REQUIRE(itr != CLOSE_REASON_NAMES.end(), "Bad close reason: " << str);
    reason = itr->first;
    return is;
}


COPYABLE_PIMPL_DEFINITIONS(PoolCreationParams)

PoolCreationParams::PoolCreationParams(const Pool& pool)
    : impl_(new Impl(pool))
{
}

PoolCreationParams& PoolCreationParams::setPrivateName(std::string name)
{
    impl_->privateName_ = std::move(name);
    return *this;
}

PoolCreationParams& PoolCreationParams::setProjectId(std::string projectId)
{
    impl_->projectId_ = std::move(projectId);
    return *this;
}

PoolCreationParams&
PoolCreationParams::setExpiresAt(chrono::TimePoint expiresAt)
{
    impl_->expiresAt_ = expiresAt;
    return *this;
}

PoolCreationParams& PoolCreationParams::setRewardPerAssignment(double reward)
{
    impl_->rewardPerAssignment_ = reward;
    return *this;
}

PoolCreationParams&
PoolCreationParams::setAssignmentMaxDurationSec(size_t duration)
{
    impl_->assignmentMaxDurationSec_ = duration;
    return *this;
}

PoolCreationParams&
PoolCreationParams::setDefaultOverlapTaskSuites(size_t overlap)
{
    impl_->defaultOverlapTaskSuites_ = overlap;
    return *this;
}

PoolCreationParams&
PoolCreationParams::setHasAdultContent(bool hasAdultContent)
{
    impl_->hasAdultContent_ = hasAdultContent;
    return *this;
}

PoolCreationParams&
PoolCreationParams::setAutoAcceptSolutions(bool autoAcceptSolutions)
{
    impl_->autoAcceptSolutions_ = autoAcceptSolutions;
    return *this;
}

PoolCreationParams& PoolCreationParams::setFilter(json::Value filter)
{
    impl_->filter_ = std::move(filter);
    return *this;
}

PoolCreationParams&
PoolCreationParams::setQualityControl(json::Value qualityControl)
{
    impl_->qualityControl_ = std::move(qualityControl);
    return *this;
}

PoolCreationParams& PoolCreationParams::setMixerConfig(json::Value mixerConfig)
{
    impl_->mixerConfig_ = std::move(mixerConfig);
    return *this;
}


std::ostream& operator<<(std::ostream& os, const PoolCreationParams& params)
{
    return os << params.impl_->toJson();
}


COPYABLE_PIMPL_DEFINITIONS(Pool)

const std::string& Pool::id() const { return impl_->id_; }

const std::string& Pool::privateName() const { return impl_->privateName_; }

PoolStatus Pool::status() const { return impl_->status_; }

const std::string& Pool::projectId() const { return impl_->projectId_; }

chrono::TimePoint Pool::createdAt() const { return impl_->createdAt_; }

chrono::TimePoint Pool::expiresAt() const { return impl_->expiresAt_; }

boost::optional<chrono::TimePoint> Pool::lastStopped() const
{
    return impl_->lastStopped_;
}

double Pool::rewardPerAssignment() const
{
    return impl_->rewardPerAssignment_;
}

size_t Pool::assignmentMaxDurationSec() const
{
    return impl_->assignmentMaxDurationSec_;
}

size_t Pool::defaultOverlapTaskSuites() const
{
    return impl_->defaultOverlapTaskSuites_;
}

bool Pool::hasAdultContent() const {
    return impl_->hasAdultContent_;
}

bool Pool::autoAcceptSolutions() const {
    return impl_->autoAcceptSolution_;
}

boost::optional<PoolCloseReason> Pool::lastCloseReason() const {
    return impl_->lastCloseReason_;
}

const OptionalJsonValue& Pool::filter() const
{
    return impl_->filter_;
}

const OptionalJsonValue& Pool::qualityControl() const
{
    return impl_->qualityControl_;
}

const OptionalJsonValue& Pool::mixerConfig() const
{
    return impl_->mixerConfig_;
}

MOVABLE_PIMPL_DEFINITIONS(PoolsResponse)

const Pools& PoolsResponse::pools() const { return impl_->pools_; }

bool PoolsResponse::hasMore() const { return impl_->hasMore_; }

} // namespace io
} // namespace toloka
} // namespace mrc
} // namespace maps
