#include "pool_impl.h"
#include "magic_strings.h"

#include <boost/lexical_cast.hpp>

#include <maps/libs/json/include/builder.h>

namespace maps {
namespace mrc {
namespace toloka {
namespace io {

PoolCreationParams::Impl::Impl(const Pool& pool)
    : privateName_(pool.privateName())
    , projectId_(pool.projectId())
    , expiresAt_(pool.expiresAt())
    , rewardPerAssignment_(pool.rewardPerAssignment())
    , assignmentMaxDurationSec_(pool.assignmentMaxDurationSec())
    , defaultOverlapTaskSuites_(pool.defaultOverlapTaskSuites())
    , hasAdultContent_(pool.hasAdultContent())
    , autoAcceptSolutions_(pool.autoAcceptSolutions())
    , filter_(pool.filter())
    , qualityControl_(pool.qualityControl())
    , mixerConfig_(pool.mixerConfig())
{
}

/**
 * https://tech.yandex.ru/toloka/api/doc/concepts/pool-docpage/
 */
Pool::Impl::Impl(const json::Value& value)
    : id_(value[FIELD_ID].as<std::string>())
    , projectId_(value[FIELD_PROJECT_ID].as<std::string>())
    , privateName_(value[FIELD_PRIVATE_NAME].as<std::string>())
    , status_(boost::lexical_cast<PoolStatus>(value[FIELD_STATUS].as<std::string>()))
    , createdAt_(chrono::parseIsoDateTime(value[FIELD_CREATED].as<std::string>()))
    , expiresAt_(chrono::parseIsoDateTime(value[FIELD_WILL_EXPIRE].as<std::string>()))
    , rewardPerAssignment_(value[FIELD_REWARD_PER_ASSIGNMENT].as<double>())
    , assignmentMaxDurationSec_(value[FIELD_ASSIGNMENT_MAX_DURATION].as<size_t>())
    , defaultOverlapTaskSuites_(value[FIELD_DEFAULTS][FIELD_DEFAULT_OVERLAP_TASK_SUITES].as<size_t>())
    , hasAdultContent_(value[FIELD_MAY_CONTAIN_ADULT_CONTENT].as<bool>())
    , autoAcceptSolution_(value[FIELD_AUTO_ACCEPT_SOLUTIONS].as<bool>())
{
    if (value.hasField(FIELD_LAST_STOPPED)) {
        lastStopped_ = chrono::parseIsoDateTime(value[FIELD_LAST_STOPPED].as<std::string>());
    }

    if (value.hasField(FIELD_LAST_CLOSE_REASON))
        lastCloseReason_ = boost::lexical_cast<PoolCloseReason>(
                value[FIELD_LAST_CLOSE_REASON].as<std::string>());

    if (value.hasField(FIELD_FILTER)) {
        filter_ = value[FIELD_FILTER];
    }

    if (value.hasField(FIELD_QUALITY_CONTROL)) {
        qualityControl_ = value[FIELD_QUALITY_CONTROL];
    }

    if (value.hasField(FIELD_MIXER_CONFIG)) {
        mixerConfig_ = value[FIELD_MIXER_CONFIG];
    }
}

std::string PoolCreationParams::Impl::toJson() const
{
    json::Builder builder;
    builder << [this](json::ObjectBuilder poolBuilder) {
        poolBuilder[FIELD_PRIVATE_NAME] = privateName_;
        poolBuilder[FIELD_PROJECT_ID] = projectId_;
        poolBuilder[FIELD_WILL_EXPIRE] = chrono::formatIsoDateTime(expiresAt_);
        poolBuilder[FIELD_REWARD_PER_ASSIGNMENT] = rewardPerAssignment_;
        poolBuilder[FIELD_ASSIGNMENT_MAX_DURATION] = assignmentMaxDurationSec_;
        poolBuilder[FIELD_DEFAULTS] << [this](json::ObjectBuilder b) {
            b[FIELD_DEFAULT_OVERLAP_TASK_SUITES] = defaultOverlapTaskSuites_;
        };
        poolBuilder[FIELD_MAY_CONTAIN_ADULT_CONTENT] = hasAdultContent_;
        poolBuilder[FIELD_AUTO_ACCEPT_SOLUTIONS] = autoAcceptSolutions_;

        if (filter_) {
            poolBuilder[FIELD_FILTER] = *filter_;
        }

        if (qualityControl_) {
            poolBuilder[FIELD_QUALITY_CONTROL] = *qualityControl_;
        }

        if (mixerConfig_) {
            poolBuilder[FIELD_MIXER_CONFIG] = *mixerConfig_;
        }
    };
    return builder.str();
}

} // namespace io
} // namespace toloka
} // namespace mrc
} // namespace maps
