#include "task_suite_impl.h"
#include "magic_strings.h"

namespace maps {
namespace mrc {
namespace toloka {
namespace io {

/**
 * https://tech.yandex.ru/toloka/api/doc/concepts/task-suite-docpage/
 */
TaskSuite::Impl::Impl(const json::Value& value)
    : id_(value[FIELD_ID].as<std::string>())
    , poolId_(value[FIELD_POOL_ID].as<std::string>())
    , createdAt_(chrono::parseIsoDateTime(value[FIELD_CREATED].as<std::string>()))
{
    if (value.hasField(FIELD_OVERLAP)) {
        overlap_ = value[FIELD_OVERLAP].as<size_t>();
    }

    for (const auto& jsonTask : value[FIELD_TASKS]) {
        tasks_.push_back(TaskSuiteItem::fromJson(jsonTask));
    }
}

namespace {

void buildTaskSuiteItem(json::ArrayBuilder tasksBuilder,
                        const TaskSuiteItem& task)
{
    tasksBuilder << [&](json::ObjectBuilder builder) {
        if (!task.id().empty()) {
            builder[FIELD_ID] = task.id();
        }
        builder[FIELD_INPUT_VALUES] = task.inputValues();
        if (!task.knownSolutions().empty()) {
            builder[FIELD_KNOWN_SOLUTIONS] << [&](json::ArrayBuilder ab) {
                for (const auto& ks : task.knownSolutions()) {
                    ab << [&](json::ObjectBuilder sb) {
                        sb[FIELD_OUTPUT_VALUES] = ks.outputValues();
                        sb[FIELD_CORRECTNESS_WEIGHT] = ks.correctnessWeight();
                    };
                }
            };
        }
        if (!task.messageOnUnknownSolution().empty()) {
            builder[FIELD_MESSAGE_ON_UNKNOWN_SOLUTION]
                    = task.messageOnUnknownSolution();
        }
    };
}

} // anonymous namespace

void TaskSuiteCreationParams::Impl::toJson(json::ObjectBuilder builder) const
{
    builder[FIELD_POOL_ID] = poolId_;
    builder[FIELD_TASKS] = [&](json::ArrayBuilder tasksBuilder) {
        for (const auto& task : tasks_) {
            buildTaskSuiteItem(tasksBuilder, task);
        }
    };
    if (overlap_) {
        builder[FIELD_OVERLAP] = overlap_;
    }
}

std::string TaskSuiteCreationParams::Impl::toJson() const
{
    json::Builder builder;

    builder << [this](json::ObjectBuilder tsBuilder) {
        toJson(tsBuilder);
    };

    return builder.str();
}

} // namespace io
} // namespace toloka
} // namespace mrc
} // namespace maps
