#include <maps/wikimap/mapspro/services/mrc/libs/toloka_common/include/json.h>

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/std.h>
#include <maps/libs/json/include/value.h>

#include <boost/range/adaptor/filtered.hpp>

namespace maps::json {

void json(const maps::mrc::toloka::common::Rect& rect, ArrayBuilder builder)
{
    builder << std::vector<double>({rect.minX(), rect.minY()});
    builder << std::vector<double>({rect.maxX(), rect.maxY()});
}

} // namespace maps::json

namespace maps::mrc::toloka::common {

std::vector<TolokaAnswer> parseTolokaAnswers(std::istream& is)
{
    std::vector<TolokaAnswer> result;
    auto inputJson = json::Value::fromStream(is);
    for (const auto& item : inputJson) {
        TolokaAnswer answer;
        answer.assignment = item["assignmentId"].as<std::string>();
        answer.photo = item["inputValues"]["source"].as<std::string>();
        if (item.hasField("outputValues")) {
            auto outVals = item["outputValues"];
            if (outVals.isObject() && outVals.hasField("polygons")) {
                auto polygons = json::Value::fromString(
                    outVals["polygons"].as<std::string>());
                if (!polygons.isNull()) {
                    for (const auto& polygon : polygons) {
                        const auto& lt = polygon[0];
                        const auto& rb = polygon[1];
                        answer.rects.push_back(
                            {geolib3::Point2{
                                 lt[0].as<double>(), lt[1].as<double>()},
                                geolib3::Point2{
                                    rb[0].as<double>(), rb[1].as<double>()}});
                    }
                }
            }
        }
        result.push_back(std::move(answer));
    }
    return result;
}

void save(const PhotoToRectsMap& aggregatedPhotos, std::ostream& os)
{
    auto notEmpty = [](const auto& aggregatedPhoto) {
        const auto & [ photo, rects ] = aggregatedPhoto;
        return !rects.empty();
    };

    json::Builder builder(os);
    builder << [&](json::ArrayBuilder b) {
        for (const auto& aggregatedPhoto :
            aggregatedPhotos | boost::adaptors::filtered(notEmpty)) {
            b << [&](json::ObjectBuilder b) {
                b["inputValues"] << [&](json::ObjectBuilder b) {
                    // structured binding doesn't work out of lambda
                    const auto & [ photo, rects ] = aggregatedPhoto;
                    b["image"] = photo;
                    b["bboxes"] << rects;
                };
            };
        }
    };
}

void save(const AssignmentToPrecisionAndRecallMap& aggregatedAssignments,
    std::ostream& os)
{
    constexpr float THRESHOLD_RECALL = 0.65;
    constexpr float THRESHOLD_PRECISION = 0.5;
    json::Builder builder(os);
    builder << [&](json::ArrayBuilder b) {
        for (const auto& aggregatedAssignment : aggregatedAssignments) {
            // structured binding doesn't work out of lambda
            const auto& assignment = aggregatedAssignment.first;
            const auto& precisionAndRecall = aggregatedAssignment.second;
            b << [&](json::ObjectBuilder b) {
                b["assignmentId"] << assignment;
                b["status"] << [&](json::ObjectBuilder b) {
                    const auto & [ precision, recall ]
                        = toPair(precisionAndRecall);
                    if (recall < THRESHOLD_RECALL) {
                        b["value"] << "REJECT_SUBMITTED";
                        b["comment"] << "Вы разметили не всё либо нарушили "
                                        "правила разметки, описанные в "
                                        "инструкции";
                    } else if (precision < THRESHOLD_PRECISION) {
                        b["value"] << "REJECT_SUBMITTED";
                        b["comment"] << "Вы ошибочно обвели области, в "
                                        "которых ничего нет либо нарушили "
                                        "правила разметки, описанные в "
                                        "инструкции";
                    } else {
                        b["value"] << "APPROVE_SUBMITTED";
                    }
                };
            };
        }
    };
}

} // namespace maps::mrc::toloka::common
